import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  Users, 
  Plus, 
  Edit, 
  Trash2, 
  Eye, 
  Star, 
  StarOff, 
  Copy, 
  Download,
  Settings,
  Target,
  Trophy,
  Calendar,
  MapPin
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
  };
  display_name?: string;
  avatar?: string;
  football_position?: string;
  jersey_number?: number;
  status: string;
}

interface Formation {
  id: number;
  name: string;
  description?: string;
  player_positions: Record<string, any>;
  formation_data: any;
  is_default: boolean;
  is_active: boolean;
  created_at: string;
  formation: {
    id: number;
    name: string;
    display_name: string;
    description: string;
    category: string;
    country: string;
  };
}

interface Team {
  id: number;
  name: string;
  formations: Formation[];
}

interface FormationsProps {
  team: Team;
  teams: Team[];
  players: Player[];
  session?: any;
}

export default function TeamFormations({ team, teams, players, session }: FormationsProps) {
  const [selectedFormation, setSelectedFormation] = useState<Formation | null>(null);


  const handleDeleteFormation = (formationId: number) => {
    if (confirm('Are you sure you want to delete this formation?')) {
      router.delete(route('team.formations.destroy', formationId), {
        data: { team_id: team.id }
      });
    }
  };

  const handleSetDefault = (formationId: number) => {
    router.post(route('team.formations.set-default', formationId), {
      team_id: team.id
    });
  };

  const getCategoryColor = (category: string) => {
    switch (category?.toLowerCase()) {
      case 'attacking':
        return 'bg-red-100 text-red-800';
      case 'defensive':
        return 'bg-blue-100 text-blue-800';
      case 'standard':
        return 'bg-green-100 text-green-800';
      case 'modern':
        return 'bg-purple-100 text-purple-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getFormationPreview = (formation: Formation) => {
    const positions = formation.formation_data;
    const playerCount = Object.keys(formation.player_positions || {}).length;
    
    return {
      totalPlayers: playerCount,
      hasGoalkeeper: positions.goalkeeper ? 1 : 0,
      defenders: positions.defenders?.length || 0,
      midfielders: positions.midfielders?.length || 0,
      forwards: positions.forwards?.length || 0,
    };
  };

  return (
    <AppLayout>
      <Head title="Team Formations" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/formations" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Formations</h1>
            <p className="text-gray-600">Create and manage your team's tactical formations</p>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => router.visit(route('team.formations.create', { team_id: team.id }) as any)}
            >
              <Plus className="w-4 h-4 mr-2" />
              Create Formation
            </Button>
          </div>
        </div>

        {/* Formation Stats */}
        <div className="grid gap-4 md:grid-cols-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Target className="w-5 h-5 text-blue-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Total Formations</p>
                  <p className="text-2xl font-bold text-gray-900">{team?.formations?.length}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 bg-green-100 rounded-lg">
                  <Star className="w-5 h-5 text-green-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Default Formation</p>
                  <p className="text-lg font-semibold text-gray-900">
                    {team?.formations?.find(f => f.is_default)?.name || 'None'}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Users className="w-5 h-5 text-purple-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Available Players</p>
                  <p className="text-2xl font-bold text-gray-900">{players?.length}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 bg-orange-100 rounded-lg">
                  <Trophy className="w-5 h-5 text-orange-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Active Formations</p>
                  <p className="text-2xl font-bold text-gray-900">
                    {team?.formations?.filter(f => f.is_active).length}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Formations Grid */}
        {team?.formations?.length > 0 ? (
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
            {team?.formations?.map((formation) => {
              const preview = getFormationPreview(formation);
              
              return (
                <Card key={formation.id} className="hover:shadow-lg transition-shadow">
                  <CardHeader className="pb-3">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <div className="p-2 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg">
                          <Target className="w-5 h-5 text-white" />
                        </div>
                        <div>
                          <h3 className="font-semibold text-gray-900">{formation.name}</h3>
                          <p className="text-sm text-gray-500">{formation.formation.display_name}</p>
                        </div>
                      </div>
                      <div className="flex items-center gap-1">
                        {formation.is_default && (
                          <Badge className="bg-yellow-100 text-yellow-800">
                            <Star className="w-3 h-3 mr-1" />
                            Default
                          </Badge>
                        )}
                      </div>
                    </div>
                  </CardHeader>
                  
                  <CardContent className="pt-0">
                    <div className="space-y-4">
                      {/* Formation Description */}
                      {formation.description && (
                        <p className="text-sm text-gray-600">{formation.description}</p>
                      )}

                      {/* Formation Category */}
                      <div className="flex items-center gap-2">
                        <Badge className={getCategoryColor(formation.formation.category)}>
                          {formation.formation.category}
                        </Badge>
                        {formation.formation.country && (
                          <Badge variant="outline" className="text-xs">
                            <MapPin className="w-3 h-3 mr-1" />
                            {formation.formation.country}
                          </Badge>
                        )}
                      </div>

                      {/* Formation Preview */}
                      <div className="bg-gray-50 rounded-lg p-4">
                        <h4 className="text-sm font-medium text-gray-700 mb-3">Formation Layout</h4>
                        <div className="grid grid-cols-2 gap-2 text-xs">
                          <div className="flex justify-between">
                            <span className="text-gray-600">Goalkeeper:</span>
                            <span className="font-medium">{preview.hasGoalkeeper}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Defenders:</span>
                            <span className="font-medium">{preview.defenders}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Midfielders:</span>
                            <span className="font-medium">{preview.midfielders}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Forwards:</span>
                            <span className="font-medium">{preview.forwards}</span>
                          </div>
                        </div>
                        <div className="mt-2 pt-2 border-t">
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-600">Players Assigned:</span>
                            <span className="font-medium">{preview.totalPlayers}/11</span>
                          </div>
                        </div>
                      </div>

                      {/* Formation Actions */}
                      <div className="flex items-center justify-between pt-2">
                        <div className="flex gap-1">
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => router.visit(route('team.formations.show', formation.id) as any)}
                            title="View Formation"
                          >
                            <Eye className="w-3 h-3" />
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => router.visit(route('team.formations.edit', formation.id))}
                            title="Edit Formation"
                          >
                            <Edit className="w-3 h-3" />
                          </Button>
                          {!formation.is_default && (
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleSetDefault(formation.id)}
                              title="Set as Default"
                            >
                              <Star className="w-3 h-3" />
                            </Button>
                          )}
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleDeleteFormation(formation.id)}
                            className="text-red-600 hover:text-red-700"
                            title="Delete Formation"
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                        
                        <div className="text-xs text-gray-500">
                          {new Date(formation.created_at).toLocaleDateString()}
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        ) : (
          <Card>
            <CardContent className="text-center py-12">
              <Target className="w-12 h-12 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">No Formations Yet</h3>
              <p className="text-gray-500 mb-4">Start creating tactical formations for your team</p>
              <Button onClick={() => router.visit(route('team.formations.create', { team_id: team?.id }))}>
                <Plus className="w-4 h-4 mr-2" />
                Create First Formation
              </Button>
            </CardContent>
          </Card>
        )}

        {/* Quick Actions */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Settings className="w-5 h-5" />
              Quick Actions
            </CardTitle>
            <CardDescription>
              Common formation management tasks
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-3">
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.formations.create', { team_id: team?.id }))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Plus className="w-6 h-6" />
                <span>Create New Formation</span>
              </Button>
              
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.players', { team_id: team.id }))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Users className="w-6 h-6" />
                <span>Manage Players</span>
              </Button>
              
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.matches', { team_id: team.id }))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Calendar className="w-6 h-6" />
                <span>View Matches</span>
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
