import React, { useState, useEffect } from 'react';
import { Head, useForm, usePage, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  ArrowLeft, 
  Save, 
  Target, 
  Users, 
  MapPin, 
  Info,
  Shirt,
  User,
  Plus,
  X,
  RotateCcw,
  Eye
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import FormationField from '@/components/TeamManager/FormationField';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
  };
  display_name?: string;
  avatar?: string;
  football_position?: string;
  jersey_number?: number;
  status: string;
}

interface Formation {
  id: number;
  name: string;
  display_name: string;
  description: string;
  positions: any;
  category: string;
  country: string;
}

interface Team {
  id: number;
  name: string;
}

interface CreateFormationProps {
  team: Team;
  teams: Team[];
  availableFormations: Formation[];
  players: Player[];
}

interface PositionData {
  x: number;
  y: number;
  label: string;
  playerId?: number;
  playerName?: string;
  playerJersey?: number;
}

interface CreateFormationProps {
  team: Team;
  teams: Team[];
  availableFormations: Formation[];
  players: Player[];
}

export default function CreateFormation({ team, teams, availableFormations, players }: CreateFormationProps) {
  const [selectedFormation, setSelectedFormation] = useState<Formation | null>(null);
  const [playerPositions, setPlayerPositions] = useState<Record<string, PositionData>>({});
  const [showFormationPreview, setShowFormationPreview] = useState(false);

  // Debug session data
  const { props } = usePage();
  const { data, setData, post, processing, errors } = useForm({
    team_id: team.id,
    formation_id: '',
    name: '',
    description: '',
    player_positions: {} as Record<string, any>,
    is_default: false,
  });

  // Update player positions when formation changes
  useEffect(() => {
    if (selectedFormation) {
      setData('formation_id', selectedFormation.id.toString());
      setData('name', selectedFormation.display_name);
      
      // Initialize player positions with formation structure
      const positions: Record<string, PositionData> = {};
      if (selectedFormation.positions.goalkeeper) {
        positions.goalkeeper = { ...selectedFormation.positions.goalkeeper };
      }
      if (selectedFormation.positions.defenders) {
        selectedFormation.positions.defenders.forEach((def: any, index: number) => {
          positions[`defender_${index}`] = { ...def };
        });
      }
      if (selectedFormation.positions.midfielders) {
        selectedFormation.positions.midfielders.forEach((mid: any, index: number) => {
          positions[`midfielder_${index}`] = { ...mid };
        });
      }
      if (selectedFormation.positions.forwards) {
        selectedFormation.positions.forwards.forEach((fwd: any, index: number) => {
          positions[`forward_${index}`] = { ...fwd };
        });
      }
      
      setPlayerPositions(positions);
      setData('player_positions', positions);
    }
  }, [selectedFormation]);

  const handleFormationSelect = (formationId: string) => {
    const formation = availableFormations.find(f => f.id.toString() === formationId);
    setSelectedFormation(formation || null);
  };

  const handlePlayerAssign = (positionKey: string, playerId: string) => {
    const player = players.find(p => p.id.toString() === playerId);
    if (player) {
      const updatedPositions = {
        ...playerPositions,
        [positionKey]: {
          ...playerPositions[positionKey],
          playerId: player.id,
          playerName: player.display_name || player.user.name,
          playerJersey: player.jersey_number,
        }
      };
      setPlayerPositions(updatedPositions);
      setData('player_positions', updatedPositions);
    }
  };

  const handleRemovePlayer = (positionKey: string) => {
    const updatedPositions = {
      ...playerPositions,
      [positionKey]: {
        ...playerPositions[positionKey],
        playerId: undefined,
        playerName: undefined,
        playerJersey: undefined,
      }
    };
    setPlayerPositions(updatedPositions);
    setData('player_positions', updatedPositions);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    post(route('team.formations.store'));
  };

  const getCategoryColor = (category: string) => {
    switch (category?.toLowerCase()) {
      case 'attacking':
        return 'bg-red-100 text-red-800';
      case 'defensive':
        return 'bg-blue-100 text-blue-800';
      case 'standard':
        return 'bg-green-100 text-green-800';
      case 'modern':
        return 'bg-purple-100 text-purple-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getPositionType = (key: string) => {
    if (key === 'goalkeeper') return 'Goalkeeper';
    if (key.startsWith('defender')) return 'Defender';
    if (key.startsWith('midfielder')) return 'Midfielder';
    if (key.startsWith('forward')) return 'Forward';
    return 'Unknown';
  };

  const getPositionColor = (key: string) => {
    if (key === 'goalkeeper') return 'bg-blue-100 text-blue-800';
    if (key.startsWith('defender')) return 'bg-green-100 text-green-800';
    if (key.startsWith('midfielder')) return 'bg-yellow-100 text-yellow-800';
    if (key.startsWith('forward')) return 'bg-red-100 text-red-800';
    return 'bg-gray-100 text-gray-800';
  };

    return (
      <AppLayout>
        <Head title="Create Formation" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/formations/create" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.visit(route('team.formations', { team_id: team.id }))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Create Formation</h1>
              <p className="text-gray-600">Set up a new tactical formation for {team.name}</p>
            </div>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => setShowFormationPreview(!showFormationPreview)}
            >
              <Eye className="w-4 h-4 mr-2" />
              {showFormationPreview ? 'Hide' : 'Show'} Preview
            </Button>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="grid gap-6 lg:grid-cols-2">
            {/* Formation Selection */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Target className="w-5 h-5" />
                  Select Formation
                </CardTitle>
                <CardDescription>
                  Choose from popular Kenyan football formations
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="formation_id">Base Formation</Label>
                  <Select onValueChange={handleFormationSelect}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select a formation" />
                    </SelectTrigger>
                    <SelectContent>
                      {availableFormations.map((formation) => (
                        <SelectItem key={formation.id} value={formation.id.toString()}>
                          <div className="flex items-center gap-2">
                            <span>{formation.display_name}</span>
                            <Badge className={getCategoryColor(formation.category)}>
                              {formation.category}
                            </Badge>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.formation_id && (
                    <p className="text-sm text-red-600 mt-1">{errors.formation_id}</p>
                  )}
                </div>

                {selectedFormation && (
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="font-medium text-gray-900 mb-2">{selectedFormation.display_name}</h4>
                    <p className="text-sm text-gray-600 mb-3">{selectedFormation.description}</p>
                    <div className="flex items-center gap-2">
                      <Badge className={getCategoryColor(selectedFormation.category)}>
                        {selectedFormation.category}
                      </Badge>
                      {selectedFormation.country && (
                        <Badge variant="outline" className="text-xs">
                          <MapPin className="w-3 h-3 mr-1" />
                          {selectedFormation.country}
                        </Badge>
                      )}
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Formation Details */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Info className="w-5 h-5" />
                  Formation Details
                </CardTitle>
                <CardDescription>
                  Customize your formation settings
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="name">Formation Name</Label>
                  <Input
                    id="name"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    placeholder="Enter formation name"
                  />
                  {errors.name && (
                    <p className="text-sm text-red-600 mt-1">{errors.name}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="description">Description (Optional)</Label>
                  <Textarea
                    id="description"
                    value={data.description}
                    onChange={(e) => setData('description', e.target.value)}
                    placeholder="Describe this formation's strategy..."
                    rows={3}
                  />
                  {errors.description && (
                    <p className="text-sm text-red-600 mt-1">{errors.description}</p>
                  )}
                </div>

                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="is_default"
                    checked={data.is_default}
                    onCheckedChange={(checked) => setData('is_default', checked as false)}
                  />
                  <Label htmlFor="is_default" className="text-sm">
                    Set as default formation
                  </Label>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Player Assignment */}
          {selectedFormation && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Users className="w-5 h-5" />
                  Assign Players
                </CardTitle>
                <CardDescription>
                  Assign your players to specific positions in the formation
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                  {Object.entries(playerPositions).map(([key, position]) => (
                    <div key={key} className="border rounded-lg p-4">
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center gap-2">
                          <Badge className={getPositionColor(key)}>
                            {position.label}
                          </Badge>
                          <span className="text-sm text-gray-600">
                            {getPositionType(key)}
                          </span>
                        </div>
                        {position.playerId && (
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleRemovePlayer(key)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <X className="w-3 h-3" />
                          </Button>
                        )}
                      </div>

                      {position.playerId ? (
                        <div className="bg-green-50 rounded-lg p-3">
                          <div className="flex items-center gap-2">
                            <div className="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                              <Shirt className="w-4 h-4 text-green-600" />
                            </div>
                            <div>
                              <p className="font-medium text-sm text-gray-900">
                                {position.playerName || ''}
                              </p>
                              <p className="text-xs text-gray-600">
                                #{position.playerJersey || ''}
                              </p>
                            </div>
                          </div>
                        </div>
                      ) : (
                        <div>
                          <Select onValueChange={(value) => handlePlayerAssign(key, value)}>
                            <SelectTrigger>
                              <SelectValue placeholder="Select player" />
                            </SelectTrigger>
                            <SelectContent>
                              {players.map((player) => (
                                <SelectItem key={player.id} value={player.id.toString()}>
                                  <div className="flex items-center gap-2">
                                    <span>{player.display_name || player.user.name}</span>
                                    {player.jersey_number && (
                                      <span className="text-xs text-gray-500">
                                        #{player.jersey_number}
                                      </span>
                                    )}
                                  </div>
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </div>
                      )}
                    </div>
                  ))}
                </div>

                {players.length === 0 && (
                  <div className="text-center py-8">
                    <Users className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">No Players Available</h3>
                    <p className="text-gray-500 mb-4">You need to add players to your team first</p>
                    <Button
                      type="button"
                      onClick={() => router.visit(route('team.players.create', { team_id: team.id }) as any)}
                    >
                      <Plus className="w-4 h-4 mr-2" />
                      Add Players
                    </Button>
                  </div>
                )}
              </CardContent>
            </Card>
          )}

          {/* Formation Preview */}
          {showFormationPreview && selectedFormation && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Eye className="w-5 h-5" />
                  Formation Preview
                </CardTitle>
                <CardDescription>
                  Visual representation of your formation
                </CardDescription>
              </CardHeader>
              <CardContent>
                <FormationField
                  formation={{
                    id: 0,
                    name: selectedFormation.display_name,
                    player_positions: playerPositions,
                    formation_data: selectedFormation.positions
                  }}
                  players={players}
                  onPlayerClick={(player, position) => {
                    console.log('Player clicked:', player, 'Position:', position);
                  }}
                  showPlayerNames={true}
                  className="w-full"
                />
              </CardContent>
            </Card>
          )}

          {/* Submit Button */}
          <div className="flex justify-end gap-4">
            <Button
              type="button"
              variant="outline"
              onClick={() => router.visit(route('team.formations', { team_id: team.id }) as any)}
            >
              Cancel
            </Button>
            <Button type="submit" disabled={processing}>
              {processing ? (
                <>
                  <RotateCcw className="w-4 h-4 mr-2 animate-spin" />
                  Creating...
                </>
              ) : (
                <>
                  <Save className="w-4 h-4 mr-2" />
                  Create Formation
                </>
              )}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
