import React, { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { 
  User, 
  Mail, 
  Phone, 
  Calendar, 
  MapPin, 
  Shirt, 
  Gamepad2, 
  Globe, 
  Save,
  ArrowLeft,
  Plus,
  FileText,
  Camera,
  IdCard
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import { toast } from 'sonner';

interface Team {
  id: number;
  name: string;
}

interface CreatePlayerProps {
  team: Team;
  teams: Team[];
}

const POSITIONS = [
  'Goalkeeper',
  'Defender',
  'Midfielder',
  'Forward',
  'Striker',
  'Winger',
  'Captain',
  'Vice-Captain'
];

const PLAYER_TYPES = [
  'Amateur',
  'Semi-Professional',
  'Professional',
  'Legend'
];

const EXPERIENCE_LEVELS = [
  'Beginner',
  'Intermediate',
  'Advanced',
  'Expert',
  'Master'
];

const GENDERS = [
  { value: 'male', label: 'Male' },
  { value: 'female', label: 'Female' },
  { value: 'other', label: 'Other' },
  { value: 'prefer_not_to_say', label: 'Prefer not to say' }
];

const GAMES = [
  'FIFA',
  'PES',
  'Football Manager',
  'Other'
];

export default function CreatePlayer({ team, teams }: CreatePlayerProps) {

  const { data, setData, post, processing, errors, reset } = useForm({
    // User creation fields
    name: '',
    email: '',
    password: '',
    password_confirmation: '',
    phone: '',
    
    // Player profile fields
    display_name: '',
    bio: '',
    country: '',
    city: '',
    date_of_birth: '',
    gender: 'male',
    discord_username: '',
    steam_id: '',
    twitch_username: '',
    youtube_channel: '',
    twitter_handle: '',
    instagram_handle: '',
    website: '',
    player_type: '',
    experience_level: '',
    primary_game: '',
    game_preferences: [],
    football_position: '',
    jersey_number: '',
    is_team_captain: false,
    is_public_profile: false,
    allow_messages: false,
    show_contact_info: false,
    
    // Avatar field
    avatar: null as File | null,
    
    // Identification fields
    id_number: '',
    birth_certificate_number: '',
    passport_photo: null as File | null,
    birth_cert_photo: null as File | null,
    approval_status: 'pending',
    
    // Team assignment
    team_id: team.id,
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Create FormData for file uploads
    const formData = new FormData();
    
    // Always create new user
    formData.append('create_new_user', 'true');
    
    // Add all form data
    Object.keys(data).forEach(key => {
      if (data[key as keyof typeof data] !== null && data[key as keyof typeof data] !== undefined) {
        if (key === 'avatar' || key === 'passport_photo' || key === 'birth_cert_photo') {
          // Handle file uploads
          if (data[key as keyof typeof data] instanceof File) {
            formData.append(key, data[key as keyof typeof data] as File);
          }
        } else {
          formData.append(key, String(data[key as keyof typeof data]));
        }
      }
    });

    try {
      const response = await fetch(route('team.players.store'), {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
          'X-Requested-With': 'XMLHttpRequest',
        },
        body: formData,
      });

      const result = await response.json();

      if (response.ok) {
        // Show success message with toast
        console.log('Success response:', result);
        toast.success(result.message || 'Player created successfully!');
        // Redirect after 5 seconds
        setTimeout(() => {
          router.visit(route('team.players'));
        }, 5000);
      } else {
        // Show error messages with toast
        console.log('Error response:', result);
        if (result.error) {
          toast.error(result.error);
        } else if (result.errors) {
          // Handle validation errors
          const errorMessages = Object.values(result.errors).flat();
          toast.error(errorMessages.join(', '));
        } else {
          toast.error('Failed to create player. Please check the form for errors.');
        }
      }
    } catch (error) {
      console.error('Error creating player:', error);
      toast.error('An unexpected error occurred. Please try again.');
    }
  };

  const handleCancel = () => {
    router.visit(route('team.players'));
  };

  return (
    <AppLayout>
      <Head title="Create Player" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/players/create" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Create New Player</h1>
            <p className="text-gray-600">Add a new player to {team.name} with complete profile</p>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={handleCancel}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Players
            </Button>
          </div>
        </div>

        {/* Player Creation Form */}
        <form onSubmit={handleSubmit}>
          <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <User className="w-5 h-5" />
                    Personal Information
                  </CardTitle>
                  <CardDescription>
                    Basic personal details and account information
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">

                  {/* Avatar Upload Section */}
                  <div className="flex items-center gap-6 mb-6">
                    <div className="flex-shrink-0">
                      <div className="w-20 h-20 rounded-full overflow-hidden ring-2 ring-gray-200">
                        {data.avatar ? (
                          <img
                            src={URL.createObjectURL(data.avatar)}
                            alt="Preview"
                            className="w-full h-full object-cover"
                          />
                        ) : (
                          <div className="w-full h-full bg-gray-100 flex items-center justify-center">
                            <User className="w-8 h-8 text-gray-600" />
                          </div>
                        )}
                      </div>
                    </div>
                    <div className="flex-1">
                      <Label htmlFor="avatar" className="text-sm font-medium text-gray-700">
                        Profile Photo
                      </Label>
                      <div className="mt-2">
                        <input
                          id="avatar"
                          name="avatar"
                          type="file"
                          accept="image/*"
                          className="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                          onChange={(e) => {
                            const file = e.target.files?.[0];
                            if (file) {
                              setData('avatar', file);
                            }
                          }}
                        />
                        <p className="text-xs text-gray-500 mt-1">PNG, JPG, GIF up to 10MB</p>
                      </div>
                      {errors.avatar && (
                        <p className="text-red-500 text-sm mt-1">{errors.avatar}</p>
                      )}
                    </div>
                  </div>

                    <div className="space-y-4">
                      <div className="grid gap-4 md:grid-cols-2">
                        <div>
                        <Label htmlFor="name">Full Name <span className="text-red-500">*</span></Label>
                          <Input
                            id="name"
                            value={data.name}
                            onChange={(e) => setData('name', e.target.value)}
                            placeholder="Enter full name"
                            required
                          />
                          {errors.name && (
                            <p className="text-red-500 text-sm mt-1">{errors.name}</p>
                          )}
                        </div>

                        <div>
                        <Label htmlFor="email">Email Address <span className="text-red-500">*</span></Label>
                          <Input
                            id="email"
                            type="email"
                            value={data.email}
                            onChange={(e) => setData('email', e.target.value)}
                            placeholder="Enter email address"
                            required
                          />
                          {errors.email && (
                            <p className="text-red-500 text-sm mt-1">{errors.email}</p>
                          )}
                        </div>

                        <div>
                        <Label htmlFor="password">Password <span className="text-red-500">*</span></Label>
                          <Input
                            id="password"
                            type="password"
                            value={data.password}
                            onChange={(e) => setData('password', e.target.value)}
                            placeholder="Enter password"
                            required
                          />
                          {errors.password && (
                            <p className="text-red-500 text-sm mt-1">{errors.password}</p>
                          )}
                        </div>

                        <div>
                        <Label htmlFor="password_confirmation">Confirm Password <span className="text-red-500">*</span></Label>
                          <Input
                            id="password_confirmation"
                            type="password"
                            value={data.password_confirmation}
                            onChange={(e) => setData('password_confirmation', e.target.value)}
                            placeholder="Confirm password"
                            required
                          />
                          {errors.password_confirmation && (
                            <p className="text-red-500 text-sm mt-1">{errors.password_confirmation}</p>
                          )}
                        </div>
                      </div>
                    </div>

                  {/* Team Selection */}
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <Label htmlFor="team_id">Team <span className="text-red-500">*</span></Label>
                      <Select
                        value={data.team_id.toString()}
                        onValueChange={(value) => setData('team_id', parseInt(value))}
                        required
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select team" />
                        </SelectTrigger>
                        <SelectContent>
                          {teams.map((team) => (
                            <SelectItem key={team.id} value={team.id.toString()}>
                              {team.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.team_id && (
                        <p className="text-red-500 text-sm mt-1">{errors.team_id}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="display_name">Display Name</Label>
                      <Input
                        id="display_name"
                        value={data.display_name}
                        onChange={(e) => setData('display_name', e.target.value)}
                        placeholder="Enter display name"
                        required
                      />
                      {errors.display_name && (
                        <p className="text-red-500 text-sm mt-1">{errors.display_name}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="phone">Phone Number</Label>
                      <Input
                        id="phone"
                        value={data.phone}
                        onChange={(e) => setData('phone', e.target.value)}
                        placeholder="Enter phone number"
                      />
                      {errors.phone && (
                        <p className="text-red-500 text-sm mt-1">{errors.phone}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="date_of_birth">Date of Birth</Label>
                      <Input
                        id="date_of_birth"
                        type="date"
                        value={data.date_of_birth}
                        onChange={(e) => setData('date_of_birth', e.target.value)}
                      />
                      {errors.date_of_birth && (
                        <p className="text-red-500 text-sm mt-1">{errors.date_of_birth}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="gender">Gender <span className="text-red-500">*</span></Label>
                      <Select
                        value={data.gender}
                        onValueChange={(value) => setData('gender', value)}
                        required
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select gender" />
                        </SelectTrigger>
                        <SelectContent>
                          {GENDERS.map((gender) => (
                            <SelectItem key={gender.value} value={gender.value}>
                              {gender.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.gender && (
                        <p className="text-red-500 text-sm mt-1">{errors.gender}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="country">Country</Label>
                      <Input
                        id="country"
                        value={data.country}
                        onChange={(e) => setData('country', e.target.value)}
                        placeholder="Enter country"
                      />
                      {errors.country && (
                        <p className="text-red-500 text-sm mt-1">{errors.country}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="city">City</Label>
                      <Input
                        id="city"
                        value={data.city}
                        onChange={(e) => setData('city', e.target.value)}
                        placeholder="Enter city"
                      />
                      {errors.city && (
                        <p className="text-red-500 text-sm mt-1">{errors.city}</p>
                      )}
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="bio">Bio</Label>
                    <textarea
                      id="bio"
                      className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      rows={4}
                      value={data.bio}
                      onChange={(e) => setData('bio', e.target.value)}
                      placeholder="Tell us about yourself..."
                    />
                    {errors.bio && (
                      <p className="text-red-500 text-sm mt-1">{errors.bio}</p>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* Identification Information Card */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <IdCard className="w-5 h-5" />
                    Identification & Documents
                  </CardTitle>
                  <CardDescription>
                    Required identification information and document uploads
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <Label htmlFor="id_number">ID Number / National ID</Label>
                      <Input
                        id="id_number"
                        value={data.id_number}
                        onChange={(e) => setData('id_number', e.target.value)}
                        placeholder="Enter ID number"
                      />
                      {errors.id_number && (
                        <p className="text-red-500 text-sm mt-1">{errors.id_number}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="birth_certificate_number">Birth Certificate Number</Label>
                      <Input
                        id="birth_certificate_number"
                        value={data.birth_certificate_number}
                        onChange={(e) => setData('birth_certificate_number', e.target.value)}
                        placeholder="Enter birth certificate number"
                      />
                      {errors.birth_certificate_number && (
                        <p className="text-red-500 text-sm mt-1">{errors.birth_certificate_number}</p>
                      )}
                    </div>
                  </div>

                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <Label htmlFor="passport_photo">Passport Photo <span className="text-red-500">*</span></Label>
                      <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md hover:border-gray-400 transition-colors">
                        <div className="space-y-1 text-center">
                          <Camera className="mx-auto h-12 w-12 text-gray-400" />
                          <div className="flex text-sm text-gray-600">
                            <label
                              htmlFor="passport_photo"
                              className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500"
                            >
                              <span>Upload passport photo <span className="text-red-500">*</span></span>
                              <input
                                id="passport_photo"
                                name="passport_photo"
                                type="file"
                                accept="image/*"
                                className="sr-only"
                                onChange={(e) => {
                                  const file = e.target.files?.[0];
                                  if (file) {
                                    setData('passport_photo', file);
                                  }
                                }}
                                required
                              />
                            </label>
                          </div>
                          <p className="text-xs text-gray-500">PNG, JPG, GIF up to 10MB</p>
                        </div>
                      </div>
                      {errors.passport_photo && (
                        <p className="text-red-500 text-sm mt-1">{errors.passport_photo}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="birth_cert_photo">Birth Certificate / ID Photo</Label>
                      <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md hover:border-gray-400 transition-colors">
                        <div className="space-y-1 text-center">
                          <FileText className="mx-auto h-12 w-12 text-gray-400" />
                          <div className="flex text-sm text-gray-600">
                            <label
                              htmlFor="birth_cert_photo"
                              className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500"
                            >
                              <span>Upload birth cert/ID photo</span>
                              <input
                                id="birth_cert_photo"
                                name="birth_cert_photo"
                                type="file"
                                accept="image/*,.pdf"
                                className="sr-only"
                                onChange={(e) => {
                                  const file = e.target.files?.[0];
                                  if (file) {
                                    setData('birth_cert_photo', file);
                                  }
                                }}
                              />
                            </label>
                          </div>
                          <p className="text-xs text-gray-500">PNG, JPG, GIF, PDF up to 10MB</p>
                        </div>
                      </div>
                      {errors.birth_cert_photo && (
                        <p className="text-red-500 text-sm mt-1">{errors.birth_cert_photo}</p>
                      )}
                    </div>
                  </div>

                  <div className="bg-yellow-50 border border-yellow-200 rounded-md p-4">
                    <div className="flex">
                      <div className="flex-shrink-0">
                        <FileText className="h-5 w-5 text-yellow-400" />
                      </div>
                      <div className="ml-3">
                        <h3 className="text-sm font-medium text-yellow-800">
                          Document Requirements
                        </h3>
                        <div className="mt-2 text-sm text-yellow-700">
                          <ul className="list-disc pl-5 space-y-1">
                            <li>Passport photo must be clear and recent</li>
                            <li>Birth certificate/ID photo must be clearly readable</li>
                            <li>All documents will be reviewed for approval</li>
                            <li>Player status will be updated after document verification</li>
                          </ul>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

            {/* Gaming Profile */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Gamepad2 className="w-5 h-5" />
                    Gaming Profile
                  </CardTitle>
                  <CardDescription>
                    Gaming preferences and experience level
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <Label htmlFor="player_type">Player Type</Label>
                      <Select
                        value={data.player_type}
                        onValueChange={(value) => setData('player_type', value)}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select player type" />
                        </SelectTrigger>
                        <SelectContent>
                          {PLAYER_TYPES.map((type) => (
                            <SelectItem key={type} value={type}>
                              {type}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.player_type && (
                        <p className="text-red-500 text-sm mt-1">{errors.player_type}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="experience_level">Experience Level</Label>
                      <Select
                        value={data.experience_level}
                        onValueChange={(value) => setData('experience_level', value)}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select experience level" />
                        </SelectTrigger>
                        <SelectContent>
                          {EXPERIENCE_LEVELS.map((level) => (
                            <SelectItem key={level} value={level}>
                              {level}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.experience_level && (
                        <p className="text-red-500 text-sm mt-1">{errors.experience_level}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="primary_game">Primary Game</Label>
                      <Select
                        value={data.primary_game}
                        onValueChange={(value) => setData('primary_game', value)}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select primary game" />
                        </SelectTrigger>
                        <SelectContent>
                          {GAMES.map((game) => (
                            <SelectItem key={game} value={game}>
                              {game}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.primary_game && (
                        <p className="text-red-500 text-sm mt-1">{errors.primary_game}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="football_position">Football Position *</Label>
                      <Select
                        value={data.football_position}
                        onValueChange={(value) => setData('football_position', value)}
                        required
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select position" />
                        </SelectTrigger>
                        <SelectContent>
                          {POSITIONS.map((position) => (
                            <SelectItem key={position} value={position}>
                              {position}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {errors.football_position && (
                        <p className="text-red-500 text-sm mt-1">{errors.football_position}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="jersey_number">Jersey Number</Label>
                      <Input
                        id="jersey_number"
                        type="number"
                        min="1"
                        max="99"
                        value={data.jersey_number}
                        onChange={(e) => setData('jersey_number', e.target.value)}
                        placeholder="e.g., 10"
                      />
                      {errors.jersey_number && (
                        <p className="text-red-500 text-sm mt-1">{errors.jersey_number}</p>
                      )}
                    </div>
                  </div>

                  {/* Social Media & Gaming Platforms */}
                  <div className="space-y-4">
                    <h4 className="font-medium text-gray-900">Social Media & Gaming Platforms</h4>
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <Label htmlFor="discord_username">Discord Username</Label>
                      <Input
                        id="discord_username"
                        value={data.discord_username}
                        onChange={(e) => setData('discord_username', e.target.value)}
                        placeholder="username#1234"
                      />
                      {errors.discord_username && (
                        <p className="text-red-500 text-sm mt-1">{errors.discord_username}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="steam_id">Steam ID</Label>
                      <Input
                        id="steam_id"
                        value={data.steam_id}
                        onChange={(e) => setData('steam_id', e.target.value)}
                        placeholder="Steam ID"
                      />
                      {errors.steam_id && (
                        <p className="text-red-500 text-sm mt-1">{errors.steam_id}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="twitch_username">Twitch Username</Label>
                      <Input
                        id="twitch_username"
                        value={data.twitch_username}
                        onChange={(e) => setData('twitch_username', e.target.value)}
                        placeholder="@username"
                      />
                      {errors.twitch_username && (
                        <p className="text-red-500 text-sm mt-1">{errors.twitch_username}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="youtube_channel">YouTube Channel</Label>
                      <Input
                        id="youtube_channel"
                        value={data.youtube_channel}
                        onChange={(e) => setData('youtube_channel', e.target.value)}
                        placeholder="Channel name or URL"
                      />
                      {errors.youtube_channel && (
                        <p className="text-red-500 text-sm mt-1">{errors.youtube_channel}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="twitter_handle">Twitter Handle</Label>
                      <Input
                        id="twitter_handle"
                        value={data.twitter_handle}
                        onChange={(e) => setData('twitter_handle', e.target.value)}
                        placeholder="@username"
                      />
                      {errors.twitter_handle && (
                        <p className="text-red-500 text-sm mt-1">{errors.twitter_handle}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="instagram_handle">Instagram Handle</Label>
                      <Input
                        id="instagram_handle"
                        value={data.instagram_handle}
                        onChange={(e) => setData('instagram_handle', e.target.value)}
                        placeholder="@username"
                      />
                      {errors.instagram_handle && (
                        <p className="text-red-500 text-sm mt-1">{errors.instagram_handle}</p>
                      )}
                    </div>

                    <div className="md:col-span-2">
                      <Label htmlFor="website">Website</Label>
                      <Input
                        id="website"
                        value={data.website}
                        onChange={(e) => setData('website', e.target.value)}
                        placeholder="https://yourwebsite.com"
                      />
                      {errors.website && (
                        <p className="text-red-500 text-sm mt-1">{errors.website}</p>
                      )}
                      </div>
                    </div>
                  </div>

                  {/* Privacy Settings */}
                  <div className="space-y-4">
                    <h4 className="font-medium text-gray-900">Privacy Settings</h4>
                    <div className="space-y-2">
                      <div className="flex items-center space-x-2">
                        <input
                          type="checkbox"
                          id="is_public_profile"
                          checked={data.is_public_profile}
                          onChange={(e) => setData('is_public_profile', e.target.checked as any)}
                          className="rounded border-gray-300"
                        />
                        <Label htmlFor="is_public_profile">Public Profile</Label>
                      </div>
                      <div className="flex items-center space-x-2">
                        <input
                          type="checkbox"
                          id="allow_messages"
                          checked={data.allow_messages}
                          onChange={(e) => setData('allow_messages', e.target.checked as any)}
                          className="rounded border-gray-300"
                        />
                        <Label htmlFor="allow_messages">Allow Messages</Label>
                      </div>
                      <div className="flex items-center space-x-2">
                        <input
                          type="checkbox"
                          id="show_contact_info"
                          checked={data.show_contact_info}
                          onChange={(e) => setData('show_contact_info', e.target.checked as any)}
                          className="rounded border-gray-300"
                        />
                        <Label htmlFor="show_contact_info">Show Contact Information</Label>
                      </div>
                      <div className="flex items-center space-x-2">
                        <input
                          type="checkbox"
                          id="is_team_captain"
                          checked={data.is_team_captain}
                          onChange={(e) => setData('is_team_captain', e.target.checked as any)}
                          className="rounded border-gray-300"
                        />
                        <Label htmlFor="is_team_captain">Team Captain</Label>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
                  </div>

          {/* Form Actions */}
          <div className="flex justify-end gap-4 pt-6 border-t">
            <Button type="button" variant="outline" onClick={handleCancel}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing}>
              {processing ? 'Creating Player...' : 'Create Player'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
