import React from 'react';
import { Head } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Trophy, TrendingUp, Users, Gamepad2, Target, Award, Calendar, BarChart3 } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Match {
  id: number;
  name: string;
  tournament: {
    id: number;
    name: string;
  };
  team1: {
    id: number;
    name: string;
  };
  team2: {
    id: number;
    name: string;
  };
  scheduled_time: string;
  status: string;
  score_team1?: number;
  score_team2?: number;
  match_type: string;
  round?: string;
}

interface TournamentStats {
  id: number;
  name: string;
  status: string;
  start_date: string;
  end_date: string;
  matches_count: number;
}

interface Team {
  id: number;
  name: string;
}

interface AnalyticsProps {
  team: Team;
  teams: Team[];
  stats: {
    total_matches: number;
    wins: number;
    losses: number;
    win_rate: number;
  };
  recentMatches: Match[];
  tournamentStats: TournamentStats[];
}

export default function TeamAnalytics({ team, teams, stats, recentMatches, tournamentStats }: AnalyticsProps) {
  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  const formatDateTime = (date: string) => {
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'scheduled':
        return 'bg-blue-100 text-blue-800';
      case 'in_progress':
        return 'bg-yellow-100 text-yellow-800';
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getMatchResult = (match: Match) => {
    if (match.status !== 'completed' || !match.score_team1 || !match.score_team2) {
      return null;
    }

    const isTeam1 = match.team1.id === team.id;
    const teamScore = isTeam1 ? match.score_team1 : match.score_team2;
    const opponentScore = isTeam1 ? match.score_team2 : match.score_team1;

    if (teamScore > opponentScore) {
      return { result: 'win', score: `${teamScore}-${opponentScore}` };
    } else if (teamScore < opponentScore) {
      return { result: 'loss', score: `${teamScore}-${opponentScore}` };
    } else {
      return { result: 'draw', score: `${teamScore}-${opponentScore}` };
    }
  };

  const getResultColor = (result: string) => {
    switch (result?.toLowerCase()) {
      case 'win':
        return 'text-green-600';
      case 'loss':
        return 'text-red-600';
      case 'draw':
        return 'text-gray-600';
      default:
        return 'text-gray-600';
    }
  };

  const getResultIcon = (result: string) => {
    switch (result?.toLowerCase()) {
      case 'win':
        return <Trophy className="w-4 h-4" />;
      case 'loss':
        return <Target className="w-4 h-4" />;
      case 'draw':
        return <Gamepad2 className="w-4 h-4" />;
      default:
        return <Gamepad2 className="w-4 h-4" />;
    }
  };

  const getMatchTypeColor = (matchType: string) => {
    switch (matchType?.toLowerCase()) {
      case 'group_stage':
        return 'bg-blue-100 text-blue-800';
      case 'knockout':
        return 'bg-red-100 text-red-800';
      case 'final':
        return 'bg-purple-100 text-purple-800';
      case 'semi_final':
        return 'bg-orange-100 text-orange-800';
      case 'quarter_final':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const draws = stats.total_matches - stats.wins - stats.losses;
  const winPercentage = stats.win_rate;
  const lossPercentage = stats.total_matches > 0 ? Math.round((stats.losses / stats.total_matches) * 100) : 0;
  const drawPercentage = stats.total_matches > 0 ? Math.round((draws / stats.total_matches) * 100) : 0;

  return (
    <AppLayout>
      <Head title="Team Analytics" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector teams={teams} selectedTeamId={team.id} currentPath="/team/analytics" />

        {/* Header */}
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Team Analytics</h1>
          <p className="text-gray-600">Performance insights and statistics for {team.name}</p>
        </div>

        {/* Key Performance Indicators */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Matches</p>
                  <p className="text-2xl font-bold">{stats.total_matches}</p>
                </div>
                <Gamepad2 className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Wins</p>
                  <p className="text-2xl font-bold text-green-600">{stats.wins}</p>
                </div>
                <Trophy className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Losses</p>
                  <p className="text-2xl font-bold text-red-600">{stats.losses}</p>
                </div>
                <Target className="w-8 h-8 text-red-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Win Rate</p>
                  <p className="text-2xl font-bold text-purple-600">{winPercentage}%</p>
                </div>
                <TrendingUp className="w-8 h-8 text-purple-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="grid gap-6 lg:grid-cols-2">
          {/* Performance Overview */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BarChart3 className="w-5 h-5" />
                Performance Overview
              </CardTitle>
              <CardDescription>
                Match results breakdown
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Wins</span>
                  <div className="flex items-center gap-2">
                    <div className="w-32 bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-green-600 h-2 rounded-full" 
                        style={{ width: `${winPercentage}%` }}
                      ></div>
                    </div>
                    <span className="text-sm font-medium w-12 text-right">{winPercentage}%</span>
                  </div>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Draws</span>
                  <div className="flex items-center gap-2">
                    <div className="w-32 bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-gray-600 h-2 rounded-full" 
                        style={{ width: `${drawPercentage}%` }}
                      ></div>
                    </div>
                    <span className="text-sm font-medium w-12 text-right">{drawPercentage}%</span>
                  </div>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Losses</span>
                  <div className="flex items-center gap-2">
                    <div className="w-32 bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-red-600 h-2 rounded-full" 
                        style={{ width: `${lossPercentage}%` }}
                      ></div>
                    </div>
                    <span className="text-sm font-medium w-12 text-right">{lossPercentage}%</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Tournament Performance */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Trophy className="w-5 h-5" />
                Tournament Performance
              </CardTitle>
              <CardDescription>
                Performance across different tournaments
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {tournamentStats.map((tournament) => (
                  <div key={tournament.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div>
                      <h4 className="font-medium">{tournament.name}</h4>
                      <p className="text-sm text-gray-500">
                        {formatDate(tournament.start_date)} - {formatDate(tournament.end_date)}
                      </p>
                    </div>
                    <div className="text-right">
                      <Badge className={getStatusColor(tournament.status)}>
                        {tournament.status}
                      </Badge>
                      <p className="text-sm text-gray-500 mt-1">
                        {tournament.matches_count} matches
                      </p>
                    </div>
                  </div>
                ))}
                {tournamentStats.length === 0 && (
                  <p className="text-sm text-gray-500 text-center py-4">
                    No tournament data available
                  </p>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Recent Matches */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Gamepad2 className="w-5 h-5" />
              Recent Matches
            </CardTitle>
            <CardDescription>
              Latest match results and performance
            </CardDescription>
          </CardHeader>
          <CardContent>
            {recentMatches.length > 0 ? (
              <div className="space-y-4">
                {recentMatches.map((match) => {
                  const result = getMatchResult(match);
                  const isTeam1 = match.team1.id === team.id;
                  const opponent = isTeam1 ? match.team2 : match.team1;

                  return (
                    <div key={match.id} className="flex items-center justify-between p-4 border rounded-lg hover:bg-gray-50">
                      <div className="flex items-center gap-4">
                        <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                          {result ? getResultIcon(result.result) : <Gamepad2 className="w-5 h-5 text-gray-600" />}
                        </div>
                        <div>
                          <h4 className="font-medium">{match.name}</h4>
                          <p className="text-sm text-gray-500">{match.tournament.name}</p>
                          <p className="text-xs text-gray-400">
                            {formatDateTime(match.scheduled_time)}
                          </p>
                        </div>
                      </div>

                      <div className="flex items-center gap-4">
                        <div className="text-center">
                          <p className="font-medium">{team.name}</p>
                          <p className="text-sm text-gray-500">Your Team</p>
                        </div>

                        <div className="text-center">
                          {result ? (
                            <div className="space-y-1">
                              <p className={`text-lg font-bold ${getResultColor(result.result)}`}>
                                {result.score}
                              </p>
                              <p className={`text-xs font-medium ${getResultColor(result.result)}`}>
                                {result.result.toUpperCase()}
                              </p>
                            </div>
                          ) : (
                            <div className="text-gray-400">VS</div>
                          )}
                        </div>

                        <div className="text-center">
                          <p className="font-medium">{opponent.name}</p>
                          <p className="text-sm text-gray-500">Opponent</p>
                        </div>

                        <div className="flex gap-2">
                          <Badge className={getStatusColor(match.status)}>
                            {match.status.replace('_', ' ')}
                          </Badge>
                          <Badge className={getMatchTypeColor(match.match_type)}>
                            {match.match_type.replace('_', ' ')}
                          </Badge>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
            ) : (
              <div className="text-center py-12">
                <Gamepad2 className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No Match Data</h3>
                <p className="text-gray-500">Your team hasn't played any matches yet</p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Performance Trends */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5" />
              Performance Trends
            </CardTitle>
            <CardDescription>
              Team performance over time
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-center py-12">
              <BarChart3 className="w-12 h-12 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Trend Analysis</h3>
              <p className="text-gray-500">
                Performance trend analysis will be available once you have more match data
              </p>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
