import React from 'react';
import { Head, useForm } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { CheckCircle, XCircle, Info } from 'lucide-react';

interface TeamInvitationProps {
    invitation: {
        id: string;
        team_name: string;
        tournament_name: string;
        registration_details: {
            name: string;
            phone: string;
        };
        invitation_date: string;
        status: 'pending' | 'accepted' | 'declined';
    };
}

export default function TeamInvitation({ invitation }: TeamInvitationProps) {
    const acceptForm = useForm({
        invitation_id: invitation.id,
    });

    const declineForm = useForm({
        invitation_id: invitation.id,
    });

    const handleAccept = () => {
        acceptForm.post(route('team.invitation.accept', invitation.id));
    };

    const handleDecline = () => {
        declineForm.post(route('team.invitation.decline', invitation.id));
    };

    const getStatusBadge = () => {
        switch (invitation.status) {
            case 'accepted':
                return <Badge className="bg-green-100 text-green-800">Accepted</Badge>;
            case 'declined':
                return <Badge className="bg-red-100 text-red-800">Declined</Badge>;
            default:
                return <Badge className="bg-yellow-100 text-yellow-800">Pending</Badge>;
        }
    };

    const getStatusIcon = () => {
        switch (invitation.status) {
            case 'accepted':
                return <CheckCircle className="h-5 w-5 text-green-600" />;
            case 'declined':
                return <XCircle className="h-5 w-5 text-red-600" />;
            default:
                return <Info className="h-5 w-5 text-yellow-600" />;
        }
    };

    const isResponded = invitation.status !== 'pending';

    return (
        <>
            <Head title="Team Captain Invitation" />
            
            <div className="min-h-screen bg-gray-50 py-8">
                <div className="mx-auto max-w-2xl px-4">
                    <Card>
                        <CardHeader className="text-center">
                            <div className="mx-auto mb-4 flex h-16 w-16 items-center justify-center rounded-full bg-blue-100">
                                {getStatusIcon()}
                            </div>
                            <CardTitle className="text-2xl">Team Captain Invitation</CardTitle>
                            <CardDescription>
                                You have been invited to become the team captain
                            </CardDescription>
                            {getStatusBadge()}
                        </CardHeader>
                        
                        <CardContent className="space-y-6">
                            {isResponded ? (
                                <Alert>
                                    <Info className="h-4 w-4" />
                                    <AlertDescription>
                                        You have already {invitation.status} this invitation.
                                    </AlertDescription>
                                </Alert>
                            ) : (
                                <Alert>
                                    <Info className="h-4 w-4" />
                                    <AlertDescription>
                                        Someone tried to register a team using your email address. 
                                        You can accept or decline this team captain role.
                                    </AlertDescription>
                                </Alert>
                            )}

                            <div className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <label className="text-sm font-medium text-gray-700">Team Name</label>
                                        <p className="text-lg font-semibold">{invitation.team_name}</p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700">Tournament</label>
                                        <p className="text-lg font-semibold">{invitation.tournament_name}</p>
                                    </div>
                                </div>

                                <div>
                                    <label className="text-sm font-medium text-gray-700">Registration Details</label>
                                    <div className="mt-2 space-y-2">
                                        <p><span className="font-medium">Name:</span> {invitation.registration_details.name}</p>
                                        <p><span className="font-medium">Phone:</span> {invitation.registration_details.phone}</p>
                                    </div>
                                </div>

                                <div>
                                    <label className="text-sm font-medium text-gray-700">Invitation Date</label>
                                    <p className="text-lg">{new Date(invitation.invitation_date).toLocaleDateString('en-US', {
                                        year: 'numeric',
                                        month: 'long',
                                        day: 'numeric',
                                        hour: '2-digit',
                                        minute: '2-digit'
                                    })}</p>
                                </div>
                            </div>

                            {!isResponded && (
                                <div className="flex gap-4 pt-4">
                                    <Button
                                        onClick={handleAccept}
                                        disabled={acceptForm.processing}
                                        className="flex-1 bg-green-600 hover:bg-green-700"
                                    >
                                        {acceptForm.processing ? 'Accepting...' : 'Accept Invitation'}
                                    </Button>
                                    <Button
                                        onClick={handleDecline}
                                        disabled={declineForm.processing}
                                        variant="outline"
                                        className="flex-1 border-red-300 text-red-700 hover:bg-red-50"
                                    >
                                        {declineForm.processing ? 'Declining...' : 'Decline Invitation'}
                                    </Button>
                                </div>
                            )}

                            <div className="pt-4 text-center">
                                <Button variant="link" onClick={() => window.history.back()}>
                                    Go Back
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </>
    );
}
