import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Calendar, Users, MapPin } from 'lucide-react';
import { Head } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';

interface Tournament {
    id: number;
    name: string;
    description: string;
    status: string;
    start_date: string;
    end_date: string;
    player_status: string;
    team: {
        id: number;
        name: string;
        logo: string;
    } | null;
    registration_date: string;
    is_team_captain: boolean;
}

interface Props {
    tournaments: Tournament[];
    user: any;
}

const statusColors = {
    'active': 'bg-green-100 text-green-800',
    'upcoming': 'bg-blue-100 text-blue-800',
    'completed': 'bg-gray-100 text-gray-800',
    'cancelled': 'bg-red-100 text-red-800',
    'draft': 'bg-yellow-100 text-yellow-800',
};

const playerStatusColors = {
    'registered': 'bg-blue-100 text-blue-800',
    'active': 'bg-green-100 text-green-800',
    'eliminated': 'bg-red-100 text-red-800',
    'withdrawn': 'bg-gray-100 text-gray-800',
    'suspended': 'bg-orange-100 text-orange-800',
    'banned': 'bg-red-100 text-red-800',
};

export default function Tournaments({ tournaments, user }: Props) {
    return (
        <AppLayout>
            <Head title="My Tournaments" />
            
            <div className="py-12">
                <div className="">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-6">
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
                                        <Trophy className="h-8 w-8 text-blue-600" />
                                        My Tournaments
                                    </h1>
                                    <p className="text-gray-600 mt-2">
                                        View all tournaments you are registered for
                                    </p>
                                </div>
                                <Badge variant="secondary" className="px-3 py-1">
                                    {tournaments.length} Tournament{tournaments.length !== 1 ? 's' : ''}
                                </Badge>
                            </div>

                            {tournaments.length === 0 ? (
                                <Card>
                                    <CardContent className="p-12 text-center">
                                        <Trophy className="h-16 w-16 text-gray-300 mx-auto mb-4" />
                                        <h3 className="text-xl font-semibold text-gray-600 mb-2">
                                            No Tournaments Yet
                                        </h3>
                                        <p className="text-gray-500">
                                            You haven't registered for any tournaments yet.
                                        </p>
                                    </CardContent>
                                </Card>
                            ) : (
                                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
                                    {tournaments.map((tournament) => (
                                        <Card key={tournament.id} className="hover:shadow-lg transition-shadow">
                                            <CardHeader className="pb-3">
                                                <div className="flex items-start justify-between">
                                                    <CardTitle className="text-xl">{tournament.name}</CardTitle>
                                                    <Badge 
                                                        className={statusColors[tournament.status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}
                                                    >
                                                        {tournament.status}
                                                    </Badge>
                                                </div>
                                                <CardDescription className="line-clamp-2">
                                                    {tournament.description}
                                                </CardDescription>
                                            </CardHeader>
                                            <CardContent className="space-y-4">
                                                {/* Tournament Dates */}
                                                <div className="flex items-center gap-2 text-sm text-gray-600">
                                                    <Calendar className="h-4 w-4" />
                                                    <span>
                                                        {new Date(tournament.start_date).toLocaleDateString()} - {' '}
                                                        {new Date(tournament.end_date).toLocaleDateString()}
                                                    </span>
                                                </div>

                                                {/* Team Information */}
                                                {tournament.team && (
                                                    <div className="flex items-center gap-2 text-sm text-gray-600">
                                                        <Users className="h-4 w-4" />
                                                        <span>Team: {tournament.team.name}</span>
                                                        {tournament.is_team_captain && (
                                                            <Badge variant="outline" className="text-xs">
                                                                Captain
                                                            </Badge>
                                                        )}
                                                    </div>
                                                )}

                                                {/* Player Status */}
                                                <div className="flex items-center justify-between">
                                                    <div className="flex items-center gap-2">
                                                        <span className="text-sm text-gray-500">Status:</span>
                                                        <Badge 
                                                            className={playerStatusColors[tournament.player_status as keyof typeof playerStatusColors] || 'bg-gray-100 text-gray-800'}
                                                        >
                                                            {tournament.player_status}
                                                        </Badge>
                                                    </div>
                                                </div>

                                                {/* Registration Date */}
                                                <div className="text-xs text-gray-500 border-t pt-3">
                                                    Registered: {new Date(tournament.registration_date).toLocaleDateString()}
                                                </div>
                                            </CardContent>
                                        </Card>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}