import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { UsersRound, Crown, User, Shirt } from 'lucide-react';
import { Head } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';

interface TeamMember {
    id: number;
    display_name: string;
    user_name: string;
    avatar: string;
    jersey_number: string;
    football_position: string;
    is_team_captain: boolean;
    status: string;
}

interface Team {
    id: number;
    name: string;
    description: string;
    logo: string;
    status: string;
    tournament: {
        id: number;
        name: string;
    };
    members: TeamMember[];
    is_captain: boolean;
}

interface Props {
    team: Team | null;
    message?: string;
    user: any;
}

const positionColors = {
    'Goalkeeper': 'bg-yellow-100 text-yellow-800',
    'Defender': 'bg-blue-100 text-blue-800',
    'Midfielder': 'bg-green-100 text-green-800',
    'Forward': 'bg-red-100 text-red-800',
};

const statusColors = {
    'registered': 'bg-blue-100 text-blue-800',
    'active': 'bg-green-100 text-green-800',
    'eliminated': 'bg-red-100 text-red-800',
    'withdrawn': 'bg-gray-100 text-gray-800',
    'suspended': 'bg-orange-100 text-orange-800',
};

export default function Team({ team, message, user }: Props) {
    return (
        <AppLayout>
            <Head title="My Team" />
            
            <div className="py-6 sm:py-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-4 sm:p-6">
                            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-6">
                                <div>
                                    <h1 className="text-2xl sm:text-3xl font-bold text-gray-900 flex items-center gap-2 sm:gap-3">
                                        <UsersRound className="h-6 w-6 sm:h-8 sm:w-8 text-blue-600" />
                                        My Team
                                    </h1>
                                    <p className="text-gray-600 mt-2 text-sm sm:text-base">
                                        View your team information and members
                                    </p>
                                </div>
                            </div>

                            {!team ? (
                                <Card>
                                    <CardContent className="p-6 sm:p-12 text-center">
                                        <UsersRound className="h-12 w-12 sm:h-16 sm:w-16 text-gray-300 mx-auto mb-4" />
                                        <h3 className="text-lg sm:text-xl font-semibold text-gray-600 mb-2">
                                            No Team Assigned
                                        </h3>
                                        <p className="text-gray-500 text-sm sm:text-base">
                                            {message || "You are not currently assigned to a team."}
                                        </p>
                                    </CardContent>
                                </Card>
                            ) : (
                                <div className="space-y-6">
                                    {/* Team Header */}
                                    <Card>
                                        <CardHeader className="pb-4">
                                            <div className="flex flex-col sm:flex-row sm:items-center gap-4">
                                                {team.logo && (
                                                    <img 
                                                        src={team.logo} 
                                                        alt={team.name}
                                                        className="w-12 h-12 sm:w-16 sm:h-16 rounded-lg object-cover mx-auto sm:mx-0"
                                                    />
                                                )}
                                                <div className="flex-1 text-center sm:text-left">
                                                    <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-3">
                                                        <CardTitle className="text-xl sm:text-2xl">{team.name}</CardTitle>
                                                        {team.is_captain && (
                                                            <Badge className="bg-yellow-100 text-yellow-800 w-fit mx-auto sm:mx-0">
                                                                <Crown className="h-3 w-3 mr-1" />
                                                                Captain
                                                            </Badge>
                                                        )}
                                                    </div>
                                                    <CardDescription className="text-base sm:text-lg mt-1">
                                                        {team.description}
                                                    </CardDescription>
                                                    <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4 mt-2">
                                                        <Badge variant="outline" className="w-fit mx-auto sm:mx-0">
                                                            Tournament: {team.tournament.name}
                                                        </Badge>
                                                        <Badge 
                                                            className={`w-fit mx-auto sm:mx-0 ${statusColors[team.status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}`}
                                                        >
                                                            {team.status}
                                                        </Badge>
                                                    </div>
                                                </div>
                                            </div>
                                        </CardHeader>
                                    </Card>

                                    {/* Team Members */}
                                    <Card>
                                        <CardHeader>
                                            <CardTitle className="flex items-center gap-2">
                                                <User className="h-5 w-5" />
                                                Team Members ({team.members.length})
                                            </CardTitle>
                                            <CardDescription>
                                                All players in your team
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent>
                                            {team.members.length === 0 ? (
                                                <div className="text-center py-8">
                                                    <User className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                                                    <p className="text-gray-500">No team members found</p>
                                                </div>
                                            ) : (
                                                <div className="grid gap-4 grid-cols-1 sm:grid-cols-2 lg:grid-cols-3">
                                                    {team.members.map((member) => (
                                                        <Card key={member.id} className="border border-gray-200">
                                                            <CardContent className="p-3 sm:p-4">
                                                                <div className="flex items-center gap-3">
                                                                    {member.avatar ? (
                                                                        <img 
                                                                            src={member.avatar} 
                                                                            alt={member.display_name}
                                                                            className="w-10 h-10 sm:w-12 sm:h-12 rounded-full object-cover"
                                                                        />
                                                                    ) : (
                                                                        <div className="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-gray-200 flex items-center justify-center">
                                                                            <User className="h-5 w-5 sm:h-6 sm:w-6 text-gray-500" />
                                                                        </div>
                                                                    )}
                                                                    <div className="flex-1 min-w-0">
                                                                        <div className="flex items-center gap-2">
                                                                            <h3 className="font-semibold text-gray-900 text-sm sm:text-base truncate">
                                                                                {member.display_name}
                                                                            </h3>
                                                                            {member.is_team_captain && (
                                                                                <Crown className="h-3 w-3 sm:h-4 sm:w-4 text-yellow-600 flex-shrink-0" />
                                                                            )}
                                                                        </div>
                                                                        <p className="text-xs sm:text-sm text-gray-600 truncate">
                                                                            {member.user_name}
                                                                        </p>
                                                                        <div className="flex flex-wrap items-center gap-1 sm:gap-2 mt-2">
                                                                            {member.football_position && (
                                                                                <Badge 
                                                                                    className={`text-xs ${positionColors[member.football_position as keyof typeof positionColors] || 'bg-gray-100 text-gray-800'}`}
                                                                                >
                                                                                    {member.football_position}
                                                                                </Badge>
                                                                            )}
                                                                            {member.jersey_number && (
                                                                                <Badge variant="outline" className="gap-1 text-xs">
                                                                                    <Shirt className="h-2 w-2 sm:h-3 sm:w-3" />
                                                                                    #{member.jersey_number}
                                                                                </Badge>
                                                                            )}
                                                                        </div>
                                                                        <Badge 
                                                                            className={`mt-2 text-xs ${statusColors[member.status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}`}
                                                                        >
                                                                            {member.status}
                                                                        </Badge>
                                                                    </div>
                                                                </div>
                                                            </CardContent>
                                                        </Card>
                                                    ))}
                                                </div>
                                            )}
                                        </CardContent>
                                    </Card>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}