import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { User, Edit, Trophy, Star, Globe, Phone, Mail, MapPin, Calendar, Gamepad2, Award } from 'lucide-react';
import { Head, useForm } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { useState } from 'react';

interface PlayerProfile {
    id: number;
    tournament: {
        id: number;
        name: string;
    };
    team: {
        id: number;
        name: string;
    } | null;
    display_name: string;
    bio: string;
    avatar: string;
    country: string;
    city: string;
    date_of_birth: string;
    gender: string;
    phone: string;
    discord_username: string;
    steam_id: string;
    twitch_username: string;
    youtube_channel: string;
    twitter_handle: string;
    instagram_handle: string;
    website: string;
    player_type: string;
    experience_level: string;
    primary_game: string;
    football_position: string;
    jersey_number: string;
    is_team_captain: boolean;
    global_rating: number;
    global_rank: number;
    tournament_rating: number;
    tournament_rank: number;
    status: string;
    verification_status: string;
    registered_at: string;
    tournament_wins: number;
    tournament_losses: number;
    tournament_draws: number;
    tournament_win_rate: number;
    tournament_matches_played: number;
    tournament_matches_won: number;
    total_earnings: number;
    tournaments_won: number;
    achievements: any[];
    performance_metrics: any;
}

interface Props {
    profiles: PlayerProfile[];
    user: any;
}

const experienceLevels = [
    { value: 'beginner', label: 'Beginner' },
    { value: 'intermediate', label: 'Intermediate' },
    { value: 'advanced', label: 'Advanced' },
    { value: 'expert', label: 'Expert' },
];

const playerTypes = [
    { value: 'amateur', label: 'Amateur' },
    { value: 'semi_pro', label: 'Semi-Professional' },
    { value: 'professional', label: 'Professional' },
    { value: 'legend', label: 'Legend' },
];

const genderOptions = [
    { value: 'male', label: 'Male' },
    { value: 'female', label: 'Female' },
    { value: 'other', label: 'Other' },
    { value: 'prefer_not_to_say', label: 'Prefer not to say' },
];

const footballPositions = [
    { value: 'Goalkeeper', label: 'Goalkeeper' },
    { value: 'Defender', label: 'Defender' },
    { value: 'Midfielder', label: 'Midfielder' },
    { value: 'Forward', label: 'Forward' },
];

export default function Profile({ profiles, user }: Props) {
    const [editingProfile, setEditingProfile] = useState<number | null>(null);
    
    const { data, setData, put, processing, errors, reset } = useForm({
        player_id: 0,
        display_name: '',
        bio: '',
        avatar: '',
        country: '',
        city: '',
        date_of_birth: '',
        gender: '',
        phone: '',
        discord_username: '',
        steam_id: '',
        twitch_username: '',
        youtube_channel: '',
        twitter_handle: '',
        instagram_handle: '',
        website: '',
    });

    const handleEdit = (profile: PlayerProfile) => {
        setEditingProfile(profile.id);
        setData({
            player_id: profile.id,
            display_name: profile.display_name || '',
            bio: profile.bio || '',
            avatar: profile.avatar || '',
            country: profile.country || '',
            city: profile.city || '',
            date_of_birth: profile.date_of_birth || '',
            gender: profile.gender || '',
            phone: profile.phone || '',
            discord_username: profile.discord_username || '',
            steam_id: profile.steam_id || '',
            twitch_username: profile.twitch_username || '',
            youtube_channel: profile.youtube_channel || '',
            twitter_handle: profile.twitter_handle || '',
            instagram_handle: profile.instagram_handle || '',
            website: profile.website || '',
        });
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        put(route('player.profile.update'), {
            onSuccess: () => {
                setEditingProfile(null);
                reset();
            }
        });
    };

    const handleCancel = () => {
        setEditingProfile(null);
        reset();
    };

    return (
        <AppLayout>
            <Head title="My Profile" />
            
            <div className="py-12">
                <div className="">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-6">
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
                                        <User className="h-8 w-8 text-blue-600" />
                                        My Profile
                                    </h1>
                                    <p className="text-gray-600 mt-2">
                                        Manage your player profiles across tournaments
                                    </p>
                                </div>
                            </div>

                            {profiles.length === 0 ? (
                                <Card>
                                    <CardContent className="p-12 text-center">
                                        <User className="h-16 w-16 text-gray-300 mx-auto mb-4" />
                                        <h3 className="text-xl font-semibold text-gray-600 mb-2">
                                            No Player Profiles
                                        </h3>
                                        <p className="text-gray-500">
                                            You don't have any player profiles yet.
                                        </p>
                                    </CardContent>
                                </Card>
                            ) : (
                                <div className="space-y-8">
                                    {profiles.map((profile) => (
                                        <Card key={profile.id}>
                                            <CardHeader>
                                                <div className="flex items-center justify-between">
                                                    <div>
                                                        <CardTitle className="text-2xl flex items-center gap-3">
                                                            {profile.avatar ? (
                                                                <img 
                                                                    src={profile.avatar} 
                                                                    alt={profile.display_name}
                                                                    className="w-12 h-12 rounded-full object-cover"
                                                                />
                                                            ) : (
                                                                <div className="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center">
                                                                    <User className="h-6 w-6 text-gray-500" />
                                                                </div>
                                                            )}
                                                            {profile.display_name}
                                                            {profile.is_team_captain && (
                                                                <Badge className="bg-yellow-100 text-yellow-800">
                                                                    Captain
                                                                </Badge>
                                                            )}
                                                        </CardTitle>
                                                        <CardDescription>
                                                            Tournament: {profile.tournament.name}
                                                            {profile.team && ` • Team: ${profile.team.name}`}
                                                        </CardDescription>
                                                    </div>
                                                    <Button 
                                                        variant="outline" 
                                                        onClick={() => handleEdit(profile)}
                                                        disabled={editingProfile === profile.id}
                                                    >
                                                        <Edit className="h-4 w-4 mr-2" />
                                                        Edit
                                                    </Button>
                                                </div>
                                            </CardHeader>
                                            <CardContent>
                                                {editingProfile === profile.id ? (
                                                    // Edit Form
                                                    <form onSubmit={handleSubmit} className="space-y-6">
                                                        <div className="grid gap-6 md:grid-cols-2">
                                                            <div className="space-y-2">
                                                                <Label htmlFor="display_name">Display Name</Label>
                                                                <Input
                                                                    id="display_name"
                                                                    value={data.display_name}
                                                                    onChange={(e) => setData('display_name', e.target.value)}
                                                                    error={errors.display_name}
                                                                />
                                                            </div>
                                                            
                                                            <div className="space-y-2">
                                                                <Label htmlFor="gender">Gender</Label>
                                                                <Select 
                                                                    value={data.gender} 
                                                                    onValueChange={(value) => setData('gender', value)}
                                                                >
                                                                    <SelectTrigger>
                                                                        <SelectValue placeholder="Select gender" />
                                                                    </SelectTrigger>
                                                                    <SelectContent>
                                                                        {genderOptions.map((option) => (
                                                                            <SelectItem key={option.value} value={option.value}>
                                                                                {option.label}
                                                                            </SelectItem>
                                                                        ))}
                                                                    </SelectContent>
                                                                </Select>
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="country">Country</Label>
                                                                <Input
                                                                    id="country"
                                                                    value={data.country}
                                                                    onChange={(e) => setData('country', e.target.value)}
                                                                    error={errors.country}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="city">City</Label>
                                                                <Input
                                                                    id="city"
                                                                    value={data.city}
                                                                    onChange={(e) => setData('city', e.target.value)}
                                                                    error={errors.city}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="phone">Phone</Label>
                                                                <Input
                                                                    id="phone"
                                                                    value={data.phone}
                                                                    onChange={(e) => setData('phone', e.target.value)}
                                                                    error={errors.phone}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="date_of_birth">Date of Birth</Label>
                                                                <Input
                                                                    id="date_of_birth"
                                                                    type="date"
                                                                    value={data.date_of_birth}
                                                                    onChange={(e) => setData('date_of_birth', e.target.value)}
                                                                    error={errors.date_of_birth}
                                                                />
                                                            </div>
                                                        </div>

                                                        <div className="space-y-2">
                                                            <Label htmlFor="bio">Bio</Label>
                                                            <Textarea
                                                                id="bio"
                                                                value={data.bio}
                                                                onChange={(e) => setData('bio', e.target.value)}
                                                                rows={4}
                                                                error={errors.bio}
                                                            />
                                                        </div>

                                                        <div className="grid gap-6 md:grid-cols-2">
                                                            <div className="space-y-2">
                                                                <Label htmlFor="discord_username">Discord Username</Label>
                                                                <Input
                                                                    id="discord_username"
                                                                    value={data.discord_username}
                                                                    onChange={(e) => setData('discord_username', e.target.value)}
                                                                    error={errors.discord_username}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="steam_id">Steam ID</Label>
                                                                <Input
                                                                    id="steam_id"
                                                                    value={data.steam_id}
                                                                    onChange={(e) => setData('steam_id', e.target.value)}
                                                                    error={errors.steam_id}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="twitch_username">Twitch Username</Label>
                                                                <Input
                                                                    id="twitch_username"
                                                                    value={data.twitch_username}
                                                                    onChange={(e) => setData('twitch_username', e.target.value)}
                                                                    error={errors.twitch_username}
                                                                />
                                                            </div>

                                                            <div className="space-y-2">
                                                                <Label htmlFor="website">Website</Label>
                                                                <Input
                                                                    id="website"
                                                                    type="url"
                                                                    value={data.website}
                                                                    onChange={(e) => setData('website', e.target.value)}
                                                                    error={errors.website}
                                                                />
                                                            </div>
                                                        </div>

                                                        <div className="flex justify-end gap-4">
                                                            <Button 
                                                                type="button" 
                                                                variant="outline" 
                                                                onClick={handleCancel}
                                                                disabled={processing}
                                                            >
                                                                Cancel
                                                            </Button>
                                                            <Button type="submit" disabled={processing}>
                                                                {processing ? 'Saving...' : 'Save Changes'}
                                                            </Button>
                                                        </div>
                                                    </form>
                                                ) : (
                                                    // View Mode
                                                    <div className="space-y-6">
                                                        {/* Stats Overview */}
                                                        <div className="grid gap-4 md:grid-cols-4">
                                                            <Card className="border border-gray-200">
                                                                <CardContent className="p-4 text-center">
                                                                    <Trophy className="h-8 w-8 text-yellow-600 mx-auto mb-2" />
                                                                    <div className="text-2xl font-bold">{profile.tournament_wins}</div>
                                                                    <div className="text-sm text-gray-600">Wins</div>
                                                                </CardContent>
                                                            </Card>
                                                            <Card className="border border-gray-200">
                                                                <CardContent className="p-4 text-center">
                                                                    <Gamepad2 className="h-8 w-8 text-blue-600 mx-auto mb-2" />
                                                                    <div className="text-2xl font-bold">{profile.tournament_matches_played}</div>
                                                                    <div className="text-sm text-gray-600">Matches</div>
                                                                </CardContent>
                                                            </Card>
                                                            <Card className="border border-gray-200">
                                                                <CardContent className="p-4 text-center">
                                                                    <Star className="h-8 w-8 text-purple-600 mx-auto mb-2" />
                                                                    <div className="text-2xl font-bold">{profile.tournament_rating}</div>
                                                                    <div className="text-sm text-gray-600">Rating</div>
                                                                </CardContent>
                                                            </Card>
                                                            <Card className="border border-gray-200">
                                                                <CardContent className="p-4 text-center">
                                                                    <Award className="h-8 w-8 text-green-600 mx-auto mb-2" />
                                                                    <div className="text-2xl font-bold">{Math.round(profile.tournament_win_rate)}%</div>
                                                                    <div className="text-sm text-gray-600">Win Rate</div>
                                                                </CardContent>
                                                            </Card>
                                                        </div>

                                                        {/* Profile Information */}
                                                        <div className="grid gap-6 md:grid-cols-2">
                                                            <div className="space-y-4">
                                                                <h3 className="text-lg font-semibold">Personal Information</h3>
                                                                {profile.bio && (
                                                                    <div>
                                                                        <Label className="text-sm font-medium text-gray-700">Bio</Label>
                                                                        <p className="text-gray-900 mt-1">{profile.bio}</p>
                                                                    </div>
                                                                )}
                                                                {(profile.country || profile.city) && (
                                                                    <div className="flex items-center gap-2">
                                                                        <MapPin className="h-4 w-4 text-gray-500" />
                                                                        <span>{[profile.city, profile.country].filter(Boolean).join(', ')}</span>
                                                                    </div>
                                                                )}
                                                                {profile.date_of_birth && (
                                                                    <div className="flex items-center gap-2">
                                                                        <Calendar className="h-4 w-4 text-gray-500" />
                                                                        <span>{new Date(profile.date_of_birth).toLocaleDateString()}</span>
                                                                    </div>
                                                                )}
                                                                {profile.phone && (
                                                                    <div className="flex items-center gap-2">
                                                                        <Phone className="h-4 w-4 text-gray-500" />
                                                                        <span>{profile.phone}</span>
                                                                    </div>
                                                                )}
                                                            </div>

                                                            <div className="space-y-4">
                                                                <h3 className="text-lg font-semibold">Gaming Information</h3>
                                                                <div className="space-y-2">
                                                                    {profile.player_type && (
                                                                        <Badge variant="outline">{profile.player_type.replace('_', ' ')}</Badge>
                                                                    )}
                                                                    {profile.experience_level && (
                                                                        <Badge variant="outline">{profile.experience_level}</Badge>
                                                                    )}
                                                                    {profile.football_position && (
                                                                        <Badge variant="outline">{profile.football_position}</Badge>
                                                                    )}
                                                                </div>
                                                                
                                                                {profile.discord_username && (
                                                                    <div>
                                                                        <Label className="text-sm font-medium text-gray-700">Discord</Label>
                                                                        <p className="text-gray-900 mt-1">{profile.discord_username}</p>
                                                                    </div>
                                                                )}
                                                                {profile.steam_id && (
                                                                    <div>
                                                                        <Label className="text-sm font-medium text-gray-700">Steam ID</Label>
                                                                        <p className="text-gray-900 mt-1">{profile.steam_id}</p>
                                                                    </div>
                                                                )}
                                                                {profile.website && (
                                                                    <div className="flex items-center gap-2">
                                                                        <Globe className="h-4 w-4 text-gray-500" />
                                                                        <a 
                                                                            href={profile.website} 
                                                                            target="_blank" 
                                                                            rel="noopener noreferrer"
                                                                            className="text-blue-600 hover:underline"
                                                                        >
                                                                            {profile.website}
                                                                        </a>
                                                                    </div>
                                                                )}
                                                            </div>
                                                        </div>
                                                    </div>
                                                )}
                                            </CardContent>
                                        </Card>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}