import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Progress } from '@/components/ui/progress';
import { 
    TrendingUp, 
    TrendingDown, 
    Trophy, 
    Target, 
    Award, 
    Star, 
    Zap, 
    BarChart3,
    Calendar,
    User,
    Gamepad2,
    Activity,
    Flame
} from 'lucide-react';
import { Head } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';

interface OverallStats {
    total_matches: number;
    wins: number;
    losses: number;
    draws: number;
    win_rate: number;
    current_rating: number;
    current_rank: number | null;
    current_streak: number;
    longest_win_streak: number;
    longest_loss_streak: number;
}

interface RecentForm {
    result: 'W' | 'L' | 'D';
    opponent: string;
    score: string;
    date: string;
    tournament: string;
}

interface TournamentPerformance {
    tournament: string;
    wins: number;
    losses: number;
    draws: number;
    total: number;
    win_rate: number;
    rating: number;
    rank: number | null;
    status: string;
}

interface RatingProgression {
    date: string;
    rating: number;
}

interface RecentMatch {
    id: number;
    opponent: string;
    result: 'W' | 'L' | 'D';
    score: string;
    tournament: string;
    date: string;
}

interface Props {
    overallStats: OverallStats;
    recentForm: RecentForm[];
    tournamentPerformance: TournamentPerformance[];
    ratingProgression: RatingProgression[];
    recentMatches: RecentMatch[];
    user: any;
}

const resultColors = {
    'W': 'bg-green-100 text-green-800',
    'L': 'bg-red-100 text-red-800',
    'D': 'bg-gray-100 text-gray-800',
};

const resultIcons = {
    'W': Trophy,
    'L': TrendingDown,
    'D': Target,
};

export default function Performance({ 
    overallStats, 
    recentForm, 
    tournamentPerformance, 
    ratingProgression,
    recentMatches,
    user 
}: Props) {
    const formatStreak = (streak: number) => {
        if (streak === 0) return 'No streak';
        if (streak > 0) return `${streak}W`;
        return `${Math.abs(streak)}L`;
    };

    const getStreakColor = (streak: number) => {
        if (streak === 0) return 'text-gray-500';
        if (streak > 0) return 'text-green-600';
        return 'text-red-600';
    };

    const getStreakIcon = (streak: number) => {
        if (streak === 0) return Target;
        if (streak > 0) return TrendingUp;
        return TrendingDown;
    };

    return (
        <AppLayout>
            <Head title="Performance Analytics" />
            
            <div className="py-12">
                <div className="">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-6">
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
                                        <BarChart3 className="h-8 w-8 text-blue-600" />
                                        Performance Analytics
                                    </h1>
                                    <p className="text-gray-600 mt-2">
                                        Detailed analysis of your gaming performance
                                    </p>
                                </div>
                            </div>

                            <div className="space-y-8">
                                {/* Overall Statistics */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <Award className="h-6 w-6 text-gold-600" />
                                        Overall Statistics
                                    </h2>
                                    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-5">
                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <Gamepad2 className="h-8 w-8 text-blue-600 mx-auto mb-2" />
                                                <div className="text-3xl font-bold text-gray-900">{overallStats.total_matches}</div>
                                                <div className="text-sm text-gray-600">Total Matches</div>
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <Trophy className="h-8 w-8 text-green-600 mx-auto mb-2" />
                                                <div className="text-3xl font-bold text-green-600">{overallStats.wins}</div>
                                                <div className="text-sm text-gray-600">Wins</div>
                                                <div className="text-xs text-gray-500 mt-1">
                                                    {overallStats.draws} draws, {overallStats.losses} losses
                                                </div>
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <Target className="h-8 w-8 text-purple-600 mx-auto mb-2" />
                                                <div className="text-3xl font-bold text-purple-600">{overallStats.win_rate}%</div>
                                                <div className="text-sm text-gray-600">Win Rate</div>
                                                <Progress value={overallStats.win_rate} className="mt-2" />
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <Star className="h-8 w-8 text-yellow-600 mx-auto mb-2" />
                                                <div className="text-3xl font-bold text-yellow-600">{Math.round(overallStats.current_rating)}</div>
                                                <div className="text-sm text-gray-600">Current Rating</div>
                                                {overallStats.current_rank && (
                                                    <div className="text-xs text-gray-500 mt-1">
                                                        Rank #{overallStats.current_rank}
                                                    </div>
                                                )}
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <div className={`mx-auto mb-2 ${getStreakColor(overallStats.current_streak)}`}>
                                                    {(() => {
                                                        const StreakIcon = getStreakIcon(overallStats.current_streak);
                                                        return <StreakIcon className="h-8 w-8" />;
                                                    })()}
                                                </div>
                                                <div className={`text-3xl font-bold ${getStreakColor(overallStats.current_streak)}`}>
                                                    {formatStreak(overallStats.current_streak)}
                                                </div>
                                                <div className="text-sm text-gray-600">Current Streak</div>
                                            </CardContent>
                                        </Card>
                                    </div>
                                </div>

                                {/* Recent Form */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <Activity className="h-6 w-6 text-orange-600" />
                                        Recent Form
                                    </h2>
                                    <Card>
                                        <CardContent className="p-6">
                                            {recentForm.length === 0 ? (
                                                <div className="text-center py-8">
                                                    <Activity className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                                                    <p className="text-gray-500">No recent matches to display</p>
                                                </div>
                                            ) : (
                                                <div className="space-y-4">
                                                    <div className="flex items-center gap-2 mb-4">
                                                        <span className="text-sm font-medium text-gray-700">Last 5 matches:</span>
                                                        <div className="flex gap-1">
                                                            {recentForm.map((match, index) => (
                                                                <Badge 
                                                                    key={index}
                                                                    className={resultColors[match.result]}
                                                                >
                                                                    {match.result}
                                                                </Badge>
                                                            ))}
                                                        </div>
                                                    </div>
                                                    
                                                    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                                                        {recentForm.map((match, index) => {
                                                            const ResultIcon = resultIcons[match.result];
                                                            return (
                                                                <Card key={index} className="border border-gray-200">
                                                                    <CardContent className="p-4">
                                                                        <div className="flex items-center justify-between mb-2">
                                                                            <Badge className={resultColors[match.result]}>
                                                                                <ResultIcon className="h-3 w-3 mr-1" />
                                                                                {match.result}
                                                                            </Badge>
                                                                            <span className="text-sm font-medium">{match.score}</span>
                                                                        </div>
                                                                        <p className="text-sm font-medium text-gray-900">vs {match.opponent}</p>
                                                                        <p className="text-xs text-gray-500 mt-1">{match.tournament}</p>
                                                                        <p className="text-xs text-gray-400">
                                                                            {new Date(match.date).toLocaleDateString()}
                                                                        </p>
                                                                    </CardContent>
                                                                </Card>
                                                            );
                                                        })}
                                                    </div>
                                                </div>
                                            )}
                                        </CardContent>
                                    </Card>
                                </div>

                                {/* Performance Streaks */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <Flame className="h-6 w-6 text-red-600" />
                                        Performance Streaks
                                    </h2>
                                    <div className="grid gap-4 md:grid-cols-3">
                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <TrendingUp className="h-8 w-8 text-green-600 mx-auto mb-2" />
                                                <div className="text-2xl font-bold text-green-600">{overallStats.longest_win_streak}</div>
                                                <div className="text-sm text-gray-600">Longest Win Streak</div>
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <TrendingDown className="h-8 w-8 text-red-600 mx-auto mb-2" />
                                                <div className="text-2xl font-bold text-red-600">{overallStats.longest_loss_streak}</div>
                                                <div className="text-sm text-gray-600">Longest Loss Streak</div>
                                            </CardContent>
                                        </Card>

                                        <Card className="hover:shadow-md transition-shadow">
                                            <CardContent className="p-6 text-center">
                                                <div className={`mx-auto mb-2 ${getStreakColor(overallStats.current_streak)}`}>
                                                    {(() => {
                                                        const StreakIcon = getStreakIcon(overallStats.current_streak);
                                                        return <StreakIcon className="h-8 w-8" />;
                                                    })()}
                                                </div>
                                                <div className={`text-2xl font-bold ${getStreakColor(overallStats.current_streak)}`}>
                                                    {formatStreak(overallStats.current_streak)}
                                                </div>
                                                <div className="text-sm text-gray-600">Current Streak</div>
                                            </CardContent>
                                        </Card>
                                    </div>
                                </div>

                                {/* Rating Progression */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <TrendingUp className="h-6 w-6 text-blue-600" />
                                        Rating Progression
                                    </h2>
                                    <Card>
                                        <CardContent className="p-6">
                                            <div className="space-y-4">
                                                <div className="flex items-center justify-between text-sm text-gray-600">
                                                    <span>7 weeks ago</span>
                                                    <span>Current</span>
                                                </div>
                                                <div className="relative h-64 flex items-end justify-between bg-gray-50 rounded-lg p-4">
                                                    {ratingProgression.map((point, index) => {
                                                        const maxRating = Math.max(...ratingProgression.map(p => p.rating));
                                                        const minRating = Math.min(...ratingProgression.map(p => p.rating));
                                                        const height = ((point.rating - minRating) / (maxRating - minRating)) * 200 + 20;
                                                        
                                                        return (
                                                            <div key={index} className="flex flex-col items-center gap-2">
                                                                <div className="relative group">
                                                                    <div 
                                                                        className="w-3 bg-blue-600 rounded-t cursor-pointer hover:bg-blue-700 transition-colors"
                                                                        style={{ height: `${height}px` }}
                                                                    />
                                                                    <div className="absolute -top-8 left-1/2 transform -translate-x-1/2 bg-black text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap">
                                                                        {point.rating}
                                                                    </div>
                                                                </div>
                                                                <span className="text-xs text-gray-500 transform -rotate-45 origin-top-left">
                                                                    {point.date}
                                                                </span>
                                                            </div>
                                                        );
                                                    })}
                                                </div>
                                                <div className="text-center">
                                                    <p className="text-sm text-gray-600">
                                                        Rating trend over the last 7 weeks
                                                    </p>
                                                </div>
                                            </div>
                                        </CardContent>
                                    </Card>
                                </div>

                                {/* Tournament Performance */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <Trophy className="h-6 w-6 text-yellow-600" />
                                        Tournament Performance
                                    </h2>
                                    <Card>
                                        <CardContent className="p-6">
                                            {tournamentPerformance.length === 0 ? (
                                                <div className="text-center py-8">
                                                    <Trophy className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                                                    <p className="text-gray-500">No tournament data available</p>
                                                </div>
                                            ) : (
                                                <div className="overflow-x-auto">
                                                    <Table>
                                                        <TableHeader>
                                                            <TableRow>
                                                                <TableHead>Tournament</TableHead>
                                                                <TableHead className="text-center">Matches</TableHead>
                                                                <TableHead className="text-center">W-L-D</TableHead>
                                                                <TableHead className="text-center">Win Rate</TableHead>
                                                                <TableHead className="text-center">Rating</TableHead>
                                                                <TableHead className="text-center">Rank</TableHead>
                                                                <TableHead className="text-center">Status</TableHead>
                                                            </TableRow>
                                                        </TableHeader>
                                                        <TableBody>
                                                            {tournamentPerformance.map((tournament, index) => (
                                                                <TableRow key={index}>
                                                                    <TableCell className="font-medium">
                                                                        {tournament.tournament}
                                                                    </TableCell>
                                                                    <TableCell className="text-center">
                                                                        {tournament.total}
                                                                    </TableCell>
                                                                    <TableCell className="text-center">
                                                                        <span className="text-green-600 font-medium">{tournament.wins}</span>-
                                                                        <span className="text-red-600 font-medium">{tournament.losses}</span>-
                                                                        <span className="text-gray-600 font-medium">{tournament.draws}</span>
                                                                    </TableCell>
                                                                    <TableCell className="text-center">
                                                                        <div className="flex items-center justify-center gap-2">
                                                                            <span className="font-medium">{tournament.win_rate}%</span>
                                                                            <Progress value={tournament.win_rate} className="w-12" />
                                                                        </div>
                                                                    </TableCell>
                                                                    <TableCell className="text-center font-medium">
                                                                        {Math.round(tournament.rating)}
                                                                    </TableCell>
                                                                    <TableCell className="text-center">
                                                                        {tournament.rank ? `#${tournament.rank}` : '-'}
                                                                    </TableCell>
                                                                    <TableCell className="text-center">
                                                                        <Badge variant="outline" className="capitalize">
                                                                            {tournament.status}
                                                                        </Badge>
                                                                    </TableCell>
                                                                </TableRow>
                                                            ))}
                                                        </TableBody>
                                                    </Table>
                                                </div>
                                            )}
                                        </CardContent>
                                    </Card>
                                </div>

                                {/* Recent Matches Table */}
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                        <Calendar className="h-6 w-6 text-green-600" />
                                        Recent Match History
                                    </h2>
                                    <Card>
                                        <CardContent className="p-6">
                                            {recentMatches.length === 0 ? (
                                                <div className="text-center py-8">
                                                    <Gamepad2 className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                                                    <p className="text-gray-500">No recent matches to display</p>
                                                </div>
                                            ) : (
                                                <div className="overflow-x-auto">
                                                    <Table>
                                                        <TableHeader>
                                                            <TableRow>
                                                                <TableHead>Date</TableHead>
                                                                <TableHead>Opponent</TableHead>
                                                                <TableHead className="text-center">Result</TableHead>
                                                                <TableHead className="text-center">Score</TableHead>
                                                                <TableHead>Tournament</TableHead>
                                                            </TableRow>
                                                        </TableHeader>
                                                        <TableBody>
                                                            {recentMatches.map((match, index) => {
                                                                const ResultIcon = resultIcons[match.result];
                                                                return (
                                                                    <TableRow key={index}>
                                                                        <TableCell>
                                                                            {new Date(match.date).toLocaleDateString()}
                                                                        </TableCell>
                                                                        <TableCell className="font-medium">
                                                                            {match.opponent}
                                                                        </TableCell>
                                                                        <TableCell className="text-center">
                                                                            <Badge className={resultColors[match.result]}>
                                                                                <ResultIcon className="h-3 w-3 mr-1" />
                                                                                {match.result}
                                                                            </Badge>
                                                                        </TableCell>
                                                                        <TableCell className="text-center font-medium">
                                                                            {match.score}
                                                                        </TableCell>
                                                                        <TableCell className="text-gray-600">
                                                                            {match.tournament}
                                                                        </TableCell>
                                                                    </TableRow>
                                                                );
                                                            })}
                                                        </TableBody>
                                                    </Table>
                                                </div>
                                            )}
                                        </CardContent>
                                    </Card>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}