import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Gamepad, Clock, Trophy, Calendar, User, Users } from 'lucide-react';
import { Head } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';

interface Match {
    id: number;
    tournament: {
        id: number;
        name: string;
    };
    bracket: {
        id: number;
        name: string;
    } | null;
    round: number;
    match_number: number;
    player1: {
        id: number;
        display_name: string;
        user_name: string;
    } | null;
    player2: {
        id: number;
        display_name: string;
        user_name: string;
    } | null;
    team1: {
        id: number;
        name: string;
        logo: string;
    } | null;
    team2: {
        id: number;
        name: string;
        logo: string;
    } | null;
    player1_score: number;
    player2_score: number;
    winner: {
        id: number;
        display_name: string;
        user_name: string;
    } | null;
    winning_team: {
        id: number;
        name: string;
    } | null;
    status: string;
    scheduled_time: string | null;
    started_at: string | null;
    completed_at: string | null;
}

interface Props {
    matches: Match[];
    user: any;
}

const statusColors = {
    'scheduled': 'bg-blue-100 text-blue-800',
    'in_progress': 'bg-yellow-100 text-yellow-800',
    'completed': 'bg-green-100 text-green-800',
    'cancelled': 'bg-red-100 text-red-800',
};

const statusIcons = {
    'scheduled': Clock,
    'in_progress': Gamepad,
    'completed': Trophy,
    'cancelled': Clock,
};

export default function Matches({ matches, user }: Props) {
    const upcomingMatches = matches.filter(match => ['scheduled', 'in_progress'].includes(match.status));
    const completedMatches = matches.filter(match => match.status === 'completed');
    const cancelledMatches = matches.filter(match => match.status === 'cancelled');

    const formatDateTime = (dateString: string | null) => {
        if (!dateString) return 'TBD';
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    };

    const MatchCard = ({ match }: { match: Match }) => {
        const StatusIcon = statusIcons[match.status as keyof typeof statusIcons] || Clock;
        
        return (
            <Card key={match.id} className="hover:shadow-lg transition-shadow">
                <CardHeader className="pb-3">
                    <div className="flex items-center justify-between">
                        <div className="flex items-center gap-2">
                            <StatusIcon className="h-5 w-5 text-gray-600" />
                            <div>
                                <CardTitle className="text-lg">
                                    Round {match.round}, Match #{match.match_number}
                                </CardTitle>
                                <CardDescription>
                                    {match.tournament.name}
                                    {match.bracket && ` - ${match.bracket.name}`}
                                </CardDescription>
                            </div>
                        </div>
                        <Badge 
                            className={statusColors[match.status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}
                        >
                            {match.status.replace('_', ' ')}
                        </Badge>
                    </div>
                </CardHeader>
                <CardContent className="space-y-4">
                    {/* Match Participants */}
                    <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                        {match.team1 && match.team2 ? (
                            // Team vs Team
                            <>
                                <div className="flex items-center gap-3">
                                    {match.team1.logo && (
                                        <img 
                                            src={match.team1.logo} 
                                            alt={match.team1.name}
                                            className="w-8 h-8 rounded object-cover"
                                        />
                                    )}
                                    <div>
                                        <p className="font-semibold">{match.team1.name}</p>
                                    </div>
                                </div>
                                <div className="text-center">
                                    {match.status === 'completed' ? (
                                        <div className="text-2xl font-bold">
                                            {match.player1_score} - {match.player2_score}
                                        </div>
                                    ) : (
                                        <div className="text-xl font-semibold text-gray-400">vs</div>
                                    )}
                                </div>
                                <div className="flex items-center gap-3">
                                    <div className="text-right">
                                        <p className="font-semibold">{match.team2.name}</p>
                                    </div>
                                    {match.team2.logo && (
                                        <img 
                                            src={match.team2.logo} 
                                            alt={match.team2.name}
                                            className="w-8 h-8 rounded object-cover"
                                        />
                                    )}
                                </div>
                            </>
                        ) : (
                            // Player vs Player
                            <>
                                <div className="flex items-center gap-3">
                                    <User className="h-8 w-8 text-gray-400" />
                                    <div>
                                        <p className="font-semibold">
                                            {match.player1?.display_name || 'TBD'}
                                        </p>
                                        <p className="text-sm text-gray-600">
                                            {match.player1?.user_name}
                                        </p>
                                    </div>
                                </div>
                                <div className="text-center">
                                    {match.status === 'completed' ? (
                                        <div className="text-2xl font-bold">
                                            {match.player1_score} - {match.player2_score}
                                        </div>
                                    ) : (
                                        <div className="text-xl font-semibold text-gray-400">vs</div>
                                    )}
                                </div>
                                <div className="flex items-center gap-3">
                                    <div className="text-right">
                                        <p className="font-semibold">
                                            {match.player2?.display_name || 'TBD'}
                                        </p>
                                        <p className="text-sm text-gray-600">
                                            {match.player2?.user_name}
                                        </p>
                                    </div>
                                    <User className="h-8 w-8 text-gray-400" />
                                </div>
                            </>
                        )}
                    </div>

                    {/* Winner Display */}
                    {match.status === 'completed' && (match.winner || match.winning_team) && (
                        <div className="flex items-center justify-center gap-2 p-2 bg-green-50 rounded-lg">
                            <Trophy className="h-4 w-4 text-green-600" />
                            <span className="text-green-800 font-medium">
                                Winner: {match.winning_team?.name || match.winner?.display_name}
                            </span>
                        </div>
                    )}

                    {/* Match Timing */}
                    <div className="flex items-center gap-2 text-sm text-gray-600">
                        <Calendar className="h-4 w-4" />
                        <span>
                            {match.status === 'scheduled' && match.scheduled_time && (
                                <>Scheduled: {formatDateTime(match.scheduled_time)}</>
                            )}
                            {match.status === 'in_progress' && match.started_at && (
                                <>Started: {formatDateTime(match.started_at)}</>
                            )}
                            {match.status === 'completed' && match.completed_at && (
                                <>Completed: {formatDateTime(match.completed_at)}</>
                            )}
                            {match.status === 'cancelled' && (
                                <>Cancelled</>
                            )}
                        </span>
                    </div>
                </CardContent>
            </Card>
        );
    };

    return (
        <AppLayout>
            <Head title="My Matches" />
            
            <div className="py-12">
                <div className="">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-6">
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
                                        <Gamepad className="h-8 w-8 text-blue-600" />
                                        My Matches
                                    </h1>
                                    <p className="text-gray-600 mt-2">
                                        View your match schedule and results
                                    </p>
                                </div>
                                <div className="flex gap-4">
                                    <Badge variant="secondary">
                                        {upcomingMatches.length} Upcoming
                                    </Badge>
                                    <Badge variant="outline">
                                        {completedMatches.length} Completed
                                    </Badge>
                                </div>
                            </div>

                            {matches.length === 0 ? (
                                <Card>
                                    <CardContent className="p-12 text-center">
                                        <Gamepad className="h-16 w-16 text-gray-300 mx-auto mb-4" />
                                        <h3 className="text-xl font-semibold text-gray-600 mb-2">
                                            No Matches Yet
                                        </h3>
                                        <p className="text-gray-500">
                                            You don't have any matches scheduled yet.
                                        </p>
                                    </CardContent>
                                </Card>
                            ) : (
                                <div className="space-y-8">
                                    {/* Upcoming Matches */}
                                    {upcomingMatches.length > 0 && (
                                        <div>
                                            <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                                <Clock className="h-6 w-6 text-blue-600" />
                                                Upcoming Matches ({upcomingMatches.length})
                                            </h2>
                                            <div className="grid gap-6 md:grid-cols-2">
                                                {upcomingMatches.map((match) => (
                                                    <MatchCard key={match.id} match={match} />
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {/* Completed Matches */}
                                    {completedMatches.length > 0 && (
                                        <div>
                                            <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                                <Trophy className="h-6 w-6 text-green-600" />
                                                Completed Matches ({completedMatches.length})
                                            </h2>
                                            <div className="grid gap-6 md:grid-cols-2">
                                                {completedMatches.map((match) => (
                                                    <MatchCard key={match.id} match={match} />
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {/* Cancelled Matches */}
                                    {cancelledMatches.length > 0 && (
                                        <div>
                                            <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                                                <Clock className="h-6 w-6 text-red-600" />
                                                Cancelled Matches ({cancelledMatches.length})
                                            </h2>
                                            <div className="grid gap-6 md:grid-cols-2">
                                                {cancelledMatches.map((match) => (
                                                    <MatchCard key={match.id} match={match} />
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}