import { useState, useCallback } from 'react';
import { toast } from 'sonner';

interface CertificateData {
    player_name: string;
    team_name?: string;
    position?: string;
    jersey_number?: string;
    tournament_matches_played?: number;
    tournament_win_rate?: number;
    global_rating?: number;
    digital_id: string;
    issued_date: string;
    tournament_name: string;
    description?: string;
    verification_status?: string;
}

interface UseCertificatePreviewReturn {
    isOpen: boolean;
    certificateData: CertificateData | null;
    playerId: number | null;
    openPreview: (playerId: number, data: CertificateData) => void;
    closePreview: () => void;
    fetchCertificateData: (playerId: number) => Promise<CertificateData | null>;
}

export const useCertificatePreview = (): UseCertificatePreviewReturn => {
    const [isOpen, setIsOpen] = useState(false);
    const [certificateData, setCertificateData] = useState<CertificateData | null>(null);
    const [playerId, setPlayerId] = useState<number | null>(null);

    const fetchCertificateData = useCallback(async (playerId: number): Promise<CertificateData | null> => {
        try {
            // Fetch player data from the API
            const response = await fetch(`/api/admin/players/${playerId}`);
            
            if (!response.ok) {
                throw new Error('Failed to fetch player data');
            }
            
            const playerData = await response.json();
            
            // Transform player data to certificate format
            const certificateId = `CERT-${String(playerId).padStart(6, '0')}-${Math.random().toString(36).substr(2, 8).toUpperCase()}`;
            
            return {
                player_name: playerData.display_name || playerData.user?.name || 'Player Name',
                team_name: playerData.team_name || '',
                position: playerData.position || 'Participant',
                jersey_number: playerData.jersey_number || undefined,
                tournament_matches_played: playerData.tournament_matches_played || 0,
                tournament_win_rate: playerData.tournament_win_rate || 0,
                global_rating: playerData.global_rating || 0,
                digital_id: certificateId,
                issued_date: new Date().toLocaleDateString('en-US', { 
                    year: 'numeric', 
                    month: 'long', 
                    day: 'numeric' 
                }),
                tournament_name: playerData.tournament?.name || 'The Milimus Tournament Cup',
                description: playerData.description || undefined,
                verification_status: playerData.verification_status || 'verified',
            };
        } catch (error) {
            console.error('Error fetching certificate data:', error);
            toast.error('Failed to load certificate data');
            return null;
        }
    }, []);

    const openPreview = useCallback(async (playerId: number, data?: CertificateData) => {
        setPlayerId(playerId);
        
        if (data) {
            setCertificateData(data);
            setIsOpen(true);
        } else {
            // Fetch data if not provided
            const fetchedData = await fetchCertificateData(playerId);
            if (fetchedData) {
                setCertificateData(fetchedData);
                setIsOpen(true);
            }
        }
    }, [fetchCertificateData]);

    const closePreview = useCallback(() => {
        setIsOpen(false);
        setCertificateData(null);
        setPlayerId(null);
    }, []);

    return {
        isOpen,
        certificateData,
        playerId,
        openPreview,
        closePreview,
        fetchCertificateData,
    };
};
