import { usePage } from '@inertiajs/react';

interface User {
    id: number;
    name: string;
    email: string;
    roles: string[];
    permissions: string[];
}

interface PageProps {
    user?: User;
    userPermissions?: string[];
    userRoles?: string[];
}

export function usePermissions() {
    const { props } = usePage<PageProps>();
    const user = props.auth.user;
    const permissions = props.auth.userPermissions || [];
    const roles = props.auth.userRoles || [];

    const hasPermission = (permission: string | string[]): boolean => {
        if (!user) return false;
        
        const permissionArray = Array.isArray(permission) ? permission : [permission];
        
        // Super admin has all permissions
        if (roles.includes('super_admin')) return true;
        
        // Filter out null values from permissions array
        const validPermissions = permissions.filter(perm => perm !== null && perm !== undefined);        
        // Check if user has any of the required permissions
        return permissionArray.some(perm => validPermissions.includes(perm));
    };

    const hasRole = (role: string | string[]): boolean => {
        if (!user) return false;
        
        const roleArray = Array.isArray(role) ? role : [role];
        return roleArray.some(r => roles.includes(r));
    };

    const hasAnyRole = (role: string[]): boolean => {
        return hasRole(role);
    };

    const hasAllRoles = (role: string[]): boolean => {
        if (!user) return false;
        return role.every(r => roles.includes(r));
    };

    const canAccess = (permission: string | string[]): boolean => {
        return hasPermission(permission);
    };

    const isAdmin = (): boolean => {
        return hasRole(['admin', 'super_admin']);
    };

    const isOrganizer = (): boolean => {
        return hasRole(['organizer', 'admin', 'super_admin']);
    };

    const isReferee = (): boolean => {
        return hasRole(['referee', 'admin', 'super_admin']);
    };

    const isPlayer = (): boolean => {
        return hasRole(['player', 'admin', 'super_admin']);
    };

    const isSpectator = (): boolean => {
        return hasRole(['spectator', 'admin', 'super_admin']);
    };

    const isTeamManager = (): boolean => {
        return hasRole(['team_manager', 'admin', 'super_admin']);
    };

    return {
        user,
        permissions,
        roles,
        hasPermission,
        hasRole,
        hasAnyRole,
        hasAllRoles,
        canAccess,
        isAdmin,
        isOrganizer,
        isReferee,
        isPlayer,
        isSpectator,
        isTeamManager,
    };
} 