import { useCallback } from 'react';
import { router } from '@inertiajs/react';
import { toast } from 'sonner';
import { usePermissions } from '@/hooks/use-permissions';

interface ErrorHandlerOptions {
    showToast?: boolean;
    redirectOnError?: boolean;
    redirectTo?: string;
    onError?: (error: any) => void;
}

export function useErrorHandler(options: ErrorHandlerOptions = {}) {
    const {
        showToast = true,
        redirectOnError = false,
        redirectTo = '/admin/dashboard',
        onError
    } = options;

    const { isAdmin, isOrganizer, isReferee, isPlayer } = usePermissions();

    const handlePermissionError = useCallback((error: any) => {
        const status = error?.response?.status || error?.status;
        const message = error?.response?.data?.message || error?.message || 'An error occurred';

        // Get role-specific error message
        const getRoleSpecificMessage = () => {
            if (isAdmin()) {
                return 'As an administrator, you may need to check system permissions or contact the system administrator.';
            } else if (isOrganizer()) {
                return 'As a tournament organizer, you may need to verify tournament settings or contact an administrator.';
            } else if (isReferee()) {
                return 'As a referee, you may need to check your match assignments or contact the tournament organizer.';
            } else if (isPlayer()) {
                return 'As a player, you may need to check your tournament participation or contact the tournament organizer.';
            }
            return 'You do not have permission to perform this action.';
        };

        switch (status) {
            case 403:
                if (showToast) {
                    toast.error('Access Denied', {
                        description: getRoleSpecificMessage(),
                        duration: 5000,
                    });
                }
                
                if (redirectOnError) {
                    setTimeout(() => {
                        router.visit(redirectTo);
                    }, 2000);
                }
                break;

            case 401:
                if (showToast) {
                    toast.error('Authentication Required', {
                        description: 'Please log in to continue.',
                        duration: 5000,
                    });
                }
                
                router.visit('/login');
                break;

            case 419:
                if (showToast) {
                    toast.error('Session Expired', {
                        description: 'Your session has expired. Please refresh the page.',
                        duration: 5000,
                        action: {
                            label: 'Refresh',
                            onClick: () => window.location.reload()
                        }
                    });
                }
                break;

            case 429:
                if (showToast) {
                    toast.error('Too Many Requests', {
                        description: 'Please wait a moment before trying again.',
                        duration: 5000,
                    });
                }
                break;

            case 500:
                if (showToast) {
                    toast.error('Server Error', {
                        description: 'Something went wrong on our end. Please try again later.',
                        duration: 5000,
                    });
                }
                break;

            default:
                if (showToast) {
                    toast.error('Error', {
                        description: message,
                        duration: 5000,
                    });
                }
                break;
        }

        // Call custom error handler if provided
        if (onError) {
            onError(error);
        }
    }, [showToast, redirectOnError, redirectTo, onError]);

    const handleFormError = useCallback((errors: Record<string, string[]>) => {
        const firstError = Object.values(errors)[0]?.[0];
        if (firstError && showToast) {
            toast.error('Validation Error', {
                description: firstError,
                duration: 4000,
            });
        }
    }, [showToast]);

    const handleSuccess = useCallback((message: string) => {
        if (showToast) {
            toast.success('Success', {
                description: message,
                duration: 3000,
            });
        }
    }, [showToast]);

    const handleWarning = useCallback((message: string) => {
        if (showToast) {
            toast.warning('Warning', {
                description: message,
                duration: 4000,
            });
        }
    }, [showToast]);

    const handleInfo = useCallback((message: string) => {
        if (showToast) {
            toast.info('Info', {
                description: message,
                duration: 3000,
            });
        }
    }, [showToast]);

    return {
        handlePermissionError,
        handleFormError,
        handleSuccess,
        handleWarning,
        handleInfo,
    };
}

// Global error handler for Inertia requests
export function setupGlobalErrorHandler() {
    // Handle Inertia errors globally
    router.on('error', (event) => {
        const { detail } = event as any;
        const status = detail?.response?.status;
        
        switch (status) {
            case 403:
                toast.error('Access Denied', {
                    description: 'You do not have permission to access this resource.',
                    duration: 5000,
                });
                break;
                
            case 401:
                toast.error('Authentication Required', {
                    description: 'Please log in to continue.',
                    duration: 5000,
                });
                router.visit('/login');
                break;
                
            case 419:
                toast.error('Session Expired', {
                    description: 'Your session has expired. Please refresh the page.',
                    duration: 5000,
                    action: {
                        label: 'Refresh',
                        onClick: () => window.location.reload()
                    }
                });
                break;
                
            case 500:
                toast.error('Server Error', {
                    description: 'Something went wrong. Please try again later.',
                    duration: 5000,
                });
                break;
        }
    });

    // Handle successful operations
    router.on('success', (event) => {
        const { detail } = event as any;
        const message = detail?.props?.flash?.success;
        
        if (message) {
            toast.success('Success', {
                description: message,
                duration: 3000,
            });
        }
    });
}

// Permission-specific error messages
export const PERMISSION_ERROR_MESSAGES = {
    tournaments: {
        view: 'You need permission to view tournaments.',
        create: 'You need permission to create tournaments.',
        edit: 'You need permission to edit tournaments.',
        delete: 'You need permission to delete tournaments.',
    },
    players: {
        view: 'You need permission to view players.',
        create: 'You need permission to add players.',
        edit: 'You need permission to edit players.',
        delete: 'You need permission to delete players.',
    },
    teams: {
        view: 'You need permission to view teams.',
        create: 'You need permission to create teams.',
        edit: 'You need permission to edit teams.',
        delete: 'You need permission to delete teams.',
    },
    matches: {
        view: 'You need permission to view matches.',
        create: 'You need permission to schedule matches.',
        edit: 'You need permission to edit matches.',
        delete: 'You need permission to delete matches.',
        start: 'You need permission to start matches.',
        complete: 'You need permission to complete matches.',
        verify: 'You need permission to verify matches.',
    },
    users: {
        view: 'You need permission to view users.',
        create: 'You need permission to create users.',
        edit: 'You need permission to edit users.',
        delete: 'You need permission to delete users.',
        impersonate: 'You need permission to impersonate users.',
    },
    roles: {
        view: 'You need permission to view roles.',
        create: 'You need permission to create roles.',
        edit: 'You need permission to edit roles.',
        delete: 'You need permission to delete roles.',
        assign: 'You need permission to assign roles.',
    },
    permissions: {
        view: 'You need permission to view permissions.',
        create: 'You need permission to create permissions.',
        edit: 'You need permission to edit permissions.',
        delete: 'You need permission to delete permissions.',
        assign: 'You need permission to assign permissions.',
    },
    settings: {
        view: 'You need permission to view settings.',
        edit: 'You need permission to edit settings.',
        system: 'You need system-level permissions for this action.',
    },
    media: {
        view: 'You need permission to view media.',
        upload: 'You need permission to upload media.',
        edit: 'You need permission to edit media.',
        delete: 'You need permission to delete media.',
    },
} as const;