import React, { useState } from "react";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "../components/ui/tabs";
import { Input } from "../components/ui/input";
import { Button } from "../components/ui/button";
import { useNavigate } from "react-router-dom";
import Header from "../components/layout/Header";
import Footer from "../components/sections/Footer";

const steps = [
  { value: "team", label: "Team" },
  { value: "players", label: "Team Players" },
  { value: "officials", label: "Team Officials" },
  { value: "payment", label: "Payment" },
];

const footballPositions = [
  "Goalkeeper",
  "Defender",
  "Midfielder",
  "Forward",
  "Coach",
  "Assistant Coach",
  "Physio",
  "Manager"
];

const officialRoles = [
  "Coach",
  "Assistant Coach",
  "Manager",
  "Physio",
  "Doctor",
  "Other"
];

const paymentMethods = [
  "Bank Transfer",
  "Card",
  "Mobile Money"
];

const defaultPlayer = { name: "", email: "", phone: "", position: "" };
const defaultOfficial = { name: "", email: "", phone: "", role: "" };

const Register = () => {
  const [currentStep, setCurrentStep] = useState(steps[0].value);
  const [team, setTeam] = useState({ name: "", email: "", phone: "", location: "", coach: "", year: "" });
  const [players, setPlayers] = useState(Array.from({ length: 11 }, () => ({ ...defaultPlayer })));
  const [officials, setOfficials] = useState([{ ...defaultOfficial }]);
  const [payment, setPayment] = useState({ method: "", reference: "", amount: "" });
  const navigate = useNavigate();

  const handlePlayerChange = (idx, field, value) => {
    setPlayers((prev) => prev.map((p, i) => i === idx ? { ...p, [field]: value } : p));
  };
  const addPlayer = () => setPlayers((prev) => [...prev, { ...defaultPlayer }]);
  const removePlayer = (idx) => setPlayers((prev) => prev.length > 1 ? prev.filter((_, i) => i !== idx) : prev);

  const handleOfficialChange = (idx, field, value) => {
    setOfficials((prev) => prev.map((o, i) => i === idx ? { ...o, [field]: value } : o));
  };
  const addOfficial = () => setOfficials((prev) => [...prev, { ...defaultOfficial }]);
  const removeOfficial = (idx) => setOfficials((prev) => prev.length > 1 ? prev.filter((_, i) => i !== idx) : prev);

  return (
    <>
      <Header />
      <div className="flex flex-col items-center justify-center min-h-[60vh] bg-gray-50">
        <div className="w-full max-w-2xl p-8 bg-white rounded shadow mt-8 mb-8">
          <h2 className="mb-6 text-2xl font-bold text-center">Register</h2>
          <Tabs value={currentStep} onValueChange={setCurrentStep}>
            <TabsList className="mb-6 w-full flex justify-between">
              {steps.map((step) => (
                <TabsTrigger
                  key={step.value}
                  value={step.value}
                  className="flex-1 text-center cursor-pointer"
                  style={{ color: "#222" }}
                >
                  {step.label}
                </TabsTrigger>
              ))}
            </TabsList>
            <TabsContent value="team">
              <form className="space-y-4">
                <Input placeholder="Team Name" value={team.name} onChange={e => setTeam({ ...team, name: e.target.value })} required />
                <Input placeholder="Team Email" type="email" value={team.email} onChange={e => setTeam({ ...team, email: e.target.value })} required />
                <Input placeholder="Team Phone" value={team.phone} onChange={e => setTeam({ ...team, phone: e.target.value })} required />
                <Input placeholder="Location" value={team.location} onChange={e => setTeam({ ...team, location: e.target.value })} required />
                <Input placeholder="Coach Name" value={team.coach} onChange={e => setTeam({ ...team, coach: e.target.value })} required />
                <Input placeholder="Year Founded" type="number" value={team.year} onChange={e => setTeam({ ...team, year: e.target.value })} required />
                <Button type="button" className="w-full" onClick={() => setCurrentStep("players")}>Next</Button>
              </form>
            </TabsContent>
            <TabsContent value="players">
              <form className="space-y-4">
                {players.map((player, idx) => (
                  <div key={idx} className="border p-4 rounded mb-2 relative">
                    <div className="flex gap-2 mb-2">
                      <Input placeholder="Name" value={player.name} onChange={e => handlePlayerChange(idx, "name", e.target.value)} required />
                      <Input placeholder="Email" type="email" value={player.email} onChange={e => handlePlayerChange(idx, "email", e.target.value)} required />
                    </div>
                    <div className="flex gap-2 mb-2">
                      <Input placeholder="Phone" value={player.phone} onChange={e => handlePlayerChange(idx, "phone", e.target.value)} />
                      <select
                        className="border rounded px-2 py-1 w-full"
                        value={player.position}
                        onChange={e => handlePlayerChange(idx, "position", e.target.value)}
                        required
                      >
                        <option value="">Select Position</option>
                        {footballPositions.map((pos) => (
                          <option key={pos} value={pos}>{pos}</option>
                        ))}
                      </select>
                    </div>
                    <Button type="button" variant="destructive" size="sm" className="absolute top-2 right-2" onClick={() => removePlayer(idx)} disabled={players.length === 1}>Remove</Button>
                  </div>
                ))}
                <Button type="button" variant="outline" onClick={addPlayer} className="w-full">Add Player</Button>
                <Button type="button" className="w-full" onClick={() => setCurrentStep("officials")}>Next</Button>
              </form>
            </TabsContent>
            <TabsContent value="officials">
              <form className="space-y-4">
                {officials.map((official, idx) => (
                  <div key={idx} className="border p-4 rounded mb-2 relative">
                    <div className="flex gap-2 mb-2">
                      <Input placeholder="Name" value={official.name} onChange={e => handleOfficialChange(idx, "name", e.target.value)} required />
                      <Input placeholder="Email" type="email" value={official.email} onChange={e => handleOfficialChange(idx, "email", e.target.value)} required />
                    </div>
                    <div className="flex gap-2 mb-2">
                      <Input placeholder="Phone" value={official.phone} onChange={e => handleOfficialChange(idx, "phone", e.target.value)} />
                      <select
                        className="border rounded px-2 py-1 w-full"
                        value={official.role}
                        onChange={e => handleOfficialChange(idx, "role", e.target.value)}
                        required
                      >
                        <option value="">Select Role</option>
                        {officialRoles.map((role) => (
                          <option key={role} value={role}>{role}</option>
                        ))}
                      </select>
                    </div>
                    <Button type="button" variant="destructive" size="sm" className="absolute top-2 right-2" onClick={() => removeOfficial(idx)} disabled={officials.length === 1}>Remove</Button>
                  </div>
                ))}
                <Button type="button" variant="outline" onClick={addOfficial} className="w-full">Add Official</Button>
                <Button type="button" className="w-full" onClick={() => setCurrentStep("payment")}>Next</Button>
              </form>
            </TabsContent>
            <TabsContent value="payment">
              <form className="space-y-4">
                <select
                  className="border rounded px-2 py-1 w-full"
                  value={payment.method}
                  onChange={e => setPayment({ ...payment, method: e.target.value })}
                  required
                >
                  <option value="">Select Payment Method</option>
                  {paymentMethods.map((method) => (
                    <option key={method} value={method}>{method}</option>
                  ))}
                </select>
                <Input placeholder="Transaction/Reference Number" value={payment.reference} onChange={e => setPayment({ ...payment, reference: e.target.value })} required />
                <Input placeholder="Amount" type="number" value={payment.amount} onChange={e => setPayment({ ...payment, amount: e.target.value })} required />
                <Button type="button" className="w-full" onClick={() => alert("Registration complete! (Prototype)")}>Finish</Button>
              </form>
            </TabsContent>
          </Tabs>
          <div className="mt-4 text-center">
            <span>Already have an account? </span>
            <button className="text-blue-600 underline" onClick={() => navigate("/login")}>Login</button>
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
};

export default Register; 