import Header from "../components/layout/Header";
import Footer from "../components/sections/Footer";
import { Camera, Heart, Share2, Download, ChevronLeft, ChevronRight, Grid, List } from "lucide-react";
import React, { useState, useEffect } from "react";
import { galleryHeroSlides, galleryImages, editions, categories } from "@/lib/data";  

const Gallery = () => {
  const [selectedEdition, setSelectedEdition] = useState("all");
  const [selectedCategory, setSelectedCategory] = useState("all");
  const [selectedImage, setSelectedImage] = useState(null);
  const [currentSlide, setCurrentSlide] = useState(0);
  const [layoutMode, setLayoutMode] = useState("row"); // "row" or "column"

  // Auto-play functionality
  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % galleryHeroSlides.length);
    }, 5000); // Change slide every 5 seconds

    return () => clearInterval(interval);
  }, []);

  const nextSlide = () => {
    setCurrentSlide((prev) => (prev + 1) % galleryHeroSlides.length);
  };

  const prevSlide = () => {
    setCurrentSlide((prev) => (prev - 1 + galleryHeroSlides.length) % galleryHeroSlides.length);
  };

  const goToSlide = (index: number) => {
    setCurrentSlide(index);
  };

  const filteredImages = galleryImages.filter(img => {
    const matchesEdition = selectedEdition === "all" || img.edition === selectedEdition;
    const matchesCategory = selectedCategory === "all" || img.category === selectedCategory;
    return matchesEdition && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Slider Section */}
      <section className="relative h-[380px] md:h-[480px] overflow-hidden">
        {/* Slides */}
        <div className="relative w-full h-full">
          {galleryHeroSlides.map((slide, index) => {
            const IconComponent = Camera;
            return (
              <div
                key={slide.id}
                className={`absolute inset-0 transition-transform duration-1000 ease-in-out ${
                  index === currentSlide ? 'translate-x-0' : 
                  index < currentSlide ? '-translate-x-full' : 'translate-x-full'
                }`}
              >
                <div className="relative flex items-center justify-center h-full bg-gradient-to-br from-primary/90 to-secondary/90 overflow-hidden">
                  <img
                    src={slide.image}
                    alt={`Slide ${slide.id}`}
                    className="absolute inset-0 w-full h-full object-cover opacity-20 pointer-events-none select-none scale-110"
                  />
                  <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
                  <div className="relative z-10 text-center text-white flex flex-col items-center justify-center w-full animate-fade-in-up">
                    <IconComponent className="mx-auto mb-4 h-16 w-16 text-yellow-400 animate-bounce-gentle" />
                    <h1 className="text-5xl md:text-6xl font-extrabold drop-shadow mb-4 tracking-tight">{slide.title}</h1>
                    <p className="text-xl md:text-2xl max-w-2xl mx-auto drop-shadow font-medium">
                      {slide.subtitle}
                    </p>
                  </div>
                </div>
              </div>
            );
          })}
        </div>

        {/* Navigation Arrows */}
        <button
          onClick={prevSlide}
          className="absolute left-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
          aria-label="Previous slide"
        >
          <ChevronLeft className="h-6 w-6" />
        </button>
        <button
          onClick={nextSlide}
          className="absolute right-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
          aria-label="Next slide"
        >
          <ChevronRight className="h-6 w-6" />
        </button>

        {/* Slide Indicators */}
        <div className="absolute bottom-6 left-1/2 -translate-x-1/2 z-20 flex gap-2">
          {galleryHeroSlides.map((_, index) => (
            <button
              key={index}
              onClick={() => goToSlide(index)}
              className={`w-3 h-3 rounded-full transition-all duration-300 ${
                index === currentSlide 
                  ? 'bg-white scale-125' 
                  : 'bg-white/50 hover:bg-white/75'
              }`}
              aria-label={`Go to slide ${index + 1}`}
            />
          ))}
        </div>
      </section>

      {/* Edition Filter */}
      <section className="max-w-6xl mx-auto py-6 px-4">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Filter by Edition</h3>
        </div>
        <div className="flex flex-wrap justify-center gap-2 md:gap-4">
          {editions.map((edition) => (
            <button
              key={edition.id}
              onClick={() => setSelectedEdition(edition.id)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
                selectedEdition === edition.id
                  ? "bg-secondary text-white shadow-lg"
                  : "bg-white/80 text-gray-700 hover:bg-secondary/10 border border-gray-200"
              }`}
            >
              {edition.name}
            </button>
          ))}
        </div>
      </section>

      {/* Category Filter */}
      <section className="max-w-6xl mx-auto py-6 px-4 mb-10">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Filter by Category</h3>
        </div>
        <div className="flex flex-wrap justify-center gap-2 md:gap-4">
          {categories.map((category) => (
            <button
              key={category.id}
              onClick={() => setSelectedCategory(category.id)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
                selectedCategory === category.id
                  ? "bg-primary text-white shadow-lg"
                  : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
              }`}
            >
              {category.name}
            </button>
          ))}
        </div>
      </section>

      {/* Layout Toggle */}
      {/* <section className="max-w-6xl mx-auto py-4 px-4">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Layout View</h3>
        </div>
        <div className="flex justify-center gap-4">
          <button
            onClick={() => setLayoutMode("row")}
            className={`flex items-center gap-2 px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
              layoutMode === "row"
                ? "bg-primary text-white shadow-lg"
                : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
            }`}
          >
            <Grid className="h-4 w-4" />
            Row Masonry
          </button>
          <button
            onClick={() => setLayoutMode("column")}
            className={`flex items-center gap-2 px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
              layoutMode === "column"
                ? "bg-primary text-white shadow-lg"
                : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
            }`}
          >
            <List className="h-4 w-4" />
            Column Masonry
          </button>
        </div>
      </section> */}

      {/* Masonry Gallery */}
      <section className="max-w-7xl mx-auto px-4 pb-16">
        {layoutMode === "row" ? (
          // Row-based Masonry (CSS Grid)
          <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-4 auto-rows-[200px]">
            {filteredImages.map((image) => (
              <div
                key={image.id}
                className={`relative group rounded-xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 cursor-pointer ${
                  image.height === 48 ? " row-span-1" :
                  image.height === 56 ? "row-span-1" :
                  image.height === 64 ? "row-span-2" :
                  image.height === 72 ? "row-span-2" :
                  image.height === 80 ? "row-span-2" : "row-span-1"
                }`}
                onClick={() => setSelectedImage(image)}
              >
                <img
                  src={image.src}
                  alt={image.alt}
                  className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                  <div className="absolute bottom-0 left-0 right-0 p-4 text-white">
                    <div className="flex items-center justify-between mb-2">
                      <h3 className="font-semibold text-sm">{image.alt}</h3>
                      <span className="text-xs bg-white/20 px-2 py-1 rounded-full">{image.edition} Edition</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Heart className="h-4 w-4" />
                      </button>
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Share2 className="h-4 w-4" />
                      </button>
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Download className="h-4 w-4" />
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        ) : (
          // Column-based Masonry (CSS Columns)
          <div className="columns-1 sm:columns-2 md:columns-3 lg:columns-4 gap-4 space-y-4">
            {filteredImages.map((image) => (
              <div
                key={image.id}
                className={`relative group break-inside-avoid ${image.height} mb-4 rounded-xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 cursor-pointer`}
                onClick={() => setSelectedImage(image)}
              >
                <img
                  src={image.src}
                  alt={image.alt}
                  className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                  <div className="absolute bottom-0 left-0 right-0 p-4 text-white">
                    <div className="flex items-center justify-between mb-2">
                      <h3 className="font-semibold text-sm">{image.alt}</h3>
                      <span className="text-xs bg-white/20 px-2 py-1 rounded-full">{image.edition} Edition</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Heart className="h-4 w-4" />
                      </button>
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Share2 className="h-4 w-4" />
                      </button>
                      <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                        <Download className="h-4 w-4" />
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
        
        {filteredImages.length === 0 && (
          <div className="text-center py-16">
            <Camera className="mx-auto h-16 w-16 text-gray-400 mb-4" />
            <h3 className="text-xl font-semibold text-gray-600 mb-2">No photos found</h3>
            <p className="text-gray-500">Try selecting different filters.</p>
          </div>
        )}
      </section>

      {/* Image Modal */}
      {selectedImage && (
        <div className="fixed inset-0 bg-black/90 z-50 flex items-center justify-center p-4">
          <div className="relative max-w-4xl max-h-[90vh]">
            <img
              src={selectedImage.src}
              alt={selectedImage.alt}
              className="w-full h-full object-contain rounded-lg"
            />
            <div className="absolute top-4 right-4 flex gap-2">
              <button
                onClick={() => setSelectedImage(null)}
                className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white"
              >
                ✕
              </button>
            </div>
            <div className="absolute bottom-4 left-4 right-4 text-white">
              <div className="flex items-center justify-between mb-2">
                <h3 className="font-semibold text-lg">{selectedImage.alt}</h3>
                <span className="text-sm bg-white/20 px-3 py-1 rounded-full">{selectedImage.edition} Edition</span>
              </div>
              <div className="flex items-center gap-2">
                <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                  <Heart className="h-4 w-4" />
                </button>
                <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                  <Share2 className="h-4 w-4" />
                </button>
                <button className="p-2 rounded-full bg-white/20 hover:bg-white/30 transition-colors">
                  <Download className="h-4 w-4" />
                </button>
              </div>
            </div>
          </div>
        </div>
      )}

      <Footer />
    </div>
  );
};

export default Gallery; 