import Header from "../components/layout/Header";
import Footer from "../components/sections/Footer";
import { Calendar, Search, Trophy, CheckCircle, Clock, LayoutGrid, List } from "lucide-react";
import React, { useState } from "react";
import { fixtures } from "@/lib/data";


const statusBadge = (status) => {
  if (status === "live") return <span className="px-3 py-1 rounded-full bg-red-500 text-white text-xs font-bold animate-pulse">Live</span>;
  if (status === "completed") return <span className="px-3 py-1 rounded-full bg-green-600 text-white text-xs font-bold flex items-center gap-1"><CheckCircle className="h-4 w-4" />Completed</span>;
  return <span className="px-3 py-1 rounded-full bg-gray-300 text-gray-700 text-xs font-bold flex items-center gap-1"><Clock className="h-4 w-4" />Upcoming</span>;
};

const Fixtures = () => {
  const [search, setSearch] = useState("");
  const [view, setView] = useState("grid"); // 'grid' or 'list'

  const filteredFixtures = fixtures.filter(f => {
    const q = search.toLowerCase();
    return (
      f.teamA.name.toLowerCase().includes(q) ||
      f.teamB.name.toLowerCase().includes(q) ||
      f.date.includes(q)
    );
  });

  return (
    <div className="min-h-screen bg-background">
      <Header />
      {/* Hero Section */}
      <section className="relative flex items-center justify-center h-[260px] md:h-[320px] bg-gradient-to-br from-primary/80 to-secondary/80 overflow-hidden">
        <img
          src="/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png"
          alt="Fixtures Hero"
          className="absolute inset-0 w-full h-full object-cover opacity-10 pointer-events-none select-none"
        />
        <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
        <div className="relative z-10 text-center text-white flex flex-col items-center justify-center w-full">
          <Calendar className="mx-auto mb-4 h-12 w-12 text-yellow-400 animate-bounce-gentle" />
          <h1 className="text-4xl md:text-5xl font-extrabold drop-shadow mb-2">Fixtures & Results</h1>
          <p className="text-lg md:text-xl max-w-2xl mx-auto drop-shadow">
            Stay up to date with the latest fixtures, results, and live action.
          </p>
        </div>
      </section>
      {/* Filter/Search Bar and View Toggle */}
      <section className="max-w-4xl mx-auto py-8 px-4">
        <form className="flex flex-col md:flex-row gap-4 items-center justify-between bg-white rounded-xl shadow p-4 mb-8">
          <div className="flex items-center w-full md:w-auto gap-2">
            <Search className="h-5 w-5 text-gray-400" />
            <input
              type="text"
              placeholder="Search team or date..."
              className="w-full md:w-64 px-3 py-2 rounded border border-gray-200 focus:outline-none focus:ring-2 focus:ring-primary"
              value={search}
              onChange={e => setSearch(e.target.value)}
            />
          </div>
          <div className="flex gap-2 items-center">
            <button
              type="button"
              className={`p-2 rounded-full border ${view === "grid" ? "bg-primary text-white border-primary" : "bg-gray-100 text-gray-500 border-gray-200"}`}
              onClick={() => setView("grid")}
              aria-label="Grid view"
            >
              <LayoutGrid className="h-5 w-5" />
            </button>
            <button
              type="button"
              className={`p-2 rounded-full border ${view === "list" ? "bg-primary text-white border-primary" : "bg-gray-100 text-gray-500 border-gray-200"}`}
              onClick={() => setView("list")}
              aria-label="List view"
            >
              <List className="h-5 w-5" />
            </button>
          </div>
        </form>
        {/* Fixtures List */}
        {view === "grid" ? (
          <div className="grid gap-6 sm:grid-cols-1 md:grid-cols-2 justify-center w-full overflow-x-hidden max-w-4xl mx-auto">
            {filteredFixtures.map((f, i) => (
              <div key={i} className="bg-white rounded-xl shadow-lg p-6 flex flex-col gap-4 border border-gray-100 hover:shadow-2xl transition-shadow duration-300 w-full">
                {/* Teams Row */}
                <div className="flex items-center gap-4 w-full justify-center">
                  <img src={f.teamA.logo} alt={f.teamA.name} className="w-12 h-12 rounded-full object-cover border-2 border-primary" />
                  <div className="font-bold text-lg text-primary">{f.teamA.name}</div>
                  <span className="text-gray-500 font-bold text-xl">vs</span>
                  <img src={f.teamB.logo} alt={f.teamB.name} className="w-12 h-12 rounded-full object-cover border-2 border-secondary" />
                  <div className="font-bold text-lg text-secondary">{f.teamB.name}</div>
                </div>
                {/* Date/Venue and Result/Status Row */}
                <div className="flex flex-col mt-4 sm:flex-row items-center justify-between gap-4 w-full">
                  <div className="flex flex-col items-center sm:items-start gap-1 min-w-[120px]">
                    <div className="text-gray-700 font-medium">{f.date} <span className="text-gray-400">|</span> {f.time}</div>
                    <div className="text-gray-500 text-sm">{f.venue}</div>
                  </div>
                  <div className="flex flex-col items-center gap-1 min-w-[100px]">
                    <div className="font-bold text-lg">{f.result}</div>
                    {statusBadge(f.status)}
                  </div>
                </div>
              </div>
            ))}
            {filteredFixtures.length === 0 && (
              <div className="col-span-full text-center text-gray-500 py-8">No fixtures found.</div>
            )}
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full text-left border rounded-xl bg-white shadow">
              <thead>
                <tr className="bg-gray-100">
                  <th className="py-2 px-4">Date</th>
                  <th className="py-2 px-4">Time</th>
                  <th className="py-2 px-4">Team A</th>
                  <th className="py-2 px-4">Team B</th>
                  <th className="py-2 px-4">Venue</th>
                  <th className="py-2 px-4">Result</th>
                  <th className="py-2 px-4">Status</th>
                </tr>
              </thead>
              <tbody>
                {filteredFixtures.map((f, i) => (
                  <tr key={i} className="border-b hover:bg-gray-50">
                    <td className="py-2 px-4">{f.date}</td>
                    <td className="py-2 px-4">{f.time}</td>
                    <td className="py-2 px-4 flex items-center gap-2"><img src={f.teamA.logo} alt={f.teamA.name} className="w-8 h-8 rounded-full object-cover border-2 border-primary" />{f.teamA.name}</td>
                    <td className="py-2 px-4 flex items-center gap-2"><img src={f.teamB.logo} alt={f.teamB.name} className="w-8 h-8 rounded-full object-cover border-2 border-secondary" />{f.teamB.name}</td>
                    <td className="py-2 px-4">{f.venue}</td>
                    <td className="py-2 px-4 font-bold">{f.result}</td>
                    <td className="py-2 px-4">{statusBadge(f.status)}</td>
                  </tr>
                ))}
                {filteredFixtures.length === 0 && (
                  <tr><td colSpan={7} className="text-center text-gray-500 py-8">No fixtures found.</td></tr>
                )}
              </tbody>
            </table>
          </div>
        )}
      </section>
      {/* Call to Action */}
      <section className="py-12 bg-gradient-to-r from-primary/10 to-secondary/10">
        <div className="max-w-2xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-4 text-primary">Want to see your team here?</h2>
          <p className="mb-6 text-lg text-gray-700">Register your team now or check out the full results!</p>
          <a href="/register" className="inline-block px-8 py-3 rounded-full bg-gradient-to-r from-primary to-secondary text-white font-bold text-lg shadow hover:from-secondary hover:to-primary transition-colors mr-4">Register Team</a>
          <a href="/certificates" className="inline-block px-8 py-3 rounded-full bg-white text-primary font-bold text-lg shadow border border-primary hover:bg-primary hover:text-white transition-colors">View Results</a>
        </div>
      </section>
      <Footer />
    </div>
  );
};

export default Fixtures; 