# Toast Component

A comprehensive toast notification system built with Radix UI and Tailwind CSS.

## Features

- ✅ Multiple variants (success, error, warning, info, default)
- ✅ Custom icons for each variant
- ✅ Promise-based toasts for async operations
- ✅ Custom actions support
- ✅ Swipe to dismiss
- ✅ Auto-dismiss with configurable delay
- ✅ TypeScript support
- ✅ Dark mode support

## Quick Start

### 1. Add ToastProvider to your app

```tsx
import { ToastProvider } from '@/frontend/components/ui/toast'

function App() {
  return (
    <ToastProvider>
      {/* Your app content */}
    </ToastProvider>
  )
}
```

### 2. Use toast utilities

```tsx
import { showSuccessToast, showErrorToast, showWarningToast, showInfoToast } from '@/frontend/components/ui/toast'

// Success toast
showSuccessToast("Success!", "Your action was completed successfully.")

// Error toast
showErrorToast("Error!", "Something went wrong. Please try again.")

// Warning toast
showWarningToast("Warning!", "Please check your input before proceeding.")

// Info toast
showInfoToast("Information", "Here's some helpful information for you.")
```

## API Reference

### Toast Variants

| Variant | Description | Icon | Color |
|---------|-------------|------|-------|
| `success` | Success messages | CheckCircle | Green |
| `error` | Error messages | XCircle | Red |
| `warning` | Warning messages | AlertTriangle | Yellow |
| `info` | Information messages | Info | Blue |
| `default` | Default messages | Info | Gray |

### Basic Usage

```tsx
import { showToast } from '@/frontend/components/ui/toast'

// Using the main showToast function
showToast.success("Title", "Description")
showToast.error("Title", "Description")
showToast.warning("Title", "Description")
showToast.info("Title", "Description")
showToast.default("Title", "Description")
```

### Custom Toast with Action

```tsx
import { showToast } from '@/frontend/components/ui/toast'
import { Button } from '@/components/ui/button'

showToast.withAction(
  "Custom Toast",
  "This toast has a custom action button.",
  <Button size="sm" variant="outline">
    Action
  </Button>
)
```

### Promise-based Toast

```tsx
import { showPromiseToast } from '@/frontend/components/ui/toast'

const handleAsyncAction = () => {
  const promise = fetch('/api/data')
    .then(response => response.json())
    .then(data => data)

  showPromiseToast(promise, {
    loading: "Loading data...",
    success: (data) => `Successfully loaded: ${data.message}`,
    error: (error) => `Failed to load: ${error.message}`,
  })
}
```

### Using the Hook

```tsx
import { useToast } from '@/frontend/components/ui/toast'

function MyComponent() {
  const { toast } = useToast()

  const handleClick = () => {
    toast({
      title: "Custom Toast",
      description: "This is a custom toast using the hook.",
      variant: "success",
    })
  }

  return <button onClick={handleClick}>Show Toast</button>
}
```

### Manual Toast Control

```tsx
import { toast } from '@/frontend/components/ui/toast'

const myToast = toast({
  title: "Manual Control",
  description: "This toast can be controlled manually.",
  variant: "info",
})

// Dismiss the toast
myToast.dismiss()

// Update the toast
myToast.update({
  title: "Updated Title",
  description: "This toast has been updated.",
})
```

## Styling

The toast component uses Tailwind CSS classes and supports dark mode automatically. You can customize the appearance by modifying the `toastVariants` in `toast.tsx`.

### Custom Variants

```tsx
// Add custom variant to toastVariants
const toastVariants = cva(
  "group pointer-events-auto relative flex w-full items-center justify-between space-x-4 overflow-hidden rounded-md border p-6 pr-8 shadow-lg transition-all",
  {
    variants: {
      variant: {
        // ... existing variants
        custom: "border-purple-200 bg-purple-50 text-purple-900",
      },
    },
  }
)
```

## Configuration

### Toast Limits

By default, only 1 toast is shown at a time. You can modify this in `use-toast.ts`:

```tsx
const TOAST_LIMIT = 1 // Change this number
```

### Auto-dismiss Delay

The auto-dismiss delay can be configured:

```tsx
const TOAST_REMOVE_DELAY = 1000000 // Change this value (in milliseconds)
```

## Examples

See `toast-examples.tsx` for comprehensive usage examples including all variants and features.

## Accessibility

The toast component is built on Radix UI primitives and includes:

- Screen reader support
- Keyboard navigation
- Focus management
- ARIA attributes
- High contrast mode support

## Browser Support

- Chrome/Edge 88+
- Firefox 78+
- Safari 14+
- Mobile browsers (iOS Safari 14+, Chrome Mobile 88+)
