import { toast } from "@/hooks/use-toast"
import { CheckCircle, XCircle, AlertTriangle, Info } from "lucide-react"

export const toastVariants = {
  success: {
    variant: "success" as const,
    icon: CheckCircle,
  },
  error: {
    variant: "destructive" as const,
    icon: XCircle,
  },
  warning: {
    variant: "warning" as const,
    icon: AlertTriangle,
  },
  info: {
    variant: "info" as const,
    icon: Info,
  },
  default: {
    variant: "default" as const,
    icon: Info
  },
} as const

export const showToast = {
  success: (title: string, description?: string) => {
    return toast({
      title,
      description,
      variant: "success",
    })
  },

  error: (title: string, description?: string) => {
    return toast({
      title,
      description,
      variant: "destructive",
    })
  },

  warning: (title: string, description?: string) => {
    return toast({
      title,
      description,
      variant: "warning",
    })
  },

  info: (title: string, description?: string) => {
    return toast({
      title,
      description,
      variant: "info",
    })
  },

  default: (title: string, description?: string) => {
    return toast({
      title,
      description,
      variant: "default",
    })
  },

  // Custom toast with action
  withAction: (title: string, description?: string, action?: React.ReactNode) => {
    return toast({
      title,
      description,
      action: action as import("./toast").ToastActionElement | undefined,
    })
  },

  // Promise-based toast for async operations
  promise: function<T>(
    promise: Promise<T>,
    {
      loading,
      success,
      error,
    }: {    
      loading: string;
      success: string | ((data: T) => string);
      error: string | ((error: any) => string);
    }
  ) {
    const loadingToast = toast({
      title: loading,
      variant: "default",
    })

    promise
      .then((data) => {
        loadingToast.dismiss()
        const successMessage = typeof success === "function" ? success(data) : success
        showToast.success("Success", successMessage)
      })
      .catch((err) => {
        loadingToast.dismiss()
        const errorMessage = typeof error === "function" ? error(err) : error
        showToast.error("Error", errorMessage)
      })

    return loadingToast
  },
}

// Convenience functions for common use cases
export const showSuccessToast = showToast.success
export const showErrorToast = showToast.error
export const showWarningToast = showToast.warning
export const showInfoToast = showToast.info
export const showDefaultToast = showToast.default
export const showPromiseToast = showToast.promise
