import React, { useState } from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Table, 
  TableBody, 
  TableCell, 
  TableHead, 
  TableHeader, 
  TableRow 
} from '@/components/ui/table';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { 
  MoreHorizontal, 
  Eye, 
  Edit, 
  Trash2,
  UserCheck,
  UserX,
  Shield,
  Download,
  Users,
  CheckCircle,
  XCircle
} from 'lucide-react';
import { router } from '@inertiajs/react';

interface User {
  id: number;
  name: string;
  email: string;
  email_verified_at: string | null;
  created_at: string;
  roles: Array<{
    id: number;
    name: string;
    display_name: string;
    color: string;
  }>;
}

interface UsersTableProps {
  users: {
    data: User[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  roles: Array<{
    id: number;
    name: string;
    display_name: string;
  }>;
  filters: {
    search: string;
    role: string;
    status: string;
    sort_by: string;
    sort_order: string;
  };
}

const UsersTable: React.FC<UsersTableProps> = ({ users, roles, filters }) => {
  const [selectedUsers, setSelectedUsers] = useState<number[]>([]);
  const [isSelectAll, setIsSelectAll] = useState(false);

  const handleFilterChange = (key: string, value: string) => {
    router.get(route('admin.users.index'), {
      ...filters,
      [key]: value,
    }, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    handleFilterChange('sort_by', column);
    handleFilterChange('sort_order', newOrder);
  };

  const handleSelectAll = (checked: boolean) => {
    setIsSelectAll(checked);
    if (checked) {
      // Exclude super admin users from bulk selection
      const selectableUsers = users.data.filter(user => 
        !user.roles.some(role => role.name === 'super_admin')
      );
      setSelectedUsers(selectableUsers.map(user => user.id));
    } else {
      setSelectedUsers([]);
    }
  };

  const handleSelectUser = (userId: number, checked: boolean) => {
    if (checked) {
      setSelectedUsers([...selectedUsers, userId]);
    } else {
      setSelectedUsers(selectedUsers.filter(id => id !== userId));
      setIsSelectAll(false);
    }
  };

  const handleDelete = (userId: number) => {
    if (confirm('Are you sure you want to delete this user?')) {
      router.delete(route('admin.users.destroy', userId));
    }
  };

  const handleBulkDelete = () => {
    if (selectedUsers.length === 0) return;
    
    if (confirm(`Are you sure you want to delete ${selectedUsers.length} users?`)) {
      router.post(route('admin.users.bulkDelete'), { ids: selectedUsers }, {
        onSuccess: () => {
          setSelectedUsers([]);
          setIsSelectAll(false);
        }
      });
    }
  };

  const handleImpersonate = (userId: number) => {
    if (confirm('Are you sure you want to impersonate this user?')) {
      router.post(route('admin.users.impersonate', userId));
    }
  };

  const handleExport = () => {
    window.location.href = route('admin.users.export', filters);
  };

  const getRoleColor = (role: any) => {
    const colors: Record<string, string> = {
      'super_admin': 'bg-red-100 text-red-800',
      'admin': 'bg-purple-100 text-purple-800',
      'organizer': 'bg-blue-100 text-blue-800',
      'referee': 'bg-green-100 text-green-800',
      'player': 'bg-yellow-100 text-yellow-800',
      'spectator': 'bg-gray-100 text-gray-800',
    };
    return colors[role.name] || 'bg-gray-100 text-gray-800';
  };

  return (
    <Card className="overflow-hidden">
      {/* Header Actions */}
      <div className="p-4 border-b bg-muted/50">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center gap-2">
            {selectedUsers.length > 0 && (
              <>
                <Button
                  variant="destructive"
                  size="sm"
                  onClick={handleBulkDelete}
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  Delete Selected ({selectedUsers.length})
                </Button>
              </>
            )}
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              size="sm"
              onClick={handleExport}
            >
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>
            <Button
              onClick={() => router.get(route('admin.users.create'))}
              size="sm"
            >
              <Users className="w-4 h-4 mr-2" />
              Add User
            </Button>
          </div>
        </div>

        {/* Filters */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Input
            placeholder="Search users..."
            value={filters.search}
            onChange={(e) => handleFilterChange('search', e.target.value)}
          />
          
          <Select value={filters.role} onValueChange={(value) => handleFilterChange('role', value)}>
            <SelectTrigger>
              <SelectValue placeholder="All Roles" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Roles</SelectItem>
              {roles.map((role) => (
                <SelectItem key={role.id} value={role.name}>
                  {role.display_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          <Select value={filters.status} onValueChange={(value) => handleFilterChange('status', value)}>
            <SelectTrigger>
              <SelectValue placeholder="All Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="verified">Verified</SelectItem>
              <SelectItem value="unverified">Unverified</SelectItem>
            </SelectContent>
          </Select>

          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">
              {users.total} users total
            </span>
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="max-h-[600px] overflow-auto">
        <Table>
          <TableHeader className="sticky top-0 bg-background z-10">
            <TableRow>
              <TableHead className="w-12">
                <Checkbox
                  checked={isSelectAll}
                  onCheckedChange={handleSelectAll}
                  disabled={users.data.every(user => user.roles.some(role => role.name === 'super_admin'))}
                />
              </TableHead>
              <TableHead 
                className="cursor-pointer"
                onClick={() => handleSort('name')}
              >
                User
                {filters.sort_by === 'name' && (
                  <span className="ml-1">{filters.sort_order === 'asc' ? '↑' : '↓'}</span>
                )}
              </TableHead>
              <TableHead 
                className="cursor-pointer"
                onClick={() => handleSort('email')}
              >
                Email
                {filters.sort_by === 'email' && (
                  <span className="ml-1">{filters.sort_order === 'asc' ? '↑' : '↓'}</span>
                )}
              </TableHead>
              <TableHead>Roles</TableHead>
              <TableHead>Status</TableHead>
              <TableHead 
                className="cursor-pointer"
                onClick={() => handleSort('created_at')}
              >
                Created
                {filters.sort_by === 'created_at' && (
                  <span className="ml-1">{filters.sort_order === 'asc' ? '↑' : '↓'}</span>
                )}
              </TableHead>
              <TableHead className="w-12">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {users.data.map((user) => (
              <TableRow 
                key={user.id}
                className={user.roles.some(role => role.name === 'super_admin') ? 'bg-red-50' : ''}
              >
                <TableCell>
                  <Checkbox
                    checked={selectedUsers.includes(user.id)}
                    onCheckedChange={(checked) => handleSelectUser(user.id, checked as boolean)}
                    disabled={user.roles.some(role => role.name === 'super_admin')}
                  />
                </TableCell>
                <TableCell>
                  <div>
                    <div className="font-medium">{user.name}</div>
                    <div className="text-sm text-muted-foreground">ID: {user.id}</div>
                  </div>
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <span>{user.email}</span>
                    {user.email_verified_at ? (
                      <CheckCircle className="w-4 h-4 text-green-600" />
                    ) : (
                      <XCircle className="w-4 h-4 text-red-600" />
                    )}
                  </div>
                </TableCell>
                <TableCell>
                  <div className="flex flex-wrap gap-1">
                    {user.roles.map((role) => (
                      <Badge key={role.id} className={getRoleColor(role)}>
                        {role.display_name}
                      </Badge>
                    ))}
                    {user.roles.length === 0 && (
                      <Badge variant="outline">No roles</Badge>
                    )}
                  </div>
                </TableCell>
                <TableCell>
                  <Badge 
                    className={user.email_verified_at 
                      ? 'bg-green-100 text-green-800' 
                      : 'bg-red-100 text-red-800'
                    }
                  >
                    {user.email_verified_at ? 'Verified' : 'Unverified'}
                  </Badge>
                </TableCell>
                <TableCell>
                  <div className="text-sm">
                    {new Date(user.created_at).toLocaleDateString()}
                  </div>
                </TableCell>
                <TableCell>
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" size="sm">
                        <MoreHorizontal className="w-4 h-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                      <DropdownMenuItem onClick={() => router.get(route('admin.users.show', user.id))}>
                        <Eye className="w-4 h-4 mr-2" />
                        View
                      </DropdownMenuItem>
                      {!user.roles.some(role => role.name === 'super_admin') ? (
                        <>
                          <DropdownMenuItem onClick={() => router.get(route('admin.users.edit', user.id))}>
                            <Edit className="w-4 h-4 mr-2" />
                            Edit
                          </DropdownMenuItem>
                          <DropdownMenuSeparator />
                          <DropdownMenuItem onClick={() => handleImpersonate(user.id)}>
                            <Shield className="w-4 h-4 mr-2" />
                            Login as User
                          </DropdownMenuItem>
                          <DropdownMenuSeparator />
                          <DropdownMenuItem 
                            onClick={() => handleDelete(user.id)}
                            className="text-red-600"
                          >
                            <Trash2 className="w-4 h-4 mr-2" />
                            Delete
                          </DropdownMenuItem>
                        </>
                      ) : (
                        <DropdownMenuItem disabled className="text-muted-foreground">
                          <Shield className="w-4 h-4 mr-2" />
                          Super Admin - Actions Restricted
                        </DropdownMenuItem>
                      )}
                    </DropdownMenuContent>
                  </DropdownMenu>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </div>

      {/* Pagination */}
      <div className="p-4 border-t bg-muted/50">
        <div className="flex items-center justify-between">
          <div className="text-sm text-muted-foreground">
            Showing {((users.current_page - 1) * users.per_page) + 1} to{' '}
            {Math.min(users.current_page * users.per_page, users.total)} of{' '}
            {users.total} results
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              size="sm"
              disabled={users.current_page === 1}
              onClick={() => handleFilterChange('page', String(users.current_page - 1))}
            >
              Previous
            </Button>
            <span className="text-sm">
              Page {users.current_page} of {users.last_page}
            </span>
            <Button
              variant="outline"
              size="sm"
              disabled={users.current_page === users.last_page}
              onClick={() => handleFilterChange('page', String(users.current_page + 1))}
            >
              Next
            </Button>
          </div>
        </div>
      </div>
    </Card>
  );
};

export default UsersTable; 