import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { Switch } from '@/components/ui/switch';
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from '@/components/ui/select';
import { 
  User, 
  Mail, 
  Lock, 
  Shield, 
  CheckCircle,
  AlertCircle,
  Phone,
  MapPin,
  Calendar,
  FileText,
  UserCheck
} from 'lucide-react';

interface Role {
  id: number;
  name: string;
  display_name: string;
  description: string;
  color: string;
}

interface UserFormProps {
  form: {
    name: string;
    email: string;
    password: string;
    password_confirmation: string;
    roles: number[];
    email_verified_at: boolean;
    send_welcome_email: boolean;
    // KYC fields
    phone?: string;
    id_number?: string;
    passport_number?: string;
    date_of_birth?: string;
    gender?: string;
    nationality?: string;
    county?: string;
    sub_county?: string;
    ward?: string;
    address?: string;
    postal_code?: string;
    occupation?: string;
    employer?: string;
    emergency_contact_name?: string;
    emergency_contact_phone?: string;
    emergency_contact_relationship?: string;
    bio?: string;
    kyc_status?: string;
    kyc_notes?: string;
  };
  setForm: (form: any) => void;
  errors: Record<string, string>;
  roles: Role[];
  isEditing?: boolean;
}

const UserForm: React.FC<UserFormProps> = ({ 
  form, 
  setForm, 
  errors, 
  roles,
  isEditing = false 
}) => {
  const handleRoleToggle = (roleId: number) => {
    const currentRoles = form.roles || [];
    if (currentRoles.includes(roleId)) {
      setForm({
        ...form,
        roles: currentRoles.filter(id => id !== roleId)
      });
    } else {
      setForm({
        ...form,
        roles: [...currentRoles, roleId]
      });
    }
  };

  const getRoleColor = (role: Role) => {
    const colors: Record<string, string> = {
      'super_admin': 'bg-red-100 text-red-800 border-red-200',
      'admin': 'bg-purple-100 text-purple-800 border-purple-200',
      'organizer': 'bg-blue-100 text-blue-800 border-blue-200',
      'referee': 'bg-green-100 text-green-800 border-green-200',
      'player': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'spectator': 'bg-gray-100 text-gray-800 border-gray-200',
    };
    return colors[role.name] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  return (
    <div className="space-y-6">
      {/* Basic Information */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <User className="w-5 h-5" />
            Basic Information
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="name">Full Name *</Label>
              <Input
                id="name"
                type="text"
                value={form.name}
                onChange={(e) => setForm({ ...form, name: e.target.value })}
                placeholder="Enter full name"
                className={errors.name ? 'border-red-500' : ''}
              />
              {errors.name && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.name}
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="email">Email Address *</Label>
              <Input
                id="email"
                type="email"
                value={form.email}
                onChange={(e) => setForm({ ...form, email: e.target.value })}
                placeholder="Enter email address"
                className={errors.email ? 'border-red-500' : ''}
              />
              {errors.email && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.email}
                </p>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* KYC Information */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <UserCheck className="w-5 h-5" />
            KYC Information (Optional)
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Personal Details */}
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Personal Details</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <div className="space-y-2">
                <Label htmlFor="phone">Phone Number</Label>
                <Input
                  id="phone"
                  type="tel"
                  value={form.phone || ''}
                  onChange={(e) => setForm({ ...form, phone: e.target.value })}
                  placeholder="e.g., +254712345678"
                  className={errors.phone ? 'border-red-500' : ''}
                />
                {errors.phone && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.phone}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="id_number">National ID Number</Label>
                <Input
                  id="id_number"
                  type="text"
                  value={form.id_number || ''}
                  onChange={(e) => setForm({ ...form, id_number: e.target.value })}
                  placeholder="e.g., 12345678"
                  className={errors.id_number ? 'border-red-500' : ''}
                />
                {errors.id_number && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.id_number}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="passport_number">Passport Number</Label>
                <Input
                  id="passport_number"
                  type="text"
                  value={form.passport_number || ''}
                  onChange={(e) => setForm({ ...form, passport_number: e.target.value })}
                  placeholder="e.g., A1234567"
                  className={errors.passport_number ? 'border-red-500' : ''}
                />
                {errors.passport_number && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.passport_number}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="date_of_birth">Date of Birth</Label>
                <Input
                  id="date_of_birth"
                  type="date"
                  value={form.date_of_birth || ''}
                  onChange={(e) => setForm({ ...form, date_of_birth: e.target.value })}
                  className={errors.date_of_birth ? 'border-red-500' : ''}
                />
                {errors.date_of_birth && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.date_of_birth}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="gender">Gender</Label>
                <Select
                  value={form.gender || ''}
                  onValueChange={(value) => setForm({ ...form, gender: value })}
                >
                  <SelectTrigger className={errors.gender ? 'border-red-500' : ''}>
                    <SelectValue placeholder="Select gender" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="male">Male</SelectItem>
                    <SelectItem value="female">Female</SelectItem>
                    <SelectItem value="other">Other</SelectItem>
                  </SelectContent>
                </Select>
                {errors.gender && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.gender}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="nationality">Nationality</Label>
                <Input
                  id="nationality"
                  type="text"
                  value={form.nationality || ''}
                  onChange={(e) => setForm({ ...form, nationality: e.target.value })}
                  placeholder="e.g., Kenyan"
                  className={errors.nationality ? 'border-red-500' : ''}
                />
                {errors.nationality && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.nationality}
                  </p>
                )}
              </div>
            </div>
          </div>

          {/* Location Details */}
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Location Details</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <div className="space-y-2">
                <Label htmlFor="county">County</Label>
                <Input
                  id="county"
                  type="text"
                  value={form.county || ''}
                  onChange={(e) => setForm({ ...form, county: e.target.value })}
                  placeholder="e.g., Nairobi"
                  className={errors.county ? 'border-red-500' : ''}
                />
                {errors.county && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.county}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="sub_county">Sub County</Label>
                <Input
                  id="sub_county"
                  type="text"
                  value={form.sub_county || ''}
                  onChange={(e) => setForm({ ...form, sub_county: e.target.value })}
                  placeholder="e.g., Westlands"
                  className={errors.sub_county ? 'border-red-500' : ''}
                />
                {errors.sub_county && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.sub_county}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="ward">Ward</Label>
                <Input
                  id="ward"
                  type="text"
                  value={form.ward || ''}
                  onChange={(e) => setForm({ ...form, ward: e.target.value })}
                  placeholder="e.g., Parklands"
                  className={errors.ward ? 'border-red-500' : ''}
                />
                {errors.ward && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.ward}
                  </p>
                )}
              </div>

              <div className="space-y-2 md:col-span-2">
                <Label htmlFor="address">Physical Address</Label>
                <Input
                  id="address"
                  type="text"
                  value={form.address || ''}
                  onChange={(e) => setForm({ ...form, address: e.target.value })}
                  placeholder="e.g., 123 Main Street, Nairobi"
                  className={errors.address ? 'border-red-500' : ''}
                />
                {errors.address && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.address}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="postal_code">Postal Code</Label>
                <Input
                  id="postal_code"
                  type="text"
                  value={form.postal_code || ''}
                  onChange={(e) => setForm({ ...form, postal_code: e.target.value })}
                  placeholder="e.g., 00100"
                  className={errors.postal_code ? 'border-red-500' : ''}
                />
                {errors.postal_code && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.postal_code}
                  </p>
                )}
              </div>
            </div>
          </div>

          {/* Employment Details */}
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Employment Details</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="occupation">Occupation</Label>
                <Input
                  id="occupation"
                  type="text"
                  value={form.occupation || ''}
                  onChange={(e) => setForm({ ...form, occupation: e.target.value })}
                  placeholder="e.g., Software Developer"
                  className={errors.occupation ? 'border-red-500' : ''}
                />
                {errors.occupation && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.occupation}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="employer">Employer</Label>
                <Input
                  id="employer"
                  type="text"
                  value={form.employer || ''}
                  onChange={(e) => setForm({ ...form, employer: e.target.value })}
                  placeholder="e.g., ABC Company Ltd"
                  className={errors.employer ? 'border-red-500' : ''}
                />
                {errors.employer && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.employer}
                  </p>
                )}
              </div>
            </div>
          </div>

          {/* Emergency Contact */}
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Emergency Contact</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <div className="space-y-2">
                <Label htmlFor="emergency_contact_name">Contact Name</Label>
                <Input
                  id="emergency_contact_name"
                  type="text"
                  value={form.emergency_contact_name || ''}
                  onChange={(e) => setForm({ ...form, emergency_contact_name: e.target.value })}
                  placeholder="e.g., John Doe"
                  className={errors.emergency_contact_name ? 'border-red-500' : ''}
                />
                {errors.emergency_contact_name && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.emergency_contact_name}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="emergency_contact_phone">Contact Phone</Label>
                <Input
                  id="emergency_contact_phone"
                  type="tel"
                  value={form.emergency_contact_phone || ''}
                  onChange={(e) => setForm({ ...form, emergency_contact_phone: e.target.value })}
                  placeholder="e.g., +254712345678"
                  className={errors.emergency_contact_phone ? 'border-red-500' : ''}
                />
                {errors.emergency_contact_phone && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.emergency_contact_phone}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="emergency_contact_relationship">Relationship</Label>
                <Input
                  id="emergency_contact_relationship"
                  type="text"
                  value={form.emergency_contact_relationship || ''}
                  onChange={(e) => setForm({ ...form, emergency_contact_relationship: e.target.value })}
                  placeholder="e.g., Spouse, Parent, Sibling"
                  className={errors.emergency_contact_relationship ? 'border-red-500' : ''}
                />
                {errors.emergency_contact_relationship && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.emergency_contact_relationship}
                  </p>
                )}
              </div>
            </div>
          </div>

          {/* Additional Information */}
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Additional Information</h4>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="bio">Biography</Label>
                <textarea
                  id="bio"
                  value={form.bio || ''}
                  onChange={(e) => setForm({ ...form, bio: e.target.value })}
                  placeholder="Tell us about yourself..."
                  rows={3}
                  className={`w-full px-3 py-2 border rounded-md ${errors.bio ? 'border-red-500' : 'border-gray-300'}`}
                />
                {errors.bio && (
                  <p className="text-sm text-red-600 flex items-center gap-1">
                    <AlertCircle className="w-4 h-4" />
                    {errors.bio}
                  </p>
                )}
              </div>

              {isEditing && (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="kyc_status">KYC Status</Label>
                    <Select
                      value={form.kyc_status || 'not_submitted'}
                      onValueChange={(value) => setForm({ ...form, kyc_status: value })}
                    >
                      <SelectTrigger className={errors.kyc_status ? 'border-red-500' : ''}>
                        <SelectValue placeholder="Select KYC status" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="not_submitted">Not Submitted</SelectItem>
                        <SelectItem value="pending">Pending</SelectItem>
                        <SelectItem value="verified">Verified</SelectItem>
                        <SelectItem value="rejected">Rejected</SelectItem>
                      </SelectContent>
                    </Select>
                    {errors.kyc_status && (
                      <p className="text-sm text-red-600 flex items-center gap-1">
                        <AlertCircle className="w-4 h-4" />
                        {errors.kyc_status}
                      </p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="kyc_notes">KYC Notes</Label>
                    <textarea
                      id="kyc_notes"
                      value={form.kyc_notes || ''}
                      onChange={(e) => setForm({ ...form, kyc_notes: e.target.value })}
                      placeholder="Additional notes about KYC verification..."
                      rows={2}
                      className={`w-full px-3 py-2 border rounded-md ${errors.kyc_notes ? 'border-red-500' : 'border-gray-300'}`}
                    />
                    {errors.kyc_notes && (
                      <p className="text-sm text-red-600 flex items-center gap-1">
                        <AlertCircle className="w-4 h-4" />
                        {errors.kyc_notes}
                      </p>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Password Section */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Lock className="w-5 h-5" />
            {isEditing ? 'Change Password' : 'Password'}
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="password">
                {isEditing ? 'New Password' : 'Password'} {!isEditing && '*'}
              </Label>
              <Input
                id="password"
                type="password"
                value={form.password}
                onChange={(e) => setForm({ ...form, password: e.target.value })}
                placeholder={isEditing ? 'Leave blank to keep current password' : 'Enter password'}
                className={errors.password ? 'border-red-500' : ''}
              />
              {errors.password && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.password}
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="password_confirmation">
                Confirm Password {!isEditing && '*'}
              </Label>
              <Input
                id="password_confirmation"
                type="password"
                value={form.password_confirmation}
                onChange={(e) => setForm({ ...form, password_confirmation: e.target.value })}
                placeholder="Confirm password"
                className={errors.password_confirmation ? 'border-red-500' : ''}
              />
              {errors.password_confirmation && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.password_confirmation}
                </p>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Roles & Permissions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Shield className="w-5 h-5" />
            Roles & Permissions
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {roles.map((role) => (
              <div
                key={role.id}
                className={`p-4 rounded-lg border-2 transition-all ${
                  form.roles?.includes(role.id)
                    ? getRoleColor(role)
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="flex items-start justify-between">
                  <div className="flex-1">
                    <div className="flex items-center gap-2">
                      <Checkbox
                        checked={form.roles?.includes(role.id) || false}
                        onCheckedChange={() => handleRoleToggle(role.id)}
                        disabled={role.name === 'super_admin'}
                      />
                      <h4 className="font-medium">{role.display_name}</h4>
                      {role.name === 'super_admin' && (
                        <Badge variant="destructive" className="text-xs">
                          Restricted
                        </Badge>
                      )}
                    </div>
                    <p className="text-sm text-muted-foreground mt-1">
                      {role.description}
                      {role.name === 'super_admin' && (
                        <span className="block text-red-600 text-xs mt-1">
                          This role cannot be assigned through the interface
                        </span>
                      )}
                    </p>
                  </div>
                </div>
              </div>
            ))}
          </div>
          
          {form.roles?.length === 0 && (
            <div className="text-center py-8 text-muted-foreground">
              <Shield className="w-12 h-12 mx-auto mb-2 opacity-50" />
              <p>No roles selected. User will have minimal permissions.</p>
            </div>
          )}

          {errors.roles && (
            <p className="text-sm text-red-600 flex items-center gap-1">
              <AlertCircle className="w-4 h-4" />
              {errors.roles}
            </p>
          )}
        </CardContent>
      </Card>

      {/* Account Settings */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CheckCircle className="w-5 h-5" />
            Account Settings
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-4">
            {isEditing && (
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="email_verified">Email Verified</Label>
                  <p className="text-sm text-muted-foreground">
                    Mark this user's email as verified
                  </p>
                </div>
                <Switch
                  id="email_verified"
                  checked={form.email_verified_at}
                  onCheckedChange={(checked) => 
                    setForm({ ...form, email_verified_at: checked })
                  }
                />
              </div>
            )}

            {!isEditing && (
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="send_welcome_email">Send Welcome Email</Label>
                  <p className="text-sm text-muted-foreground">
                    Send a welcome email with login instructions
                  </p>
                </div>
                <Switch
                  id="send_welcome_email"
                  checked={form.send_welcome_email}
                  onCheckedChange={(checked) => 
                    setForm({ ...form, send_welcome_email: checked })
                  }
                />
              </div>
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default UserForm; 