import React, { useState, useMemo } from 'react';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { ChevronLeft, ChevronRight, Trophy, Users, Clock, Play, Settings, BarChart3 } from 'lucide-react';

interface Participant {
  id: number;
  name: string;
  type: 'player' | 'team';
  avatar?: string;
  score?: number;
  rating?: number;
  ranking?: number;
}

interface Match {
  id: number;
  round: number;
  match_number: number;
  player1_id?: number;
  player2_id?: number;
  team1_id?: number;
  team2_id?: number;
  player1_score?: number;
  player2_score?: number;
  team1_score?: number;
  team2_score?: number;
  winner_id?: number;
  winning_team_id?: number;
  status: 'scheduled' | 'in_progress' | 'completed' | 'cancelled';
  scheduled_time?: string;
  started_at?: string;
  completed_at?: string;
  player1?: Participant;
  player2?: Participant;
  team1?: Participant;
  team2?: Participant;
  winner?: Participant;
  winning_team?: Participant;
  group_number?: number;
  bracket_type?: string;
}

interface BracketStructure {
  type: string;
  total_participants: number;
  total_matches: number;
  rounds: number;
  byes?: number;
  power_of_two?: number;
  groups?: number;
  participants_per_group?: number;
  group_stage_completed?: boolean;
  matches_per_round?: number;
}

interface BracketVisualizationProps {
  bracket: {
    id: number;
    name: string;
    type: string;
    rounds: number;
    structure: BracketStructure;
    seeding: any;
    is_active: boolean;
  };
  matches: Match[];
  players?: Participant[];
  teams?: Participant[];
  onMatchUpdate?: (matchId: number, data: any) => void;
  onAdvanceRound?: (round: number) => void;
  showControls?: boolean;
}

const EnhancedBracketVisualization: React.FC<BracketVisualizationProps> = ({
  bracket,
  matches = [],
  players = [],
  teams = [],
  onMatchUpdate,
  onAdvanceRound,
  showControls = true
}) => {
  const [currentRound, setCurrentRound] = useState(1);
  const [expandedRounds, setExpandedRounds] = useState<Set<number>>(new Set([1]));
  const [viewMode, setViewMode] = useState<'bracket' | 'stats'>('bracket');

  // Safety check: ensure bracket.structure exists
  if (!bracket.structure) {
    return (
      <div className="p-8 text-center">
        <p className="text-gray-500">Bracket structure not available</p>
      </div>
    );
  }

  // Create lookup maps for participants
  const participantMap = useMemo(() => {
    const map = new Map<number, Participant>();
    [...players, ...teams].forEach(p => map.set(p.id, p));
    return map;
  }, [players, teams]);

  const getParticipant = (id?: number): Participant | undefined => {
    return id ? participantMap.get(id) : undefined;
  };

  const getParticipantName = (participant?: Participant): string => {
    return participant?.name || 'TBD';
  };

  const getParticipantScore = (match: Match, participantId?: number): number => {
    if (match.player1_id === participantId) return match.player1_score || 0;
    if (match.player2_id === participantId) return match.player2_score || 0;
    if (match.team1_id === participantId) return match.team1_score || 0;
    if (match.team2_id === participantId) return match.team2_score || 0;
    return 0;
  };

  const isWinner = (match: Match, participantId?: number): boolean => {
    return match.winner_id === participantId || match.winning_team_id === participantId;
  };

  const getMatchStatusColor = (status: string): string => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800 border-green-200';
      case 'in_progress': return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'cancelled': return 'bg-red-100 text-red-800 border-red-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getRoundTitle = (round: number): string => {
    const totalRounds = bracket.structure.rounds;
    if (round === totalRounds) return 'Final';
    if (round === totalRounds - 1) return 'Semi-Final';
    if (round === totalRounds - 2) return 'Quarter-Final';
    return `Round ${round}`;
  };

  const renderMatch = (match: Match) => {
    const participant1 = getParticipant(match.player1_id || match.team1_id);
    const participant2 = getParticipant(match.player2_id || match.team2_id);
    const score1 = getParticipantScore(match, match.player1_id || match.team1_id);
    const score2 = getParticipantScore(match, match.player2_id || match.team2_id);
    const isWinner1 = isWinner(match, match.player1_id || match.team1_id);
    const isWinner2 = isWinner(match, match.player2_id || match.team2_id);

    return (
      <Card key={match.id} className={`p-4 border-2 ${getMatchStatusColor(match.status)}`}>
        <div className="space-y-2">
          <div className="flex items-center justify-between">
            <Badge variant="outline" className="text-xs">
              Match {match.match_number}
            </Badge>
            <Badge variant="secondary" className="text-xs">
              {match.status}
            </Badge>
          </div>
          
          <div className="space-y-1">
            <div className={`flex items-center justify-between p-2 rounded ${
              isWinner1 ? 'bg-green-50 border-green-200' : 'bg-gray-50'
            }`}>
              <span className="font-medium">{getParticipantName(participant1)}</span>
              <span className="font-bold">{score1}</span>
            </div>
            
            <div className={`flex items-center justify-between p-2 rounded ${
              isWinner2 ? 'bg-green-50 border-green-200' : 'bg-gray-50'
            }`}>
              <span className="font-medium">{getParticipantName(participant2)}</span>
              <span className="font-bold">{score2}</span>
            </div>
          </div>

          {match.scheduled_time && (
            <div className="text-xs text-gray-500 flex items-center gap-1">
              <Clock className="w-3 h-3" />
              {new Date(match.scheduled_time).toLocaleString()}
            </div>
          )}

          {showControls && match.status === 'scheduled' && (
            <Button 
              size="sm" 
              className="w-full"
              onClick={() => onMatchUpdate?.(match.id, { status: 'in_progress' })}
            >
              <Play className="w-3 h-3 mr-1" />
              Start Match
            </Button>
          )}
        </div>
      </Card>
    );
  };

  const renderSingleElimination = () => {
    const rounds = bracket.structure.rounds;
    const maxMatches = bracket.structure.power_of_two || Math.pow(2, rounds);
    
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Trophy className="w-5 h-5" />
            {bracket.name} - Single Elimination
          </h3>
          <div className="flex items-center gap-2">
            <Badge variant="outline">
              <Users className="w-3 h-3 mr-1" />
              {bracket.structure.total_participants} participants
            </Badge>
            <Badge variant="outline">
              {bracket.structure.total_matches} matches
            </Badge>
          </div>
        </div>
        
        <div className="flex gap-8 overflow-x-auto pb-4">
          {Array.from({ length: rounds }, (_, roundIndex) => {
            const round = roundIndex + 1;
            const matchesInRound = matches.filter(match => match.round === round);
            const isExpanded = expandedRounds.has(round);
            
            return (
              <div key={roundIndex} className="flex flex-col gap-4 min-w-[250px]">
                <div className="flex items-center justify-between">
                  <h4 className="text-sm font-medium text-gray-600">
                    {getRoundTitle(round)}
                    {round === 1 && ` (${maxMatches} participants)`}
                  </h4>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => {
                      const newExpanded = new Set(expandedRounds);
                      if (isExpanded) {
                        newExpanded.delete(round);
                      } else {
                        newExpanded.add(round);
                      }
                      setExpandedRounds(newExpanded);
                    }}
                  >
                    {isExpanded ? <ChevronLeft className="w-4 h-4" /> : <ChevronRight className="w-4 h-4" />}
                  </Button>
                </div>
                
                {isExpanded && (
                  <div className="space-y-3">
                    {matchesInRound.map(renderMatch)}
                  </div>
                )}
              </div>
            );
          })}
        </div>
      </div>
    );
  };

  const renderRoundRobin = () => {
    const rounds = bracket.structure.rounds;
    
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Users className="w-5 h-5" />
            {bracket.name} - Round Robin
          </h3>
          <div className="flex items-center gap-2">
            <Badge variant="outline">
              <Users className="w-3 h-3 mr-1" />
              {bracket.structure.total_participants} participants
            </Badge>
            <Badge variant="outline">
              {bracket.structure.total_matches} matches
            </Badge>
          </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: rounds }, (_, roundIndex) => {
            const round = roundIndex + 1;
            const matchesInRound = matches.filter(match => match.round === round);
            
            return (
              <Card key={round} className="p-4">
                <h4 className="text-sm font-medium text-gray-600 mb-3">
                  Round {round}
                </h4>
                <div className="space-y-2">
                  {matchesInRound.map(renderMatch)}
                </div>
              </Card>
            );
          })}
        </div>
      </div>
    );
  };

  const renderGroupKnockout = () => {
    const groups = bracket.structure.groups || 4;
    
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Users className="w-5 h-5" />
            {bracket.name} - Group Stage + Knockout
          </h3>
          <div className="flex items-center gap-2">
            <Badge variant="outline">
              <Users className="w-3 h-3 mr-1" />
              {bracket.structure.total_participants} participants
            </Badge>
            <Badge variant="outline">
              {groups} groups
            </Badge>
          </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {Array.from({ length: groups }, (_, groupIndex) => {
            const groupNumber = groupIndex + 1;
            const groupMatches = matches.filter(match => 
              match.round <= bracket.structure.participants_per_group - 1
            );
            
            return (
              <Card key={groupNumber} className="p-4">
                <h4 className="text-sm font-medium text-gray-600 mb-3">
                  Group {groupNumber}
                </h4>
                <div className="space-y-2">
                  {groupMatches.slice(groupIndex * 3, (groupIndex + 1) * 3).map(renderMatch)}
                </div>
              </Card>
            );
          })}
        </div>
      </div>
    );
  };

  const renderDoubleElimination = () => {
    const winnersMatches = matches.filter(match => !match.bracket_type || match.bracket_type === 'winners');
    const losersMatches = matches.filter(match => match.bracket_type === 'losers');
    
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Trophy className="w-5 h-5" />
            {bracket.name} - Double Elimination
          </h3>
          <div className="flex items-center gap-2">
            <Badge variant="outline">
              <Users className="w-3 h-3 mr-1" />
              {bracket.structure.total_participants} participants
            </Badge>
            <Badge variant="outline">
              {bracket.structure.total_matches} matches
            </Badge>
          </div>
        </div>
        
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <div>
            <h4 className="text-md font-medium mb-3 flex items-center gap-2">
              <Trophy className="w-4 h-4" />
              Winners Bracket
            </h4>
            <div className="space-y-3">
              {winnersMatches.map(renderMatch)}
            </div>
          </div>
          <div>
            <h4 className="text-md font-medium mb-3 flex items-center gap-2">
              <Users className="w-4 h-4" />
              Losers Bracket
            </h4>
            <div className="space-y-3">
              {losersMatches.map(renderMatch)}
            </div>
          </div>
        </div>
      </div>
    );
  };

  const renderStatistics = () => {
    const completedMatches = matches.filter(m => m.status === 'completed');
    const inProgressMatches = matches.filter(m => m.status === 'in_progress');
    const scheduledMatches = matches.filter(m => m.status === 'scheduled');
    
    // Calculate participant statistics
    const participantStats = new Map<number, { wins: number; losses: number; matches: number }>();
    
    completedMatches.forEach(match => {
      const p1Id = match.player1_id || match.team1_id;
      const p2Id = match.player2_id || match.team2_id;
      
      if (p1Id) {
        const stats = participantStats.get(p1Id) || { wins: 0, losses: 0, matches: 0 };
        stats.matches++;
        if (match.winner_id === p1Id || match.winning_team_id === p1Id) {
          stats.wins++;
        } else {
          stats.losses++;
        }
        participantStats.set(p1Id, stats);
      }
      
      if (p2Id) {
        const stats = participantStats.get(p2Id) || { wins: 0, losses: 0, matches: 0 };
        stats.matches++;
        if (match.winner_id === p2Id || match.winning_team_id === p2Id) {
          stats.wins++;
        } else {
          stats.losses++;
        }
        participantStats.set(p2Id, stats);
      }
    });

    return (
      <div className="space-y-6">
        <h3 className="text-lg font-semibold flex items-center gap-2">
          <BarChart3 className="w-5 h-5" />
          Tournament Statistics
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Card className="p-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{completedMatches.length}</div>
              <div className="text-sm text-gray-600">Completed Matches</div>
            </div>
          </Card>
          <Card className="p-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">{inProgressMatches.length}</div>
              <div className="text-sm text-gray-600">In Progress</div>
            </div>
          </Card>
          <Card className="p-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-gray-600">{scheduledMatches.length}</div>
              <div className="text-sm text-gray-600">Scheduled</div>
            </div>
          </Card>
        </div>
        
        <Card className="p-4">
          <h4 className="text-md font-medium mb-3">Participant Standings</h4>
          <div className="space-y-2">
            {Array.from(participantStats.entries()).map(([participantId, stats]) => {
              const participant = getParticipant(participantId);
              const winRate = stats.matches > 0 ? (stats.wins / stats.matches * 100).toFixed(1) : '0';
              
              return (
                <div key={participantId} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                  <span className="font-medium">{getParticipantName(participant)}</span>
                  <div className="flex items-center gap-4 text-sm">
                    <span>{stats.wins}W - {stats.losses}L</span>
                    <span className="font-medium">{winRate}%</span>
                  </div>
                </div>
              );
            })}
          </div>
        </Card>
      </div>
    );
  };

  const renderBracket = () => {
    switch (bracket.structure.type) {
      case 'single_elimination':
        return renderSingleElimination();
      case 'double_elimination':
        return renderDoubleElimination();
      case 'round_robin':
        return renderRoundRobin();
      case 'group_knockout':
        return renderGroupKnockout();
      default:
        return renderSingleElimination();
    }
  };

  return (
    <div className="space-y-6">
      {/* Header with controls */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-xl font-semibold">{bracket.name}</h2>
          <p className="text-sm text-gray-600">
            {bracket.structure?.type ? bracket.structure.type.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()) : 'Tournament'} Tournament
          </p>
        </div>
        
        <div className="flex items-center gap-2">
          <Button
            variant={viewMode === 'bracket' ? 'default' : 'outline'}
            size="sm"
            onClick={() => setViewMode('bracket')}
          >
            <Settings className="w-4 h-4 mr-1" />
            Bracket
          </Button>
          <Button
            variant={viewMode === 'stats' ? 'default' : 'outline'}
            size="sm"
            onClick={() => setViewMode('stats')}
          >
            <BarChart3 className="w-4 h-4 mr-1" />
            Stats
          </Button>
        </div>
      </div>
      
      {/* Main content */}
      {viewMode === 'bracket' ? renderBracket() : renderStatistics()}
      
      {/* Footer controls */}
      {showControls && bracket.structure.type === 'single_elimination' && (
        <div className="flex justify-center gap-4">
          <Button
            variant="outline"
            onClick={() => {
              const nextRound = currentRound + 1;
              if (nextRound <= bracket.structure.rounds) {
                setCurrentRound(nextRound);
                onAdvanceRound?.(nextRound);
              }
            }}
            disabled={currentRound >= bracket.structure.rounds}
          >
            Advance to Next Round
          </Button>
        </div>
      )}
    </div>
  );
};

export default EnhancedBracketVisualization;
