import React from 'react';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';

interface Match {
  id: number;
  name: string;
  player1_id?: number;
  player2_id?: number;
  team1_id?: number;
  team2_id?: number;
  score1?: number;
  score2?: number;
  status: 'scheduled' | 'in_progress' | 'completed' | 'cancelled';
  scheduled_at?: string;
}

interface BracketVisualizationProps {
  bracket: {
    id: number;
    name: string;
    type: 'single_elimination' | 'double_elimination' | 'round_robin';
    tournament_id: number;
  };
  matches?: Match[];
  players?: Array<{ id: number; display_name: string }>;
  teams?: Array<{ id: number; name: string }>;
}

const BracketVisualization: React.FC<BracketVisualizationProps> = ({
  bracket,
  matches = [],
  players = [],
  teams = []
}) => {
  const getParticipantName = (participantId: number, isPlayer: boolean = true) => {
    if (isPlayer) {
      return players.find(p => p.id === participantId)?.display_name || `Player ${participantId}`;
    } else {
      return teams.find(t => t.id === participantId)?.name || `Team ${participantId}`;
    }
  };

  const getMatchStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200';
      case 'in_progress':
        return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200';
      case 'cancelled':
        return 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200';
      default:
        return 'bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200';
    }
  };

  const renderMatch = (match: Match) => {
    const participant1 = match.player1_id ? getParticipantName(match.player1_id, true) : 
                        match.team1_id ? getParticipantName(match.team1_id, false) : 'TBD';
    const participant2 = match.player2_id ? getParticipantName(match.player2_id, true) : 
                        match.team2_id ? getParticipantName(match.team2_id, false) : 'TBD';

    return (
      <Card key={match.id} className="p-3 border-l-4 border-l-green-500 hover:shadow-md transition-shadow">
        <div className="flex justify-between items-start mb-2">
          <h4 className="font-medium text-sm">{match.name}</h4>
          <Badge className={`text-xs ${getMatchStatusColor(match.status)}`}>
            {match.status.replace('_', ' ')}
          </Badge>
        </div>
        
        <div className="space-y-1">
          <div className="flex justify-between items-center">
            <span className="text-sm">{participant1}</span>
            <span className="text-sm font-medium">{match.score1 || 0}</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-sm">{participant2}</span>
            <span className="text-sm font-medium">{match.score2 || 0}</span>
          </div>
        </div>
        
        {match.scheduled_at && (
          <div className="mt-2 text-xs text-gray-500">
            {new Date(match.scheduled_at).toLocaleDateString()} {new Date(match.scheduled_at).toLocaleTimeString()}
          </div>
        )}
      </Card>
    );
  };

  const renderSingleElimination = () => {
    const rounds = Math.ceil(Math.log2(matches.length || 1));
    const maxMatches = Math.pow(2, rounds);
    
    return (
      <div className="space-y-6">
        <h3 className="text-lg font-semibold">Single Elimination Bracket</h3>
        <div className="flex gap-8 overflow-x-auto pb-4">
          {Array.from({ length: rounds }, (_, roundIndex) => (
            <div key={roundIndex} className="flex flex-col gap-4 min-w-[200px]">
              <h4 className="text-sm font-medium text-gray-600">
                Round {roundIndex + 1}
                {roundIndex === 0 && ` (${maxMatches} participants)`}
              </h4>
              <div className="space-y-3">
                {matches
                  .filter(match => {
                    // Simple logic to group matches by round
                    const matchIndex = matches.indexOf(match);
                    const matchesPerRound = maxMatches / Math.pow(2, roundIndex);
                    const roundStart = roundIndex * matchesPerRound;
                    return matchIndex >= roundStart && matchIndex < roundStart + matchesPerRound;
                  })
                  .map(renderMatch)}
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderDoubleElimination = () => {
    return (
      <div className="space-y-6">
        <h3 className="text-lg font-semibold">Double Elimination Bracket</h3>
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <div>
            <h4 className="text-md font-medium mb-3">Winners Bracket</h4>
            <div className="space-y-3">
              {matches.filter(match => match.name.toLowerCase().includes('winner')).map(renderMatch)}
            </div>
          </div>
          <div>
            <h4 className="text-md font-medium mb-3">Losers Bracket</h4>
            <div className="space-y-3">
              {matches.filter(match => match.name.toLowerCase().includes('loser')).map(renderMatch)}
            </div>
          </div>
        </div>
      </div>
    );
  };

  const renderRoundRobin = () => {
    // Group matches by round for round robin
    const matchesByRound = matches.reduce((acc, match) => {
      const roundMatch = match.name.match(/Round (\d+)/);
      const round = roundMatch ? parseInt(roundMatch[1]) : 1;
      if (!acc[round]) acc[round] = [];
      acc[round].push(match);
      return acc;
    }, {} as Record<number, Match[]>);

    return (
      <div className="space-y-6">
        <h3 className="text-lg font-semibold">Round Robin Tournament</h3>
        <div className="space-y-6">
          {Object.entries(matchesByRound).map(([round, roundMatches]) => (
            <div key={round}>
              <h4 className="text-md font-medium mb-3">Round {round}</h4>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                {roundMatches.map(renderMatch)}
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderBracket = () => {
    switch (bracket.type) {
      case 'single_elimination':
        return renderSingleElimination();
      case 'double_elimination':
        return renderDoubleElimination();
      case 'round_robin':
        return renderRoundRobin();
      default:
        return (
          <div className="text-center text-gray-500 py-8">
            <p>Bracket visualization not available for this type.</p>
          </div>
        );
    }
  };

  return (
    <Card className="p-6">
      <div className="mb-4">
        <h2 className="text-xl font-semibold mb-2">Bracket Visualization</h2>
        <p className="text-sm text-gray-600">
          {bracket.type.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())} Tournament
        </p>
      </div>
      
      {matches.length > 0 ? (
        renderBracket()
      ) : (
        <div className="text-center text-gray-500 py-8">
          <p>No matches found for this bracket.</p>
          <p className="text-sm mt-2">Create matches to see the bracket visualization.</p>
        </div>
      )}
    </Card>
  );
};

export default BracketVisualization; 