import React from 'react';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Star, Trophy, Award, Target, TrendingUp, Users, Zap } from 'lucide-react';

interface Achievement {
  name: string;
  data?: {
    description?: string;
    difficulty?: string;
    icon?: string;
    category?: string;
    points?: number;
  };
  earned_at: string;
}

interface AchievementCardProps {
  achievement: Achievement;
  size?: 'sm' | 'md' | 'lg';
}

const getAchievementIcon = (name: string, category?: string) => {
  const lowerName = name.toLowerCase();
  const lowerCategory = category?.toLowerCase();
  
  if (lowerName.includes('winner') || lowerName.includes('champion') || lowerName.includes('victory')) {
    return <Trophy className="w-5 h-5 text-yellow-600" />;
  }
  if (lowerName.includes('streak') || lowerName.includes('consecutive')) {
    return <TrendingUp className="w-5 h-5 text-green-600" />;
  }
  if (lowerName.includes('first') || lowerName.includes('debut')) {
    return <Star className="w-5 h-5 text-blue-600" />;
  }
  if (lowerName.includes('social') || lowerName.includes('community')) {
    return <Users className="w-5 h-5 text-purple-600" />;
  }
  if (lowerName.includes('speed') || lowerName.includes('fast')) {
    return <Zap className="w-5 h-5 text-orange-600" />;
  }
  if (lowerName.includes('accuracy') || lowerName.includes('precision')) {
    return <Target className="w-5 h-5 text-red-600" />;
  }
  
  return <Award className="w-5 h-5 text-gray-600" />;
};

const getDifficultyColor = (difficulty?: string) => {
  switch (difficulty?.toLowerCase()) {
    case 'easy': return 'bg-green-100 text-green-800';
    case 'medium': return 'bg-yellow-100 text-yellow-800';
    case 'hard': return 'bg-orange-100 text-orange-800';
    case 'legendary': return 'bg-purple-100 text-purple-800';
    default: return 'bg-gray-100 text-gray-800';
  }
};

const getSizeClasses = (size: string) => {
  switch (size) {
    case 'sm':
      return 'p-3 text-sm';
    case 'lg':
      return 'p-6 text-lg';
    default:
      return 'p-4 text-base';
  }
};

const AchievementCard: React.FC<AchievementCardProps> = ({ achievement, size = 'md' }) => {
  const sizeClasses = getSizeClasses(size);
  
  return (
    <Card className={`${sizeClasses} border-l-4 border-l-green-500 hover:shadow-md transition-all duration-200 hover:scale-105 bg-gradient-to-br from-yellow-50 to-orange-50`}>
      <div className="flex items-start gap-3">
        <div className="flex-shrink-0">
          {getAchievementIcon(achievement.name, achievement.data?.category)}
        </div>
        
        <div className="flex-1 min-w-0">
          <div className="flex items-center justify-between mb-2">
            <h4 className="font-semibold text-gray-900 truncate">
              {achievement.name}
            </h4>
            {achievement.data?.points && (
              <Badge variant="secondary" className="text-xs">
                +{achievement.data.points} pts
              </Badge>
            )}
          </div>
          
          {achievement.data?.description && (
            <p className="text-sm text-gray-600 mb-3 line-clamp-2">
              {achievement.data.description}
            </p>
          )}
          
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              {achievement.data?.difficulty && (
                <Badge className={`text-xs ${getDifficultyColor(achievement.data.difficulty)}`}>
                  {achievement.data.difficulty}
                </Badge>
              )}
              {achievement.data?.category && (
                <Badge variant="outline" className="text-xs">
                  {achievement.data.category}
                </Badge>
              )}
            </div>
            
            <div className="text-xs text-gray-500">
              {new Date(achievement.earned_at).toLocaleDateString()}
            </div>
          </div>
        </div>
      </div>
    </Card>
  );
};

export default AchievementCard; 