import React from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { router } from '@inertiajs/react';

interface TournamentsTableProps {
  tournaments: any;
  selectedTournaments: number[];
  onTournamentSelect: (tournamentId: number, checked: boolean) => void;
  onSelectAll: (checked: boolean) => void;
  onEdit: (tournament: any) => void;
  onDelete: (tournament: any) => void;
  onPageChange: (page: number) => void;
}

const TournamentsTable: React.FC<TournamentsTableProps> = ({
  tournaments,
  selectedTournaments,
  onTournamentSelect,
  onSelectAll,
  onEdit,
  onDelete,
  onPageChange,
}) => {
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'completed': return 'bg-blue-100 text-blue-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      case 'draft': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'single_elimination': return 'bg-purple-100 text-purple-800';
      case 'double_elimination': return 'bg-orange-100 text-orange-800';
      case 'round_robin': return 'bg-cyan-100 text-cyan-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getFormatColor = (format: string) => {
    switch (format) {
      case 'individual': return 'bg-blue-100 text-blue-800';
      case 'team': return 'bg-green-100 text-green-800';
      case 'mixed': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'N/A';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  const formatCurrency = (amount: number, currency: string = 'KES') => {
    if (!amount) return 'Free';
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency,
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  if (tournaments.data.length === 0) {
    return (
      <Card className="p-6">
        <div className="text-center text-gray-500 dark:text-gray-400">
          <div className="mb-4">
            <svg className="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
            </svg>
          </div>
          <h3 className="text-lg font-medium">No tournaments found</h3>
          <p className="mt-1">Try adjusting your search or filter criteria.</p>
        </div>
      </Card>
    );
  }

  return (
    <Card className="overflow-hidden">
      {/* Fixed Header */}
      <div className="bg-muted/50 border-b sticky top-0 z-10">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead>
              <tr className="border-b">
                <th className="text-left p-4 w-12">
                  <input
                    type="checkbox"
                    checked={selectedTournaments.length === tournaments.data.length}
                    onChange={(e) => onSelectAll(e.target.checked)}
                    className="rounded border-gray-300"
                  />
                </th>
                <th className="text-left p-4 font-medium">Tournament</th>
                <th className="text-left p-4 font-medium">Type & Format</th>
                <th className="text-left p-4 font-medium">Status</th>
                <th className="text-left p-4 font-medium">Participants</th>
                <th className="text-left p-4 font-medium">Schedule</th>
                <th className="text-left p-4 font-medium">Prize Pool</th>
                <th className="text-left p-4 font-medium w-32">Actions</th>
              </tr>
            </thead>
          </table>
        </div>
      </div>

      {/* Scrollable Body */}
      <div className="overflow-x-auto max-h-[600px] overflow-y-auto">
        <table className="w-full min-w-[1000px]">
          <tbody>
            {tournaments.data.map((tournament: any) => (
              <tr key={tournament.id} className="border-b hover:bg-muted/25 transition-colors">
                <td className="p-4 w-12">
                  <input
                    type="checkbox"
                    checked={selectedTournaments.includes(tournament.id)}
                    onChange={(e) => onTournamentSelect(tournament.id, e.target.checked)}
                    className="rounded border-gray-300"
                  />
                </td>
                <td className="p-4">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 rounded-lg bg-gradient-to-br from-primary/20 to-primary/40 flex items-center justify-center">
                      {tournament.logo_image ? (
                        <img src={tournament.logo_image} alt={tournament.name} className="w-8 h-8 rounded" />
                      ) : (
                        <span className="text-primary font-bold text-lg">
                          {tournament.name.charAt(0).toUpperCase()}
                        </span>
                      )}
                    </div>
                    <div>
                      <div className="font-medium">{tournament.name}</div>
                      <div className="text-sm text-muted-foreground">
                        {tournament.organizer && `by ${tournament.organizer}`}
                      </div>
                      {tournament.location && (
                        <div className="text-xs text-muted-foreground">{tournament.location}</div>
                      )}
                    </div>
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <Badge className={`text-xs ${getTypeColor(tournament.type)}`}>
                      {tournament.type?.replace('_', ' ').toUpperCase()}
                    </Badge>
                    <div className="text-sm">
                      <Badge variant="outline" className={`text-xs ${getFormatColor(tournament.format)}`}>
                        {tournament.format?.toUpperCase()}
                      </Badge>
                    </div>
                    {tournament.game_title && (
                      <div className="text-xs text-muted-foreground">{tournament.game_title}</div>
                    )}
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <Badge className={`text-xs ${getStatusColor(tournament.status)}`}>
                      {tournament.status}
                    </Badge>
                    <div className="text-xs text-muted-foreground">
                      {tournament.visibility}
                    </div>
                    {tournament.is_featured && (
                      <Badge variant="outline" className="text-xs bg-yellow-100 text-yellow-800">
                        Featured
                      </Badge>
                    )}
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <div className="font-medium">
                      {tournament.total_participants || 0}
                      {tournament.max_participants && ` / ${tournament.max_participants}`}
                    </div>
                    <div className="text-xs text-muted-foreground">
                      {tournament.registration_status === 'open' ? 'Open' : 
                       tournament.registration_status === 'closed' ? 'Closed' : 'Full'}
                    </div>
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <div className="text-sm font-medium">
                      {formatDate(tournament.start_date)}
                    </div>
                    <div className="text-xs text-muted-foreground">
                      to {formatDate(tournament.end_date)}
                    </div>
                    {tournament.registration_end && (
                      <div className="text-xs text-muted-foreground">
                        Reg: {formatDate(tournament.registration_end)}
                      </div>
                    )}
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <div className="font-medium">
                      {formatCurrency(tournament.prize_pool, tournament.currency)}
                    </div>
                    {tournament.entry_fee > 0 && (
                      <div className="text-xs text-muted-foreground">
                        Entry: {formatCurrency(tournament.entry_fee, tournament.currency)}
                      </div>
                    )}
                  </div>
                </td>
                <td className="p-4 w-32">
                  <div className="flex gap-1">
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => router.get(route('admin.tournaments.show', { tournament: tournament.id }))}
                      className="text-xs px-2 py-1"
                    >
                      View
                    </Button>
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => onEdit(tournament)}
                      className="text-xs px-2 py-1"
                    >
                      Edit
                    </Button>
                    <Button 
                      size="sm" 
                      variant="destructive"
                      onClick={() => onDelete(tournament)}
                      className="text-xs px-2 py-1"
                    >
                      Del
                    </Button>
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {/* Fixed Footer */}
      <div className="bg-muted/50 border-t p-4 sticky bottom-0 z-10">
        <div className="flex justify-between items-center">
          <div className="text-sm text-muted-foreground">
            Showing {tournaments.from} to {tournaments.to} of {tournaments.total} results
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              size="sm"
              disabled={tournaments.current_page === 1}
              onClick={() => onPageChange(tournaments.current_page - 1)}
            >
              Previous
            </Button>
            <div className="flex items-center gap-1">
              {Array.from({ length: Math.min(5, tournaments.last_page) }, (_, i) => {
                const page = tournaments.current_page - 2 + i;
                if (page < 1 || page > tournaments.last_page) return null;
                return (
                  <Button
                    key={page}
                    variant={page === tournaments.current_page ? "default" : "outline"}
                    size="sm"
                    onClick={() => onPageChange(page)}
                    className="w-8 h-8 p-0"
                  >
                    {page}
                  </Button>
                );
              })}
            </div>
            <Button
              variant="outline"
              size="sm"
              disabled={tournaments.current_page === tournaments.last_page}
              onClick={() => onPageChange(tournaments.current_page + 1)}
            >
              Next
            </Button>
          </div>
        </div>
      </div>
    </Card>
  );
};

export default TournamentsTable; 