import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { Settings } from 'lucide-react';

interface TournamentSetupStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
}

const TournamentSetupStep: React.FC<TournamentSetupStepProps> = ({
  form,
  errors,
  onInputChange,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
        
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="type">Tournament Type *</Label>
            <Select value={form.type} onValueChange={(value) => onInputChange('type', value)}>
              <SelectTrigger className={errors.type ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select tournament type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="knockout">Knockout Tournament</SelectItem>
                <SelectItem value="elimination">Knockout Tournament (Elimination)</SelectItem>
                <SelectItem value="round-robin">League Tournament (Round-Robin)</SelectItem>
                <SelectItem value="group-knockout">Group + Knockout Tournament (Mixed Format)</SelectItem>
                <SelectItem value="double-elimination">Double Elimination Tournament</SelectItem>
              </SelectContent>
            </Select>
            {errors.type && <p className="text-red-500 text-sm mt-1">{errors.type}</p>}
          </div>

          <div>
            <Label htmlFor="format">Format *</Label>
            <Select value={form.format} onValueChange={(value) => onInputChange('format', value)}>
              <SelectTrigger className={errors.format ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select format" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="individual">Individual</SelectItem>
                <SelectItem value="team">Team</SelectItem>
                <SelectItem value="mixed">Mixed</SelectItem>
              </SelectContent>
            </Select>
            {errors.format && <p className="text-red-500 text-sm mt-1">{errors.format}</p>}
          </div>

          <div>
            <Label htmlFor="status">Status *</Label>
            <Select value={form.status} onValueChange={(value) => onInputChange('status', value)}>
              <SelectTrigger className={errors.status ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="draft">Draft</SelectItem>
                <SelectItem value="active">Active</SelectItem>
                <SelectItem value="completed">Completed</SelectItem>
                <SelectItem value="cancelled">Cancelled</SelectItem>
              </SelectContent>
            </Select>
            {errors.status && <p className="text-red-500 text-sm mt-1">{errors.status}</p>}
          </div>

          <div>
            <Label htmlFor="registration_status">Registration Status *</Label>
            <Select value={form.registration_status} onValueChange={(value) => onInputChange('registration_status', value)}>
              <SelectTrigger className={errors.registration_status ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select registration status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="open">Open</SelectItem>
                <SelectItem value="closed">Closed</SelectItem>
                <SelectItem value="full">Full</SelectItem>
              </SelectContent>
            </Select>
            {errors.registration_status && <p className="text-red-500 text-sm mt-1">{errors.registration_status}</p>}
          </div>

          <div>
            <Label htmlFor="game_title">Game Title</Label>
            <Input
              id="game_title"
              value={form.game_title}
              onChange={(e) => onInputChange('game_title', e.target.value)}
              placeholder="e.g., League of Legends"
            />
          </div>

          <div>
            <Label htmlFor="game_version">Game Version</Label>
            <Input
              id="game_version"
              value={form.game_version}
              onChange={(e) => onInputChange('game_version', e.target.value)}
              placeholder="e.g., 13.24"
            />
          </div>

          {form.format === 'team' && (
            <div>
              <Label htmlFor="team_size">Team Size</Label>
              <Input
                id="team_size"
                type="number"
                min="1"
                max="50"
                value={form.team_size}
                onChange={(e) => onInputChange('team_size', parseInt(e.target.value) || '')}
                placeholder="e.g., 5"
              />
            </div>
          )}

          <div>
            <Label htmlFor="rounds">Number of Rounds</Label>
            <Input
              id="rounds"
              type="number"
              min="1"
              max="100"
              value={form.rounds}
              onChange={(e) => onInputChange('rounds', parseInt(e.target.value) || '')}
              placeholder="e.g., 4"
            />
          </div>

          <div>
            <Label htmlFor="matches_per_round">Matches per Round</Label>
            <Input
              id="matches_per_round"
              type="number"
              min="1"
              max="1000"
              value={form.matches_per_round}
              onChange={(e) => onInputChange('matches_per_round', parseInt(e.target.value) || '')}
              placeholder="e.g., 8"
            />
          </div>
        </div>
      </Card>

      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4">Tournament Settings</h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between space-x-3">
            <div className="space-y-1">
              <Label htmlFor="auto_accept_registrations" className="text-sm font-medium">
                Auto-accept Registrations
              </Label>
              <p className="text-sm text-muted-foreground">
                Automatically accept player registrations without manual approval
              </p>
            </div>
            <Switch
              id="auto_accept_registrations"
              checked={form.auto_accept_registrations}
              onCheckedChange={(checked) => onInputChange('auto_accept_registrations', checked)}
            />
          </div>

          <div className="flex items-center justify-between space-x-3">
            <div className="space-y-1">
              <Label htmlFor="allow_spectators" className="text-sm font-medium">
                Allow Spectators
              </Label>
              <p className="text-sm text-muted-foreground">
                Allow non-participants to watch tournament matches
              </p>
            </div>
            <Switch
              id="allow_spectators"
              checked={form.allow_spectators}
              onCheckedChange={(checked) => onInputChange('allow_spectators', checked)}
            />
          </div>

          <div className="flex items-center justify-between space-x-3">
            <div className="space-y-1">
              <Label htmlFor="require_approval" className="text-sm font-medium">
                Require Approval
              </Label>
              <p className="text-sm text-muted-foreground">
                Manually approve each participant before they can join
              </p>
            </div>
            <Switch
              id="require_approval"
              checked={form.require_approval}
              onCheckedChange={(checked) => onInputChange('require_approval', checked)}
            />
          </div>

          <div className="flex items-center justify-between space-x-3">
            <div className="space-y-1">
              <Label htmlFor="is_featured" className="text-sm font-medium">
                Featured Tournament
              </Label>
              <p className="text-sm text-muted-foreground">
                Display this tournament prominently on the platform
              </p>
            </div>
            <Switch
              id="is_featured"
              checked={form.is_featured}
              onCheckedChange={(checked) => onInputChange('is_featured', checked)}
            />
          </div>
        </div>
      </Card>
    </div>
  );
};

export default TournamentSetupStep; 