import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Calendar, Users, Info, Trophy } from 'lucide-react';

interface ScheduleParticipantsStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
}

const ScheduleParticipantsStep: React.FC<ScheduleParticipantsStepProps> = ({
  form,
  errors,
  onInputChange,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
        
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="start_date">Start Date & Time *</Label>
            <Input
              id="start_date"
              type="datetime-local"
              value={form.start_date}
              onChange={(e) => onInputChange('start_date', e.target.value)}
              className={errors.start_date ? 'border-red-500' : ''}
            />
            {errors.start_date && <p className="text-red-500 text-sm mt-1">{errors.start_date}</p>}
          </div>

          <div>
            <Label htmlFor="end_date">End Date & Time *</Label>
            <Input
              id="end_date"
              type="datetime-local"
              value={form.end_date}
              onChange={(e) => onInputChange('end_date', e.target.value)}
              className={errors.end_date ? 'border-red-500' : ''}
            />
            {errors.end_date && <p className="text-red-500 text-sm mt-1">{errors.end_date}</p>}
          </div>

          <div>
            <Label htmlFor="registration_start">Registration Start</Label>
            <Input
              id="registration_start"
              type="datetime-local"
              value={form.registration_start}
              onChange={(e) => onInputChange('registration_start', e.target.value)}
              className={errors.registration_start ? 'border-red-500' : ''}
            />
            {errors.registration_start && <p className="text-red-500 text-sm mt-1">{errors.registration_start}</p>}
          </div>

          <div>
            <Label htmlFor="registration_end">Registration End</Label>
            <Input
              id="registration_end"
              type="datetime-local"
              value={form.registration_end}
              onChange={(e) => onInputChange('registration_end', e.target.value)}
              className={errors.registration_end ? 'border-red-500' : ''}
            />
            {errors.registration_end && <p className="text-red-500 text-sm mt-1">{errors.registration_end}</p>}
          </div>
        </div>
      </Card>

      {/* Number of Teams */}
      <Card className="p-6">
        <div className="mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Trophy className="w-5 h-5" />
            Tournament Capacity
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            Define how many teams can participate in this tournament
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="min_participants">
              Minimum Number of Teams *
              <span className="text-xs text-gray-500 font-normal ml-2">
                (Required to start tournament)
              </span>
            </Label>
            <Input
              id="min_participants"
              type="number"
              min="2"
              max="1000"
              value={form.min_participants}
              onChange={(e) => onInputChange('min_participants', parseInt(e.target.value) || '')}
              placeholder="8"
              className={errors.min_participants ? 'border-red-500' : ''}
            />
            {errors.min_participants && <p className="text-red-500 text-sm mt-1">{errors.min_participants}</p>}
            <p className="text-xs text-gray-500 mt-1">
              💡 Minimum teams needed to start the tournament
            </p>
          </div>

          <div>
            <Label htmlFor="max_participants">
              Maximum Number of Teams *
              <span className="text-xs text-gray-500 font-normal ml-2">
                (Tournament capacity)
              </span>
            </Label>
            <Input
              id="max_participants"
              type="number"
              min={form.min_participants || "2"}
              max="1000"
              value={form.max_participants}
              onChange={(e) => onInputChange('max_participants', parseInt(e.target.value) || '')}
              placeholder="64"
              className={errors.max_participants ? 'border-red-500' : ''}
            />
            {errors.max_participants && <p className="text-red-500 text-sm mt-1">{errors.max_participants}</p>}
            <p className="text-xs text-gray-500 mt-1">
              💡 Must be ≥ minimum teams. Maximum allowed in tournament
            </p>
          </div>
        </div>
      </Card>

      {/* Players per Team */}
      <Card className="p-6">
        <div className="mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Users className="w-5 h-5" />
            Team Size Requirements
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            Define how many players each team must have for registration
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="min_players_per_team">
              Minimum Players per Team *
              <span className="text-xs text-gray-500 font-normal ml-2">
                (Required for each team)
              </span>
            </Label>
            <Input
              id="min_players_per_team"
              type="number"
              min="1"
              max="100"
              value={form.min_players_per_team}
              onChange={(e) => onInputChange('min_players_per_team', parseInt(e.target.value) || '')}
              placeholder="11"
              className={errors.min_players_per_team ? 'border-red-500' : ''}
            />
            {errors.min_players_per_team && <p className="text-red-500 text-sm mt-1">{errors.min_players_per_team}</p>}
            <p className="text-xs text-gray-500 mt-1">
              💡 Football standard: 11 players minimum (1 GK + 10 field players)
            </p>
          </div>

          <div>
            <Label htmlFor="max_players_per_team">
              Maximum Players per Team *
              <span className="text-xs text-gray-500 font-normal ml-2">
                (Squad roster limit)
              </span>
            </Label>
            <Input
              id="max_players_per_team"
              type="number"
              min={form.min_players_per_team || "1"}
              max="100"
              value={form.max_players_per_team}
              onChange={(e) => onInputChange('max_players_per_team', parseInt(e.target.value) || '')}
              placeholder="23"
              className={errors.max_players_per_team ? 'border-red-500' : ''}
            />
            {errors.max_players_per_team && <p className="text-red-500 text-sm mt-1">{errors.max_players_per_team}</p>}
            <p className="text-xs text-gray-500 mt-1">
              💡 Must be ≥ minimum. Football standard: 23 players (11 starting + 12 subs)
            </p>
          </div>

          <div>
            <Label htmlFor="entry_fee">Entry Fee (KES)</Label>
            <div className="relative">
              <span className="absolute left-0 top-1/2 transform -translate-y-1/2 text-muted-foreground">
                KES 
              </span>
              <Input
                id="entry_fee"
                type="number"
                min="0"
                step="0.01"
                value={form.entry_fee}
                onChange={(e) => onInputChange('entry_fee', parseFloat(e.target.value) || '')}
                className={`pl-8 KES {errors.entry_fee ? 'border-red-500' : ''}`}
                placeholder="0.00"
              />
            </div>
            {errors.entry_fee && <p className="text-red-500 text-sm mt-1">{errors.entry_fee}</p>}
          </div>

          <div>
            <Label htmlFor="currency">Currency</Label>
            <Input
              id="currency"
              value={form.currency}
              onChange={(e) => onInputChange('currency', e.target.value.toUpperCase())}
              placeholder="KES"
              maxLength={3}
            />
          </div>

          <div>
            <Label htmlFor="prize_pool">Prize Pool (KES)</Label>
            <div className="relative">
              <span className="absolute left-0 top-1/2 transform -translate-y-1/2 text-muted-foreground">
                KES 
              </span>
              <Input
                id="prize_pool"
                type="number"
                min="0"
                step="0.01"
                value={form.prize_pool}
                onChange={(e) => onInputChange('prize_pool', parseFloat(e.target.value) || '')}
                className={`pl-8 KES {errors.prize_pool ? 'border-red-500' : ''}`}
                placeholder="0.00"
              />
            </div>
            {errors.prize_pool && <p className="text-red-500 text-sm mt-1">{errors.prize_pool}</p>}
          </div>
        </div>

        <div className="mt-4">
          <Label htmlFor="announcement">Announcement</Label>
          <textarea
            id="announcement"
            value={form.announcement}
            onChange={(e) => onInputChange('announcement', e.target.value)}
            className="w-full mt-1 px-3 py-2 border border-input rounded-md resize-none focus:outline-none focus:ring-2 focus:ring-ring"
            rows={3}
            placeholder="Any important announcements for participants..."
          />
        </div>
      </Card>

      {/* Category Settings */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h3 className="text-lg font-semibold">Team Categories</h3>
            <p className="text-sm text-muted-foreground">Configure different categories with separate registration fees</p>
          </div>
          <Switch
            checked={form.enable_categories || false}
            onCheckedChange={(checked) => {
              onInputChange('enable_categories', checked);
              if (!checked) {
                onInputChange('category_fees', null);
                onInputChange('available_categories', null);
              } else {
                // Initialize with default fees
                onInputChange('category_fees', {
                  men: 1000,
                  women: 800,
                  youth_male: 500,
                  youth_female: 500,
                  children: 300,
                });
                onInputChange('available_categories', ['men', 'women', 'youth_male', 'youth_female', 'children']);
              }
            }}
          />
        </div>

        {form.enable_categories && (
          <div className="space-y-4">
            <Alert>
              <Info className="h-4 w-4" />
              <AlertDescription>
                Enable multiple team categories with different fees. Teams will select their category during registration.
              </AlertDescription>
            </Alert>

            {/* Category Selection & Fees Combined */}
            <div>
              <Label className="text-base font-medium mb-3 block">Configure Categories & Fees</Label>
              <div className="space-y-3">
                {[
                  { id: 'men', name: 'Men', icon: '👨', desc: '18+ years, Male', defaultFee: 1000 },
                  { id: 'women', name: 'Women', icon: '👩', desc: '18+ years, Female', defaultFee: 800 },
                  { id: 'youth_male', name: 'Youth Male', icon: '🧑', desc: '13-17 years, Male', defaultFee: 500 },
                  { id: 'youth_female', name: 'Youth Female', icon: '👧', desc: '13-17 years, Female', defaultFee: 500 },
                  { id: 'children', name: 'Children', icon: '🧒', desc: 'Under 13, Any gender', defaultFee: 300 },
                ].map((category) => {
                  const isEnabled = form.available_categories?.includes(category.id) || false;
                  return (
                    <div
                      key={category.id}
                      className={`p-4 border rounded-md transition-colors ${
                        isEnabled ? 'bg-accent border-primary' : 'bg-gray-50'
                      }`}
                    >
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center gap-2">
                          <span className="text-xl">{category.icon}</span>
                          <div>
                            <div className="font-medium">{category.name}</div>
                            <p className="text-xs text-muted-foreground">{category.desc}</p>
                          </div>
                        </div>
                        <Switch
                          checked={isEnabled}
                          onCheckedChange={(checked) => {
                            const currentCategories = form.available_categories || [];
                            const newCategories = checked
                              ? [...currentCategories, category.id]
                              : currentCategories.filter((c: string) => c !== category.id);
                            onInputChange('available_categories', newCategories);
                          }}
                        />
                      </div>
                      
                      {isEnabled && (
                        <div>
                          <Label htmlFor={`fee_${category.id}`} className="text-sm">
                            Registration Fee ({form.currency || 'KES'})
                          </Label>
                          <div className="relative mt-1">
                            <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground text-sm">
                              {form.currency || 'KES'}
                            </span>
                            <Input
                              id={`fee_${category.id}`}
                              type="number"
                              min="0"
                              step="0.01"
                              value={form.category_fees?.[category.id] || category.defaultFee}
                              onChange={(e) => {
                                const newFees = { ...(form.category_fees || {}) };
                                newFees[category.id] = parseFloat(e.target.value) || 0;
                                onInputChange('category_fees', newFees);
                              }}
                              className="pl-16"
                              placeholder={category.defaultFee.toString()}
                            />
                          </div>
                          {errors?.[`category_fees.${category.id}`] && (
                            <p className="text-red-500 text-sm mt-1">{errors[`category_fees.${category.id}`]}</p>
                          )}
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                Enable categories and set their registration fees. Only enabled categories will be available during team registration.
              </p>
            </div>
          </div>
        )}

        {!form.enable_categories && (
          <p className="text-sm text-muted-foreground">
            Categories are disabled. All teams will use the default entry fee. Enable categories to configure different fees for men, women, youth, and children divisions.
          </p>
        )}
      </Card>
    </div>
  );
};

export default ScheduleParticipantsStep; 