import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { Button } from '@/components/ui/button';
import { Video, Image, Upload, X } from 'lucide-react';
import MediaManager from '@/components/MediaManager';

interface MediaSettingsStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
}

const MediaSettingsStep: React.FC<MediaSettingsStepProps> = ({
  form,
  errors,
  onInputChange,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
        
        
        <div className="space-y-4">
          <div className="flex items-center justify-between space-x-3">
            <div className="space-y-1">
              <Label htmlFor="is_streamed" className="text-sm font-medium">
                Enable Streaming
              </Label>
              <p className="text-sm text-muted-foreground">
                This tournament will be streamed live
              </p>
            </div>
            <Switch
              id="is_streamed"
              checked={form.is_streamed}
              onCheckedChange={(checked) => onInputChange('is_streamed', checked)}
            />
          </div>

          {form.is_streamed && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 ml-6">
              <div>
                <Label htmlFor="stream_url">Main Stream URL</Label>
                <Input
                  id="stream_url"
                  type="url"
                  value={form.stream_url}
                  onChange={(e) => onInputChange('stream_url', e.target.value)}
                  className={errors.stream_url ? 'border-red-500' : ''}
                  placeholder="https://stream-platform.com/tournament"
                />
                {errors.stream_url && <p className="text-red-500 text-sm mt-1">{errors.stream_url}</p>}
              </div>

              <div>
                <Label htmlFor="twitch_url">Twitch URL</Label>
                <Input
                  id="twitch_url"
                  type="url"
                  value={form.twitch_url}
                  onChange={(e) => onInputChange('twitch_url', e.target.value)}
                  className={errors.twitch_url ? 'border-red-500' : ''}
                  placeholder="https://twitch.tv/channel"
                />
                {errors.twitch_url && <p className="text-red-500 text-sm mt-1">{errors.twitch_url}</p>}
              </div>

              <div>
                <Label htmlFor="youtube_url">YouTube URL</Label>
                <Input
                  id="youtube_url"
                  type="url"
                  value={form.youtube_url}
                  onChange={(e) => onInputChange('youtube_url', e.target.value)}
                  className={errors.youtube_url ? 'border-red-500' : ''}
                  placeholder="https://youtube.com/watch?v=..."
                />
                {errors.youtube_url && <p className="text-red-500 text-sm mt-1">{errors.youtube_url}</p>}
              </div>
            </div>
          )}
        </div>
      </Card>

      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Image className="w-5 h-5" />
          Media & Branding
        </h3>
        
        <div className="space-y-6">
          {/* Banner Image */}
          <div>
            <Label className="text-sm font-medium mb-2 block">Banner Image</Label>
            <div className="space-y-3">
              {form.banner_image ? (
                <div className="relative">
                  <img
                    src={form.banner_image}
                    alt="Banner preview"
                    className="w-full h-32 object-cover rounded-lg border"
                  />
                  <Button
                    variant="destructive"
                    size="sm"
                    className="absolute top-2 right-2"
                    onClick={() => onInputChange('banner_image', '')}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                </div>
              ) : (
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <Image className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-500 mb-3">No banner image selected</p>
                  <MediaManager
                    trigger={
                      <Button variant="outline" size="sm">
                        <Upload className="h-4 w-4 mr-2" />
                        Select Banner Image
                      </Button>
                    }
                    onSelect={(file) => onInputChange('banner_image', file.url)}
                    acceptedTypes={['image/*']}
                    multiple={false}
                    collection="tournament-banners"
                  />
                </div>
              )}
            </div>
          </div>

          {/* Logo Image */}
          <div>
            <Label className="text-sm font-medium mb-2 block">Logo Image</Label>
            <div className="space-y-3">
              {form.logo_image ? (
                <div className="relative">
                  <img
                    src={form.logo_image}
                    alt="Logo preview"
                    className="w-24 h-24 object-contain rounded-lg border"
                  />
                  <Button
                    variant="destructive"
                    size="sm"
                    className="absolute top-2 right-2"
                    onClick={() => onInputChange('logo_image', '')}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                </div>
              ) : (
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <Image className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-500 mb-3">No logo image selected</p>
                  <MediaManager
                    trigger={
                      <Button variant="outline" size="sm">
                        <Upload className="h-4 w-4 mr-2" />
                        Select Logo Image
                      </Button>
                    }
                    onSelect={(file) => onInputChange('logo_image', file.url)}
                    acceptedTypes={['image/*']}
                    multiple={false}
                    collection="tournament-logos"
                  />
                </div>
              )}
            </div>
          </div>
        </div>
      </Card>

      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4">Summary</h3>
        <div className="space-y-3 text-sm">
          <div className="flex justify-between">
            <span className="text-muted-foreground">Tournament Name:</span>
            <span className="font-medium">{form.name || 'Not set'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Type:</span>
            <span className="font-medium capitalize">{form.type?.replace('_', ' ') || 'Not set'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Format:</span>
            <span className="font-medium capitalize">{form.format || 'Not set'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Status:</span>
            <span className="font-medium capitalize">{form.status || 'Not set'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Max Participants:</span>
            <span className="font-medium">{form.max_participants || 'Unlimited'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Entry Fee:</span>
            <span className="font-medium">
              {form.entry_fee ? `$${form.entry_fee} ${form.currency || 'KES'}` : 'Free'}
            </span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Prize Pool:</span>
            <span className="font-medium">
              {form.prize_pool ? `$${form.prize_pool} ${form.currency || 'KES'}` : 'None'}
            </span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Visibility:</span>
            <span className="font-medium capitalize">{form.visibility || 'Not set'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Streaming:</span>
            <span className="font-medium">{form.is_streamed ? 'Yes' : 'No'}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Featured:</span>
            <span className="font-medium">{form.is_featured ? 'Yes' : 'No'}</span>
          </div>
        </div>
      </Card>
    </div>
  );
};

export default MediaSettingsStep; 