import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from '@/components/ui/select';
import { 
  Users, 
  Trophy, 
  User,
  AlertCircle
} from 'lucide-react';

interface TeamFormProps {
  form: {
    name: string;
    description: string;
    tournament_id: string;
    captain_id: string;
    status: string;
  };
  setForm: (form: any) => void;
  errors: Record<string, string>;
  tournaments: any[];
  users: any[];
  isEditing?: boolean;
}

const TeamForm: React.FC<TeamFormProps> = ({ 
  form, 
  setForm, 
  errors, 
  tournaments,
  users,
  isEditing = false 
}) => {
  return (
    <div className="space-y-6">
      {/* Basic Information */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Users className="w-5 h-5" />
            Team Information
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="name">Team Name *</Label>
              <Input
                id="name"
                type="text"
                value={form.name}
                onChange={(e) => setForm({ ...form, name: e.target.value })}
                placeholder="Enter team name"
                className={errors.name ? 'border-red-500' : ''}
              />
              {errors.name && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.name}
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="status">Status *</Label>
              <Select 
                value={form.status} 
                onValueChange={(value) => setForm({ ...form, status: value })}
              >
                <SelectTrigger className={errors.status ? 'border-red-500' : ''}>
                  <SelectValue placeholder="Select status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="eliminated">Eliminated</SelectItem>
                  <SelectItem value="withdrawn">Withdrawn</SelectItem>
                </SelectContent>
              </Select>
              {errors.status && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.status}
                </p>
              )}
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              value={form.description}
              onChange={(e) => setForm({ ...form, description: e.target.value })}
              placeholder="Enter team description"
              rows={3}
              className={errors.description ? 'border-red-500' : ''}
            />
            {errors.description && (
              <p className="text-sm text-red-600 flex items-center gap-1">
                <AlertCircle className="w-4 h-4" />
                {errors.description}
              </p>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Tournament & Captain */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Trophy className="w-5 h-5" />
            Tournament & Leadership
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="tournament_id">Tournament *</Label>
              <Select 
                value={form.tournament_id} 
                onValueChange={(value) => setForm({ ...form, tournament_id: value })}
              >
                <SelectTrigger className={errors.tournament_id ? 'border-red-500' : ''}>
                  <SelectValue placeholder="Select tournament" />
                </SelectTrigger>
                <SelectContent>
                  {tournaments && tournaments.length > 0 ? (
                    tournaments.map((tournament) => (
                      <SelectItem key={tournament.id} value={String(tournament.id)}>
                        {tournament.name}
                      </SelectItem>
                    ))
                  ) : (
                    <SelectItem value="0" disabled>No tournaments available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.tournament_id && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.tournament_id}
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="captain_id">Team Captain *</Label>
              <Select 
                value={form.captain_id} 
                onValueChange={(value) => setForm({ ...form, captain_id: value })}
              >
                <SelectTrigger className={errors.captain_id ? 'border-red-500' : ''}>
                  <SelectValue placeholder="Select captain" />
                </SelectTrigger>
                <SelectContent>
                  {users && users.length > 0 ? (
                    users.map((user) => (
                      <SelectItem key={user.id} value={String(user.id)}>
                        {user.name} ({user.email})
                      </SelectItem>
                    ))
                  ) : (
                    <SelectItem value="0" disabled>No users available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.captain_id && (
                <p className="text-sm text-red-600 flex items-center gap-1">
                  <AlertCircle className="w-4 h-4" />
                  {errors.captain_id}
                </p>
              )}
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default TeamForm; 
