import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { CheckCircle, Circle, ArrowLeft, ArrowRight } from 'lucide-react';

interface Step {
  id: string;
  title: string;
  description: string;
  icon: React.ReactNode;
}

interface TeamFormStepsProps {
  steps: Step[];
  currentStep: number;
  onStepChange: (step: number) => void;
  onNext: () => void;
  onPrevious: () => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  canProceed: boolean;
  isLastStep: boolean;
  children?: React.ReactNode;
}

const TeamFormSteps: React.FC<TeamFormStepsProps> = ({
  steps,
  currentStep,
  onStepChange,
  onNext,
  onPrevious,
  onSubmit,
  isSubmitting,
  canProceed,
  isLastStep,
  children,
}) => {
  const progress = ((currentStep + 1) / steps.length) * 100;

  return (
    <div className="space-y-6">
      <Card className="p-4 sm:p-6">
        {/* Step Indicators */}
        <div className="mb-6">
          {/* Mobile Step Indicators */}
          <div className="block sm:hidden mb-4">
            <div className="flex items-center justify-between">
              {steps.map((step, index) => (
                <div key={step.id} className="flex flex-col items-center">
                  <button
                    onClick={() => onStepChange(index)}
                    className={`flex items-center justify-center w-8 h-8 rounded-full border-2 transition-colors ${
                      index <= currentStep
                        ? 'border-primary bg-primary text-primary-foreground'
                        : 'border-muted-foreground text-muted-foreground'
                    }`}
                  >
                    {index < currentStep ? (
                      <CheckCircle className="w-4 h-4" />
                    ) : (
                      <Circle className="w-4 h-4" />
                    )}
                  </button>
                  <span className="text-xs text-muted-foreground mt-1 text-center">
                    {step.title.split(' ')[0]}
                  </span>
                </div>
              ))}
            </div>
          </div>

          {/* Desktop Step Indicators */}
          <div className="hidden sm:flex items-center justify-between mb-4">
            {steps.map((step, index) => (
              <div key={step.id} className="flex items-center">
                <button
                  onClick={() => onStepChange(index)}
                  className={`flex items-center justify-center w-10 h-10 rounded-full border-2 transition-colors ${
                    index <= currentStep
                      ? 'border-primary bg-primary text-primary-foreground'
                      : 'border-muted-foreground text-muted-foreground'
                  }`}
                >
                  {index < currentStep ? (
                    <CheckCircle className="w-5 h-5" />
                  ) : (
                    <Circle className="w-5 h-5" />
                  )}
                </button>
                <label className="text-sm text-muted-foreground ml-2">{step.title}</label>
                {index < steps.length - 1 && (
                  <div
                    className={`w-5 h-0.5 mx-2 ${
                      index < currentStep ? 'bg-primary' : 'bg-muted'
                    }`}
                  />
                )}
              </div>
            ))}
          </div>
          <div className="w-full bg-muted rounded-full h-2">
            <div
              className="bg-primary h-2 rounded-full transition-all duration-300"
              style={{ width: `${progress}%` }}
            />
          </div>
        </div>

        {/* Step Content */}
        <div className="mb-6">
          <h3 className="text-lg sm:text-xl font-semibold mb-2 flex items-center gap-2">
            {steps[currentStep]?.icon}
            {steps[currentStep]?.title}
          </h3>
          <p className="text-muted-foreground text-sm sm:text-base">
            {steps[currentStep]?.description}
          </p>
        </div>

        {/* Form Content */}
        <div className="mb-6">
          {children}
        </div>

        {/* Navigation */}
        <div className="flex flex-col sm:flex-row justify-between pt-6 border-t gap-4">
          <Button
            type="button"
            variant="outline"
            onClick={onPrevious}
            disabled={currentStep === 0}
            className="w-full sm:w-auto"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Previous
          </Button>

          {isLastStep ? (
            <Button
              type="button"
              onClick={onSubmit}
              disabled={!canProceed || isSubmitting}
              className="w-full sm:w-auto min-w-[120px]"
            >
              {isSubmitting ? 'Processing...' : 'Complete Payment'}
            </Button>
          ) : (
            <Button
              type="button"
              onClick={onNext}
              disabled={!canProceed}
              className="w-full sm:w-auto"
            >
              Next
              <ArrowRight className="w-4 h-4 ml-2" />
            </Button>
          )}
        </div>
      </Card>
    </div>
  );
};

export default TeamFormSteps;