import React from 'react';
import { Card } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { Settings, Shield, Info, Trophy, Users, Crown } from 'lucide-react';

interface TeamSettingsStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  tournaments: any[];
  users: any[];
}

const TeamSettingsStep: React.FC<TeamSettingsStepProps> = ({
  form,
  errors,
  onInputChange,
  tournaments,
  users,
}) => {
  const selectedTournament = tournaments.find(t => t.id === parseInt(form.tournament_id));
  const selectedCaptain = users.find(u => u.id === parseInt(form.captain_id));

  return (
    <div className="space-y-6">
      <Card className="p-6">
        

        <div className="space-y-6">
          {/* Team Summary */}
          <div>
            <h4 className="font-medium mb-3 flex items-center gap-2">
              <Info className="w-4 h-4" />
              Team Summary
            </h4>
            <Card className="p-4 bg-gray-50">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label className="text-sm font-medium text-gray-600">Team Name</Label>
                  <p className="font-medium">{form.name || 'Not specified'}</p>
                </div>
                
                <div>
                  <Label className="text-sm font-medium text-gray-600">Status</Label>
                  <div className="mt-1">
                    <Badge variant="outline" className="capitalize">
                      {form.status || 'Not specified'}
                    </Badge>
                  </div>
                </div>

                <div>
                  <Label className="text-sm font-medium text-gray-600">Tournament</Label>
                  <div className="flex items-center gap-2 mt-1">
                    <Trophy className="w-4 h-4 text-gray-500" />
                    <span>{selectedTournament?.name || 'Not selected'}</span>
                  </div>
                </div>

                <div>
                  <Label className="text-sm font-medium text-gray-600">Team Captain</Label>
                  <div className="flex items-center gap-2 mt-1">
                    <Crown className="w-4 h-4 text-yellow-600" />
                    <span>{selectedCaptain?.name || 'Not selected'}</span>
                  </div>
                </div>

                {form.description && (
                  <div className="md:col-span-2">
                    <Label className="text-sm font-medium text-gray-600">Description</Label>
                    <p className="text-sm text-gray-800 mt-1">{form.description}</p>
                  </div>
                )}
              </div>
            </Card>
          </div>

          {/* Additional Settings */}
          <div>
            <h4 className="font-medium mb-3 flex items-center gap-2">
              <Shield className="w-4 h-4" />
              Additional Settings
            </h4>
            
            <div className="space-y-4">
              <div>
                <Label htmlFor="max_players">Maximum Players</Label>
                <Input
                  id="max_players"
                  type="number"
                  value={form.max_players || ''}
                  onChange={(e) => onInputChange('max_players', e.target.value)}
                  placeholder="e.g., 11"
                  min="1"
                  max="50"
                />
                <p className="text-xs text-gray-500 mt-1">Maximum number of players allowed in this team</p>
              </div>

              <div>
                <Label htmlFor="notes">Internal Notes</Label>
                <Textarea
                  id="notes"
                  value={form.notes || ''}
                  onChange={(e) => onInputChange('notes', e.target.value)}
                  placeholder="Add any internal notes about this team..."
                  rows={3}
                />
                <p className="text-xs text-gray-500 mt-1">These notes are only visible to administrators</p>
              </div>

              {/* Notification Settings */}
              <div className="space-y-2">
                <Label className="text-sm font-medium">Notification Settings</Label>
                <div className="space-y-2">
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="notify_match_updates"
                      checked={form.notify_match_updates || false}
                      onCheckedChange={(checked) => onInputChange('notify_match_updates', checked)}
                    />
                    <Label htmlFor="notify_match_updates" className="text-sm">
                      Notify team about match updates
                    </Label>
                  </div>
                  
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="notify_tournament_updates"
                      checked={form.notify_tournament_updates || false}
                      onCheckedChange={(checked) => onInputChange('notify_tournament_updates', checked)}
                    />
                    <Label htmlFor="notify_tournament_updates" className="text-sm">
                      Notify team about tournament updates
                    </Label>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Validation Messages */}
          {Object.keys(errors).length > 0 && (
            <Card className="p-4 bg-red-50 border-red-200">
              <h4 className="font-medium text-red-800 mb-2">Please fix the following errors:</h4>
              <ul className="text-sm text-red-700 space-y-1">
                {Object.keys(errors).map(key => (
                  <li key={key}>• {errors[key]}</li>
                ))}
              </ul>
            </Card>
          )}
        </div>
      </Card>
    </div>
  );
};

export default TeamSettingsStep;