import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';

import { 
  UserCheck, 
  User, 
  UserPlus, 
  Search, 
  Mail, 
  Phone, 
  Plus, 
  Minus, 
  Users,
  Shield,
  Star,
  Crown
} from 'lucide-react';

interface Player {
  name: string;
  email: string;
  phone: string;
  position: string;
  jersey_number: string;
  existing_user_id?: number | null;
}

interface PlayersSelectionStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  users: any[];
}

const positions = [
  'Goalkeeper', 'Right Back', 'Left Back', 'Centre Back', 
  'Defensive Midfielder', 'Central Midfielder', 'Attacking Midfielder', 
  'Right Winger', 'Left Winger', 'Centre Forward', 'Striker'
];

const PlayersSelectionStep: React.FC<PlayersSelectionStepProps> = ({ 
  form, 
  errors, 
  onInputChange, 
  users 
}) => {
  const [selectedPlayerIndex, setSelectedPlayerIndex] = useState<number | null>(null);
  const [activeTab, setActiveTab] = useState('create');
  const [searchTerm, setSearchTerm] = useState('');
  const [newUserForm, setNewUserForm] = useState({
    name: '',
    email: '',
    phone: '',
    password: '',
    password_confirmation: ''
  });
  const [isCreatingUser, setIsCreatingUser] = useState(false);

  // Filter existing users based on search
  const filteredUsers = users.filter(user =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handlePlayerChange = (index: number, field: keyof Player, value: string) => {
    const updatedPlayers = form.players.map((player: Player, i: number) => 
      i === index ? { ...player, [field]: value } : player
    );
    onInputChange('players', updatedPlayers);
  };

  const addPlayer = () => {
    if (form.players.length < 23) {
      const newPlayers = [...form.players, {
        name: '',
        email: '',
        phone: '',
        position: 'Midfielder',
        jersey_number: '',
        existing_user_id: null
      }];
      onInputChange('players', newPlayers);
    }
  };

  const removePlayer = (index: number) => {
    if (form.players.length > 11) {
      const updatedPlayers = form.players.filter((_: any, i: number) => i !== index);
      onInputChange('players', updatedPlayers);
      if (selectedPlayerIndex === index) {
        setSelectedPlayerIndex(null);
      }
    }
  };

  const selectExistingUser = (user: any) => {
    if (selectedPlayerIndex !== null) {
      // Update the player data with the selected user information
      const updatedPlayers = form.players.map((player: Player, i: number) => 
        i === selectedPlayerIndex ? {
          ...player,
          name: user.name,
          email: user.email,
          phone: user.phone || '',
          existing_user_id: user.id
        } : player
      );
      onInputChange('players', updatedPlayers);
      setSelectedPlayerIndex(null);
      setActiveTab('create');
    }
  };

  const openUserSelection = (index: number) => {
    setSelectedPlayerIndex(index);
    setActiveTab('existing');
    setSearchTerm('');
  };

  const handleCreateUser = async () => {
    // Basic validation
    if (!newUserForm.name || !newUserForm.email || !newUserForm.phone) {
      return;
    }

    if (newUserForm.password !== newUserForm.password_confirmation) {
      return;
    }

    setIsCreatingUser(true);
    try {
      const response = await fetch(route('admin.users.store'), {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify(newUserForm),
      });

      if (response.ok) {
        const newUser = await response.json();
        // Add the new user to the users list and select for current player
        users.push(newUser);
        if (selectedPlayerIndex !== null) {
          selectExistingUser(newUser);
        }
        
        setNewUserForm({ name: '', email: '', phone: '', password: '', password_confirmation: '' });
      } else {
        const errorData = await response.json();
      }
    } catch (error) {
    } finally {
      setIsCreatingUser(false);
    }
  };

  const getPlayerIcon = (index: number) => {
    if (index === 0) return <Crown className="w-4 h-4 text-yellow-500" />;
    if (index < 4) return <Shield className="w-4 h-4 text-blue-500" />;
    if (index < 8) return <Star className="w-4 h-4 text-green-500" />;
    return <User className="w-4 h-4 text-gray-500" />;
  };

  const validPlayers = form.players.filter((p: Player) => p.name && p.email && p.phone);

  return (
    <div className="space-y-6">
      <Card>
        
        <CardContent className="space-y-6">
          {/* Squad Progress */}
          <div className="bg-gradient-to-r from-blue-50 to-green-50 p-4 rounded-lg border">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium">Squad Completion</span>
              <span className="text-sm text-muted-foreground">
                {validPlayers.length}/{form.players.length} players ready
              </span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div 
                className="bg-gradient-to-r from-blue-500 to-green-500 h-2 rounded-full transition-all duration-300"
                style={{ width: `${Math.min((validPlayers.length / 11) * 100, 100)}%` }}
              />
            </div>
            {validPlayers.length >= 11 && (
              <div className="flex items-center gap-1 mt-2 text-green-600">
                <Shield className="w-4 h-4" />
                <span className="text-sm font-medium">Squad ready for battle! 🏆</span>
              </div>
            )}
          </div>

          

          {/* Players List */}
          <div className="space-y-4">
            {form.players.map((player: Player, index: number) => (
                             <Card key={index} className={`transition-all duration-200 ${
                 player.name && player.email ? 'border-green-200 bg-green-50' : 'border-gray-200'
               } ${player.existing_user_id ? 'border-blue-300 bg-blue-50' : ''}`}>
                <CardContent className="p-4">
                  <div className="flex items-center gap-3 mb-3">
                    {getPlayerIcon(index)}
                                         <h4 className="font-medium">
                       Player #{index + 1}
                       {index === 0 && <Badge variant="secondary" className="ml-2">Captain</Badge>}
                       {player.name && <Badge variant="secondary" className="ml-2 bg-green-100 text-green-800">Ready</Badge>}
                       {player.existing_user_id && <Badge variant="secondary" className="ml-2 bg-blue-100 text-blue-800">User Selected</Badge>}
                     </h4>
                                         <div className="flex-1" />
                     {player.existing_user_id && (
                       <div className="text-xs text-blue-600 mr-2">
                         <UserCheck className="w-3 h-3 inline mr-1" />
                         User #{player.existing_user_id}
                       </div>
                     )}
                     <Button
                       variant="outline"
                       size="sm"
                       onClick={() => openUserSelection(index)}
                       className="text-xs"
                     >
                       <Search className="w-3 h-3 mr-1" />
                       {player.existing_user_id ? 'Change User' : 'Select User'}
                     </Button>
                    {form.players.length > 11 && (
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => removePlayer(index)}
                        className="text-red-600 hover:text-red-700"
                      >
                        <Minus className="w-3 h-3" />
                      </Button>
                    )}
                  </div>

                                     {/* User Selection Section */}
                   {selectedPlayerIndex === index && (
                     <div className="mt-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
                       <div className="flex items-center gap-2 mb-3">
                         <Search className="w-4 h-4 text-blue-600" />
                         <h5 className="font-medium text-blue-900">Select User for Player #{index + 1}</h5>
                       </div>
                       
                       <Tabs value={activeTab} onValueChange={setActiveTab}>
                         <TabsList className="grid w-full grid-cols-2">
                           <TabsTrigger value="existing" className="flex items-center gap-2">
                             <Search className="w-4 h-4" />
                             Select Existing
                           </TabsTrigger>
                           <TabsTrigger value="create" className="flex items-center gap-2">
                             <UserPlus className="w-4 h-4" />
                             Create New
                           </TabsTrigger>
                         </TabsList>

                         <TabsContent value="existing" className="space-y-3 mt-3">
                           <div>
                             <Label htmlFor={`user_search_${index}`}>Search Users</Label>
                             <div className="relative">
                               <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                               <Input
                                 id={`user_search_${index}`}
                                 placeholder="Search by name or email..."
                                 value={searchTerm}
                                 onChange={(e) => setSearchTerm(e.target.value)}
                                 className="pl-10"
                               />
                             </div>
                           </div>

                           <div className="max-h-32 overflow-y-auto space-y-2">
                             {filteredUsers.map(user => {
                               const isSelected = form.players.some((p: Player) => p.existing_user_id === user.id);
                               return (
                                 <Card 
                                   key={user.id} 
                                   className={`p-2 cursor-pointer transition-colors ${
                                     isSelected ? 'border-green-300 bg-green-50' : 'hover:border-blue-300'
                                   }`}
                                   onClick={() => selectExistingUser(user)}
                                 >
                                   <div className="flex items-center gap-2">
                                     <User className="w-3 h-3 text-blue-500" />
                                     <div className="flex-1 min-w-0">
                                       <div className="font-medium text-sm truncate">{user.name}</div>
                                       <div className="text-xs text-gray-500 truncate">{user.email}</div>
                                     </div>
                                     <UserCheck className="w-3 h-3 text-green-500" />
                                     {isSelected && (
                                       <Badge variant="secondary" className="bg-green-100 text-green-800 text-xs">
                                         Selected
                                       </Badge>
                                     )}
                                   </div>
                                 </Card>
                               );
                             })}
                           </div>

                           {filteredUsers.length === 0 && searchTerm && (
                             <div className="text-center py-2 text-gray-500">
                               <p className="text-xs">No users found matching "{searchTerm}"</p>
                               <Button 
                                 variant="outline" 
                                 size="sm" 
                                 className="mt-1"
                                 onClick={() => setActiveTab('create')}
                               >
                                 Create New User Instead
                               </Button>
                             </div>
                           )}

                           <div className="flex gap-2">
                             <Button 
                               variant="outline" 
                               size="sm"
                               onClick={() => setSelectedPlayerIndex(null)}
                             >
                               Cancel
                             </Button>
                           </div>
                         </TabsContent>

                         <TabsContent value="create" className="mt-3">
                           <div className="space-y-3">
                             <div>
                               <Label htmlFor={`new_user_name_${index}`}>Full Name</Label>
                               <Input
                                 id={`new_user_name_${index}`}
                                 value={newUserForm.name}
                                 onChange={(e) => setNewUserForm(prev => ({ ...prev, name: e.target.value }))}
                                 placeholder="Enter full name"
                               />
                             </div>
                             <div>
                               <Label htmlFor={`new_user_email_${index}`}>Email</Label>
                               <Input
                                 id={`new_user_email_${index}`}
                                 type="email"
                                 value={newUserForm.email}
                                 onChange={(e) => setNewUserForm(prev => ({ ...prev, email: e.target.value }))}
                                 placeholder="user@example.com"
                               />
                             </div>
                             <div>
                               <Label htmlFor={`new_user_phone_${index}`}>Phone</Label>
                               <Input
                                 id={`new_user_phone_${index}`}
                                 value={newUserForm.phone}
                                 onChange={(e) => setNewUserForm(prev => ({ ...prev, phone: e.target.value }))}
                                 placeholder="+254712345678"
                               />
                             </div>
                             <div>
                               <Label htmlFor={`new_user_password_${index}`}>Password</Label>
                               <Input
                                 id={`new_user_password_${index}`}
                                 type="password"
                                 value={newUserForm.password}
                                 onChange={(e) => setNewUserForm(prev => ({ ...prev, password: e.target.value }))}
                                 placeholder="Enter password"
                               />
                             </div>
                             <div>
                               <Label htmlFor={`new_user_password_confirmation_${index}`}>Confirm Password</Label>
                               <Input
                                 id={`new_user_password_confirmation_${index}`}
                                 type="password"
                                 value={newUserForm.password_confirmation}
                                 onChange={(e) => setNewUserForm(prev => ({ ...prev, password_confirmation: e.target.value }))}
                                 placeholder="Confirm password"
                               />
                             </div>
                             <div className="flex gap-2">
                               <Button 
                                 variant="outline" 
                                 size="sm"
                                 onClick={() => setSelectedPlayerIndex(null)}
                               >
                                 Cancel
                               </Button>
                               <Button 
                                 onClick={handleCreateUser} 
                                 disabled={isCreatingUser}
                                 size="sm"
                               >
                                 {isCreatingUser ? 'Creating...' : 'Create User'}
                               </Button>
                             </div>
                           </div>
                         </TabsContent>
                       </Tabs>
                     </div>
                   )}

                   <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                     <div>
                       <Label htmlFor={`player_${index}_name`}>Full Name *</Label>
                       <Input
                         id={`player_${index}_name`}
                         value={player.name}
                         onChange={(e) => handlePlayerChange(index, 'name', e.target.value)}
                         placeholder="Enter player name"
                         className={errors[`players.${index}.name`] ? 'border-red-500' : ''}
                       />
                       {errors[`players.${index}.name`] && (
                         <p className="text-red-500 text-xs mt-1">{errors[`players.${index}.name`]}</p>
                       )}
                     </div>

                     <div>
                       <Label htmlFor={`player_${index}_email`}>Email *</Label>
                       <Input
                         id={`player_${index}_email`}
                         type="email"
                         value={player.email}
                         onChange={(e) => handlePlayerChange(index, 'email', e.target.value)}
                         placeholder="player@example.com"
                         className={errors[`players.${index}.email`] ? 'border-red-500' : ''}
                       />
                       {errors[`players.${index}.email`] && (
                         <p className="text-red-500 text-xs mt-1">{errors[`players.${index}.email`]}</p>
                       )}
                     </div>

                     <div>
                       <Label htmlFor={`player_${index}_phone`}>Phone *</Label>
                       <Input
                         id={`player_${index}_phone`}
                         value={player.phone}
                         onChange={(e) => handlePlayerChange(index, 'phone', e.target.value)}
                         placeholder="+254712345678"
                         className={errors[`players.${index}.phone`] ? 'border-red-500' : ''}
                       />
                       {errors[`players.${index}.phone`] && (
                         <p className="text-red-500 text-xs mt-1">{errors[`players.${index}.phone`]}</p>
                       )}
                     </div>

                     <div>
                       <Label htmlFor={`player_${index}_position`}>Position</Label>
                       <Select 
                         value={player.position} 
                         onValueChange={(value) => handlePlayerChange(index, 'position', value)}
                       >
                         <SelectTrigger>
                           <SelectValue placeholder="Select position" />
                         </SelectTrigger>
                         <SelectContent>
                           {positions.map(position => (
                             <SelectItem key={position} value={position}>
                               {position}
                             </SelectItem>
                           ))}
                         </SelectContent>
                       </Select>
                     </div>

                     <div>
                       <Label htmlFor={`player_${index}_jersey`}>Jersey Number</Label>
                       <Input
                         id={`player_${index}_jersey`}
                         value={player.jersey_number}
                         onChange={(e) => handlePlayerChange(index, 'jersey_number', e.target.value)}
                         placeholder="e.g., 10"
                         maxLength={3}
                       />
                     </div>
                   </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Add More Players */}
          {form.players.length < 23 && (
            <div className="text-center">
              <Button 
                variant="outline" 
                onClick={addPlayer}
                className="border-dashed border-2 hover:border-blue-300"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Another Player ({form.players.length}/23)
              </Button>
            </div>
          )}

          {/* Validation Errors */}
          {errors.players && (
            <div className="bg-red-50 border border-red-200 rounded-lg p-4">
              <p className="text-red-600 font-medium">{errors.players}</p>
            </div>
          )}
        </CardContent>
      </Card>

      
    </div>
  );
};

export default PlayersSelectionStep;