import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { 
  CreditCard, 
  DollarSign, 
  Clock, 
  CheckCircle, 
  AlertCircle, 
  Loader2,
  Shield,
  Info
} from 'lucide-react';

interface PaymentStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  onPaymentComplete: (paymentData: any) => void;
  teamData: any;
}

interface PaymentStatus {
  status: 'pending' | 'processing' | 'completed' | 'failed';
  transactionId?: string;
  message?: string;
}

const PaymentStep: React.FC<PaymentStepProps> = ({
  form,
  errors,
  onInputChange,
  onPaymentComplete,
  teamData,
}) => {
  const [paymentMethod, setPaymentMethod] = useState('card');
  const [paymentStatus, setPaymentStatus] = useState<PaymentStatus>({ status: 'pending' });
  const [isProcessing, setIsProcessing] = useState(false);
  const [pollingActive, setPollingActive] = useState(false);
  const [timeElapsed, setTimeElapsed] = useState(0);

  // Mock payment data - in real app, this would come from payment provider
  const registrationFee = 500;
  const processingFee = 25;
  const totalAmount = registrationFee + processingFee;

  // Simulate payment processing
  const initiatePayment = async () => {
    setIsProcessing(true);
    setPaymentStatus({ status: 'processing', message: 'Initializing payment...' });

    try {
      // Simulate payment API call
      const paymentResponse = await fetch(route('api.payments.initiate'), {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify({
          amount: totalAmount,
          currency: 'USD',
          team_data: teamData,
          payment_method: paymentMethod,
          card_details: {
            number: form.card_number,
            expiry: form.card_expiry,
            cvv: form.card_cvv,
            name: form.card_name,
          }
        }),
      });

      const result = await paymentResponse.json();
      
      if (result.success) {
        setPaymentStatus({ 
          status: 'processing', 
          transactionId: result.transaction_id,
          message: 'Payment submitted successfully. Processing...' 
        });

        // Wait 5 seconds before starting polling
        setTimeout(() => {
          startPolling(result.transaction_id);
        }, 5000);
      } else {
        setPaymentStatus({ 
          status: 'failed', 
          message: result.message || 'Payment initiation failed' 
        });
        setIsProcessing(false);
      }
    } catch (error) {
      // Simulate successful payment for demo
      setPaymentStatus({ 
        status: 'processing', 
        transactionId: 'TXN_' + Date.now(),
        message: 'Payment submitted successfully. Processing...' 
      });

      // Wait 5 seconds before starting polling
      setTimeout(() => {
        startPolling('TXN_' + Date.now());
      }, 5000);
    }
  };

  // Start polling for payment status
  const startPolling = (transactionId: string) => {
    setPollingActive(true);
    setTimeElapsed(0);
    
    const pollInterval = setInterval(async () => {
      setTimeElapsed(prev => prev + 2);

      try {
        // Check payment status
        const statusResponse = await fetch(route('api.payments.status', transactionId));
        const statusResult = await statusResponse.json();

        if (statusResult.status === 'completed') {
          setPaymentStatus({ 
            status: 'completed', 
            transactionId,
            message: 'Payment completed successfully!' 
          });
          setPollingActive(false);
          setIsProcessing(false);
          clearInterval(pollInterval);

          // Call completion handler
          onPaymentComplete({
            transaction_id: transactionId,
            amount: totalAmount,
            status: 'completed'
          });

          // Redirect to team view after 2 seconds
          setTimeout(() => {
            window.location.href = '/admin/teams';
          }, 2000);

        } else if (statusResult.status === 'failed') {
          setPaymentStatus({ 
            status: 'failed', 
            message: statusResult.message || 'Payment failed' 
          });
          setPollingActive(false);
          setIsProcessing(false);
          clearInterval(pollInterval);
        }

        // Simulate completion after 10 seconds for demo
        if (timeElapsed >= 10) {
          setPaymentStatus({ 
            status: 'completed', 
            transactionId,
            message: 'Payment completed successfully!' 
          });
          setPollingActive(false);
          setIsProcessing(false);
          clearInterval(pollInterval);

          onPaymentComplete({
            transaction_id: transactionId,
            amount: totalAmount,
            status: 'completed'
          });

          setTimeout(() => {
            window.location.href = '/admin/teams';
          }, 2000);
        }
      } catch (error) {
         
      }
    }, 2000); // Poll every 2 seconds

    // Stop polling after 60 seconds
    setTimeout(() => {
      if (pollingActive) {
        clearInterval(pollInterval);
        setPollingActive(false);
        setIsProcessing(false);
        setPaymentStatus({ 
          status: 'failed', 
          message: 'Payment verification timeout. Please contact support.' 
        });
      }
    }, 60000);
  };

  const canProcessPayment = () => {
    if (paymentMethod === 'card') {
      return form.card_number && form.card_expiry && form.card_cvv && form.card_name;
    }
    return true;
  };

  return (
    <div className="space-y-6">
      {/* Payment Status */}
      {paymentStatus.status !== 'pending' && (
        <Card className={`${
          paymentStatus.status === 'completed' ? 'bg-green-50 border-green-200' :
          paymentStatus.status === 'failed' ? 'bg-red-50 border-red-200' :
          'bg-blue-50 border-blue-200'
        }`}>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              {paymentStatus.status === 'completed' && <CheckCircle className="w-6 h-6 text-green-600" />}
              {paymentStatus.status === 'failed' && <AlertCircle className="w-6 h-6 text-red-600" />}
              {paymentStatus.status === 'processing' && <Loader2 className="w-6 h-6 text-blue-600 animate-spin" />}
              
              <div>
                <h4 className={`font-medium ${
                  paymentStatus.status === 'completed' ? 'text-green-800' :
                  paymentStatus.status === 'failed' ? 'text-red-800' :
                  'text-blue-800'
                }`}>
                  {paymentStatus.status === 'completed' && 'Payment Successful'}
                  {paymentStatus.status === 'failed' && 'Payment Failed'}
                  {paymentStatus.status === 'processing' && 'Processing Payment'}
                </h4>
                <p className={`text-sm ${
                  paymentStatus.status === 'completed' ? 'text-green-700' :
                  paymentStatus.status === 'failed' ? 'text-red-700' :
                  'text-blue-700'
                }`}>
                  {paymentStatus.message}
                </p>
                {pollingActive && (
                  <p className="text-xs text-blue-600 mt-1">
                    Verifying payment... ({timeElapsed}s)
                  </p>
                )}
                {paymentStatus.transactionId && (
                  <p className="text-xs text-gray-600 mt-1">
                    Transaction ID: {paymentStatus.transactionId}
                  </p>
                )}
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Payment Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <DollarSign className="w-5 h-5 text-green-600" />
            Payment Summary
          </CardTitle>
          <CardDescription>
            Complete payment to finalize team registration
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="bg-gray-50 p-4 rounded-lg space-y-2">
            <div className="flex justify-between">
              <span>Registration Fee</span>
              <span>${registrationFee}.00</span>
            </div>
            <div className="flex justify-between">
              <span>Processing Fee</span>
              <span>${processingFee}.00</span>
            </div>
            <div className="border-t pt-2 flex justify-between font-medium">
              <span>Total Amount</span>
              <span>${totalAmount}.00 USD</span>
            </div>
          </div>

          {/* Team Summary */}
          <div className="bg-blue-50 p-4 rounded-lg">
            <h4 className="font-medium text-blue-800 mb-2">Team Details</h4>
            <div className="grid grid-cols-2 gap-2 text-sm">
              <div>
                <span className="text-blue-600">Team:</span>
                <p className="font-medium">{teamData.name}</p>
              </div>
              <div>
                <span className="text-blue-600">Captain:</span>
                <p className="font-medium">{teamData.captain_name}</p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Payment Method */}
      {paymentStatus.status === 'pending' && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CreditCard className="w-5 h-5 text-blue-600" />
              Payment Method
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Payment Method Selection */}
            <div>
              <Label>Select Payment Method</Label>
              <Select value={paymentMethod} onValueChange={setPaymentMethod}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="card">Credit/Debit Card</SelectItem>
                  <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                  <SelectItem value="paypal">PayPal</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Card Details */}
            {paymentMethod === 'card' && (
              <div className="space-y-4">
                <div>
                  <Label htmlFor="card_name">Cardholder Name *</Label>
                  <Input
                    id="card_name"
                    value={form.card_name || ''}
                    onChange={(e) => onInputChange('card_name', e.target.value)}
                    placeholder="John Smith"
                    className={errors.card_name ? 'border-red-500' : ''}
                  />
                  {errors.card_name && <p className="text-red-500 text-sm mt-1">{errors.card_name}</p>}
                </div>

                <div>
                  <Label htmlFor="card_number">Card Number *</Label>
                  <Input
                    id="card_number"
                    value={form.card_number || ''}
                    onChange={(e) => onInputChange('card_number', e.target.value)}
                    placeholder="1234 5678 9012 3456"
                    className={errors.card_number ? 'border-red-500' : ''}
                  />
                  {errors.card_number && <p className="text-red-500 text-sm mt-1">{errors.card_number}</p>}
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="card_expiry">Expiry Date *</Label>
                    <Input
                      id="card_expiry"
                      value={form.card_expiry || ''}
                      onChange={(e) => onInputChange('card_expiry', e.target.value)}
                      placeholder="MM/YY"
                      className={errors.card_expiry ? 'border-red-500' : ''}
                    />
                    {errors.card_expiry && <p className="text-red-500 text-sm mt-1">{errors.card_expiry}</p>}
                  </div>

                  <div>
                    <Label htmlFor="card_cvv">CVV *</Label>
                    <Input
                      id="card_cvv"
                      value={form.card_cvv || ''}
                      onChange={(e) => onInputChange('card_cvv', e.target.value)}
                      placeholder="123"
                      className={errors.card_cvv ? 'border-red-500' : ''}
                    />
                    {errors.card_cvv && <p className="text-red-500 text-sm mt-1">{errors.card_cvv}</p>}
                  </div>
                </div>
              </div>
            )}

            {/* Bank Transfer Instructions */}
            {paymentMethod === 'bank_transfer' && (
              <Card className="bg-yellow-50 border-yellow-200">
                <CardContent className="p-4">
                  <h4 className="font-medium text-yellow-800 mb-2">Bank Transfer Instructions</h4>
                  <div className="text-sm text-yellow-700 space-y-2">
                    <p><strong>Account Name:</strong> Milimus Tournament</p>
                    <p><strong>Account Number:</strong> 1234567890</p>
                    <p><strong>Routing Number:</strong> 123456789</p>
                    <p><strong>Reference:</strong> TEAM-{teamData.name?.replace(/\s/g, '')}</p>
                  </div>
                </CardContent>
              </Card>
            )}

            {/* PayPal Notice */}
            {paymentMethod === 'paypal' && (
              <Card className="bg-blue-50 border-blue-200">
                <CardContent className="p-4">
                  <h4 className="font-medium text-blue-800 mb-2">PayPal Payment</h4>
                  <p className="text-sm text-blue-700">
                    You will be redirected to PayPal to complete your payment securely.
                  </p>
                </CardContent>
              </Card>
            )}

            {/* Payment Button */}
            <Button 
              onClick={initiatePayment}
              disabled={!canProcessPayment() || isProcessing}
              className="w-full"
              size="lg"
            >
              {isProcessing ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Processing Payment...
                </>
              ) : (
                <>
                  <CreditCard className="w-4 h-4 mr-2" />
                  Pay ${totalAmount}.00
                </>
              )}
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Security Notice */}
      <Card className="bg-gray-50 border-gray-200">
        <CardContent className="p-4">
          <h4 className="font-medium text-gray-800 mb-2 flex items-center gap-2">
            <Shield className="w-4 h-4" />
            Secure Payment
          </h4>
          <div className="text-sm text-gray-700 space-y-1">
            <p>• Your payment information is encrypted and secure</p>
            <p>• We use industry-standard SSL encryption</p>
            <p>• Card details are processed by certified payment providers</p>
            <p>• No payment information is stored on our servers</p>
          </div>
        </CardContent>
      </Card>

      {/* Processing Info */}
      {pollingActive && (
        <Card className="bg-blue-50 border-blue-200">
          <CardContent className="p-4">
            <h4 className="font-medium text-blue-800 mb-2 flex items-center gap-2">
              <Clock className="w-4 h-4" />
              Payment Processing
            </h4>
            <div className="text-sm text-blue-700 space-y-1">
              <p>• Please do not close this page or refresh the browser</p>
              <p>• Payment verification typically takes 10-30 seconds</p>
              <p>• You will be automatically redirected when payment is confirmed</p>
              <p>• If you experience issues, please contact our support team</p>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
};

export default PaymentStep;