import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { 
  CreditCard, 
  Smartphone, 
  Clock, 
  CheckCircle, 
  AlertCircle, 
  Loader2,
  Shield,
  Phone,
  Users,
  Trophy,
  Star
} from 'lucide-react';

interface MPesaPaymentStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  onPaymentComplete: (paymentData: any) => void;
  teamData: any;
  registrationFee: number;
}

interface PaymentState {
  status: 'idle' | 'initiating' | 'processing' | 'waiting' | 'completed' | 'failed';
  message: string;
  checkoutRequestId?: string;
  error?: string;
}

const MPesaPaymentStep: React.FC<MPesaPaymentStepProps> = ({
  form,
  errors,
  onInputChange,
  onPaymentComplete,
  teamData,
  registrationFee,
}) => {
  const [paymentState, setPaymentState] = useState<PaymentState>({
    status: 'idle',
    message: ''
  });
  const [pollingInterval, setPollingInterval] = useState<NodeJS.Timeout | null>(null);
  const [countdownInterval, setCountdownInterval] = useState<NodeJS.Timeout | null>(null);
  const [timeRemaining, setTimeRemaining] = useState(0);
  const [pollingDelay, setPollingDelay] = useState(0);
  const [isPolling, setIsPolling] = useState(false);

  // Payment configuration
  const totalAmount = registrationFee;

  // Cleanup intervals on unmount
  useEffect(() => {
    return () => {
      if (pollingInterval) {
        clearInterval(pollingInterval);
      }
      if (countdownInterval) {
        clearInterval(countdownInterval);
      }
    };
  }, [pollingInterval, countdownInterval]);

  const validatePhoneNumber = (phone: string): boolean => {
    // Remove any spaces, dashes, or plus signs
    const cleanPhone = phone.replace(/[\s\-\+]/g, '');
    
    // Check if it matches 254XXXXXXXXX format (Kenyan M-Pesa format)
    const phoneRegex = /^254[0-9]{9}$/;
    
    return phoneRegex.test(cleanPhone);
  };

  const formatPhoneNumber = (phone: string): string => {
    // Remove any spaces, dashes, or plus signs and ensure it starts with 254
    let cleanPhone = phone.replace(/[\s\-\+]/g, '');
    
    // If it starts with 0, replace with 254
    if (cleanPhone.startsWith('0')) {
      cleanPhone = '254' + cleanPhone.substring(1);
    }
    
    // If it starts with 7, add 254
    if (cleanPhone.startsWith('7') || cleanPhone.startsWith('1')) {
      cleanPhone = '254' + cleanPhone;
    }
    
    return cleanPhone;
  };

  const startCountdown = (duration: number) => {
    setTimeRemaining(duration);
    const interval = setInterval(() => {
      setTimeRemaining((prev) => {
        if (prev <= 1) {
          clearInterval(interval);
          return 0;
        }
        return prev - 1;
      });
    }, 1000);
    setCountdownInterval(interval);
    return interval;
  };

  const startPolling = (paymentId: string, checkoutRequestId: string) => {
    setIsPolling(true);
    setPollingDelay(10); // Start polling delay
    
    const countdownInterval = setInterval(() => {
      setPollingDelay((prev) => prev > 0 ? prev - 1 : 0);
    }, 1000);

    setTimeout(() => {
      clearInterval(countdownInterval);
      setPollingDelay(0);
      
      const interval = setInterval(async () => {
        try {
          const response = await fetch('/payment/poll-status', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
            },
            body: JSON.stringify({
              checkout_request_id: checkoutRequestId
            })
          });
          
          const data = await response.json();
          
          if (data.success && data.status === 'completed') {
            clearInterval(interval);
            setIsPolling(false);
            setPaymentState({
              status: 'completed',
              message: 'Payment completed successfully! 🎉'
            });
            onPaymentComplete({
              transaction_id: data.payment?.reference_number,
              amount: totalAmount,
              phone_number: form.phone_number
            });
          } else if (data.success && data.status === 'failed') {
            clearInterval(interval);
            setIsPolling(false);
            setPaymentState({
              status: 'failed',
              message: data.message || 'Payment failed. Please try again.',
              error: data.error
            });
          } else if (data.success && data.status === 'processing') {
            // Continue polling
          }
        } catch (error) {
        }
      }, 3000); // Poll every 3 seconds

      setPollingInterval(interval);

      // Stop polling after 5 minutes
      setTimeout(() => {
        clearInterval(interval);
        setIsPolling(false);
        if (paymentState.status !== 'completed') {
          setPaymentState({
            status: 'failed',
            message: 'Payment timeout. Please try again or contact support.',
            error: 'timeout'
          });
        }
      }, 300000); // 5 minutes
    }, 10000); // Wait 10 seconds before starting to poll
  };

  const initiatePayment = async () => {
    if (!validatePhoneNumber(form.phone_number)) {
      return;
    }

    const formattedPhone = formatPhoneNumber(form.phone_number);
    onInputChange('phone_number', formattedPhone);

    setPaymentState({
      status: 'initiating',
      message: 'Initiating M-Pesa payment...'
    });

    try {
      const response = await fetch('/payment/initiate', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify({
          amount: totalAmount,
          phone_number: formattedPhone,
          description: `Tournament Registration - ${teamData.name || 'Team Registration'}`
        })
      });

      const data = await response.json();

      if (data.success) {
        setPaymentState({
          status: 'processing',
          message: data.message || 'Payment request sent to your phone.',
          checkoutRequestId: data.checkout_request_id
        });

        // Start countdown and polling
        startCountdown(120); // 2 minutes countdown
        startPolling(data.payment_id, data.checkout_request_id);
      } else {
        setPaymentState({
          status: 'failed',
          message: data.message || 'Failed to initiate payment. Please try again.',
          error: data.error
        });
      }
    } catch (error) {
      setPaymentState({
        status: 'failed',
        message: 'Network error. Please check your connection and try again.',
        error: 'network_error'
      });
    }
  };

  const retryPayment = () => {
    if (pollingInterval) {
      clearInterval(pollingInterval);
    }
    if (countdownInterval) {
      clearInterval(countdownInterval);
    }
    setPaymentState({ status: 'idle', message: '' });
    setTimeRemaining(0);
    setIsPolling(false);
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  return (
    <div className="space-y-6">
      <Card>
        
        <CardContent className="space-y-6">
          {/* Team Summary */}
          <div className="bg-gradient-to-r from-blue-50 to-green-50 p-3 sm:p-4 rounded-lg border">
            <h4 className="font-semibold text-base sm:text-lg mb-3 flex items-center gap-2">
              <Trophy className="w-4 h-4 sm:w-5 sm:h-5 text-yellow-500" />
              Team Registration Summary
            </h4>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4 text-xs sm:text-sm">
              <div>
                <span className="text-muted-foreground">Team Name:</span>
                <div className="font-medium truncate">{teamData.name || 'Not specified'}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Captain:</span>
                <div className="font-medium truncate">{teamData.captain_name || 'Not selected'}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Tournament:</span>
                <div className="font-medium truncate">{teamData.tournament || 'Not selected'}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Squad Size:</span>
                <div className="font-medium flex items-center gap-1">
                  <Users className="w-3 h-3 sm:w-4 sm:h-4" />
                  {teamData.players_count || 0} players
                </div>
              </div>
            </div>
          </div>

          {/* Payment Details */}
          <div className="border rounded-lg p-3 sm:p-4">
            <h4 className="font-semibold mb-3 flex items-center gap-2 text-sm sm:text-base">
              <CreditCard className="w-4 h-4 sm:w-5 sm:h-5 text-blue-500" />
              Payment Breakdown
            </h4>
            <div className="space-y-2 text-xs sm:text-sm">
              <div className="flex justify-between">
                <span>Tournament Registration Fee</span>
                <span className="font-medium">KES {registrationFee.toLocaleString()}</span>
              </div>
              
              <div className="border-t pt-2 flex justify-between font-semibold text-base sm:text-lg">
                <span>Total Amount</span>
                <span className="text-green-600">KES {totalAmount.toLocaleString()}</span>
              </div>
            </div>
          </div>

          {/* Phone Number Input */}
          {paymentState.status === 'idle' && (
            <div className="space-y-4">
              <div>
                <Label htmlFor="phone_number" className="flex items-center gap-2">
                  <Phone className="w-4 h-4" />
                  M-Pesa Phone Number *
                </Label>
                <Input
                  id="phone_number"
                  value={form.phone_number}
                  onChange={(e) => onInputChange('phone_number', e.target.value)}
                  placeholder="254712345678"
                  className={errors.phone_number ? 'border-red-500' : ''}
                />
                {errors.phone_number && (
                  <p className="text-red-500 text-sm mt-1">{errors.phone_number}</p>
                )}
                <p className="text-xs text-muted-foreground mt-1">
                  Enter your M-Pesa registered phone number (format: 254XXXXXXXXX)
                </p>
              </div>

              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <h4 className="font-medium text-blue-900 mb-2 flex items-center gap-2">
                  <Shield className="w-4 h-4" />
                  How M-Pesa Payment Works
                </h4>
                <ol className="space-y-1 text-sm text-blue-800">
                  <li className="flex items-start gap-2">
                    <Badge variant="secondary" className="bg-blue-100 text-blue-800 text-xs">1</Badge>
                    <span>Click "Pay with M-Pesa" to initiate payment</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <Badge variant="secondary" className="bg-blue-100 text-blue-800 text-xs">2</Badge>
                    <span>You'll receive an M-Pesa prompt on your phone</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <Badge variant="secondary" className="bg-blue-100 text-blue-800 text-xs">3</Badge>
                    <span>Enter your M-Pesa PIN to complete payment</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <Badge variant="secondary" className="bg-blue-100 text-blue-800 text-xs">4</Badge>
                    <span>You'll receive confirmation and can proceed</span>
                  </li>
                </ol>
              </div>

              <Button 
                onClick={initiatePayment}
                disabled={!form.phone_number || !validatePhoneNumber(form.phone_number)}
                className="w-full bg-green-600 hover:bg-green-700 text-sm sm:text-base"
                size="lg"
              >
                <Smartphone className="w-4 h-4 sm:w-5 sm:h-5 mr-2" />
                Pay KES {totalAmount.toLocaleString()} with M-Pesa
              </Button>
            </div>
          )}

          {/* Payment Status */}
          {paymentState.status !== 'idle' && (
            <div className="space-y-4">
              {paymentState.status === 'initiating' && (
                <Alert>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  <AlertDescription>
                    {paymentState.message}
                  </AlertDescription>
                </Alert>
              )}

              {paymentState.status === 'processing' && (
                <Alert>
                  <Smartphone className="h-4 w-4" />
                  <AlertDescription>
                    <div className="space-y-2">
                      <p className="font-medium">{paymentState.message}</p>
                      <div className="bg-yellow-50 border border-yellow-200 rounded p-3 text-sm">
                        <div className="flex items-center gap-2 mb-2">
                          <Phone className="w-4 h-4 text-yellow-600" />
                          <span className="font-medium text-yellow-800">Check your phone now!</span>
                        </div>
                        <p className="text-yellow-700">
                          Look for the M-Pesa notification on <strong>{form.phone_number}</strong> and enter your PIN to complete the payment.
                        </p>
                      </div>
                      {pollingDelay > 0 ? (
                        <p className="text-sm text-muted-foreground">
                          <Clock className="w-4 h-4 inline mr-1" />
                          Checking payment status in {pollingDelay} seconds...
                        </p>
                      ) : (
                        <p className="text-sm text-muted-foreground">
                          <Loader2 className="w-4 h-4 inline mr-1 animate-spin" />
                          Waiting for payment confirmation...
                        </p>
                      )}
                      {timeRemaining > 0 && (
                        <p className="text-sm text-orange-600">
                          Time remaining: {formatTime(timeRemaining)}
                        </p>
                      )}
                    </div>
                  </AlertDescription>
                </Alert>
              )}

              {paymentState.status === 'completed' && (
                <Alert className="border-green-200 bg-green-50">
                  <CheckCircle className="h-4 w-4 text-green-600" />
                  <AlertDescription>
                    <div className="space-y-2">
                      <p className="font-medium text-green-800">{paymentState.message}</p>
                      <p className="text-green-700 text-sm">
                        Your team registration is now complete! Proceed to finalize your team setup.
                      </p>
                    </div>
                  </AlertDescription>
                </Alert>
              )}

              {paymentState.status === 'failed' && (
                <Alert variant="destructive" className='bg-red-50 border-red-200'>
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>
                    <div className="space-y-3 text-red-800">
                      <p className="font-medium">{paymentState.message}</p>
                      {paymentState.error && (
                        <div className="text-sm">
                          <p className="font-medium mb-1">Troubleshooting tips:</p>
                          <ul className="list-disc list-inside space-y-1 text-xs">
                            <li>Ensure you have sufficient M-Pesa balance</li>
                            <li>Check that your phone number is correct</li>
                            <li>Make sure your M-Pesa PIN is correct</li>
                            <li>Try again in a few minutes if the service is busy</li>
                          </ul>
                        </div>
                      )}
                      <Button 
                        onClick={retryPayment}
                        variant="outline"
                        size="sm"
                        className="mt-2"
                      >
                        Try Again
                      </Button>
                    </div>
                  </AlertDescription>
                </Alert>
              )}
            </div>
          )}

          {/* Security Notice */}
          <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
            <div className="flex items-start gap-3">
              <Shield className="w-5 h-5 text-gray-600 mt-0.5" />
              <div className="text-sm text-gray-600">
                <p className="font-medium mb-1">🔒 Secure Payment</p>
                <p>
                  Your payment is processed securely through Safaricom M-Pesa. 
                  We don't store your payment information. All transactions are encrypted and protected.
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default MPesaPaymentStep;