import React, { useState } from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Crown, Search, User, Mail, UserPlus } from 'lucide-react';

interface CaptainSelectionStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  users: any[];
}

const CaptainSelectionStep: React.FC<CaptainSelectionStepProps> = ({
  form,
  errors,
  onInputChange,
  users,
}) => {
  const [searchTerm, setSearchTerm] = useState('');

  // Filter users based on search term
  const filteredUsers = users.filter(user => 
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const selectedCaptain = users.find(user => user.id === parseInt(form.captain_id));

  return (
    <div className="space-y-6">
      <Card className="p-6">
        

        <div className="space-y-4">
          {/* Search Input */}
          <div>
            <Label htmlFor="captain_search">Search Users</Label>
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <Input
                id="captain_search"
                placeholder="Search by name or email..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
          </div>

          {/* Captain Selection */}
          <div>
            <Label htmlFor="captain_id">Select Team Captain *</Label>
            <Select value={form.captain_id} onValueChange={(value) => onInputChange('captain_id', value)}>
              <SelectTrigger className={errors.captain_id ? 'border-red-500' : ''}>
                <SelectValue placeholder="Choose a captain" />
              </SelectTrigger>
              <SelectContent>
                {filteredUsers.map(user => (
                  <SelectItem key={user.id} value={String(user.id)}>
                    <div className="flex items-center gap-2">
                      <User className="w-4 h-4" />
                      <div>
                        <div className="font-medium">{user.name}</div>
                        <div className="text-xs text-gray-500">{user.email}</div>
                      </div>
                    </div>
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {errors.captain_id && <p className="text-red-500 text-sm mt-1">{errors.captain_id}</p>}
          </div>

          {/* Selected Captain Preview */}
          {selectedCaptain && (
            <Card className="p-4 bg-yellow-50 border-yellow-200">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center">
                  <Crown className="w-5 h-5 text-yellow-600" />
                </div>
                <div>
                  <h4 className="font-medium text-yellow-800">{selectedCaptain.name}</h4>
                  <div className="flex items-center gap-4 text-sm text-yellow-600">
                    <div className="flex items-center gap-1">
                      <Mail className="w-3 h-3" />
                      {selectedCaptain.email}
                    </div>
                  </div>
                </div>
                <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
                  Team Captain
                </Badge>
              </div>
            </Card>
          )}

          {/* No Users Found */}
          {filteredUsers.length === 0 && searchTerm && (
            <Card className="p-6 text-center border-dashed">
              <User className="w-12 h-12 text-gray-300 mx-auto mb-3" />
              <p className="text-gray-500 mb-3">No users found matching "{searchTerm}"</p>
              <p className="text-sm text-gray-400">Try searching with a different term or create a user first</p>
            </Card>
          )}

          {/* Captain Responsibilities */}
          <Card className="p-4 bg-blue-50 border-blue-200">
            <h4 className="font-medium text-blue-800 mb-2 flex items-center gap-2">
              <Crown className="w-4 h-4" />
              Captain Responsibilities
            </h4>
            <ul className="text-sm text-blue-700 space-y-1">
              <li>• Lead team communications and coordination</li>
              <li>• Represent the team in official matters</li>
              <li>• Manage team roster and player positions</li>
              <li>• Make strategic decisions during matches</li>
            </ul>
          </Card>
        </div>
      </Card>
    </div>
  );
};

export default CaptainSelectionStep;