import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Users, Upload, X, Image } from 'lucide-react';
import MediaManager from '@/components/MediaManager';

interface BasicTeamInfoStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  tournaments: any[];
}

const BasicTeamInfoStep: React.FC<BasicTeamInfoStepProps> = ({
  form,
  errors,
  onInputChange,
  tournaments,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
      
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="name">Team Name *</Label>
            <Input
              id="name"
              value={form.name}
              onChange={(e) => onInputChange('name', e.target.value)}
              className={errors.name ? 'border-red-500' : ''}
              placeholder="Enter team name"
            />
            {errors.name && <p className="text-red-500 text-sm mt-1">{errors.name}</p>}
          </div>

          <div>
            <Label htmlFor="tournament_id">Tournament *</Label>
            <Select value={form.tournament_id} onValueChange={(value) => onInputChange('tournament_id', value)}>
              <SelectTrigger className={errors.tournament_id ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Tournament" />
              </SelectTrigger>
              <SelectContent>
                {tournaments.map(tournament => (
                  <SelectItem key={tournament.id} value={String(tournament.id)}>
                    {tournament.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {errors.tournament_id && <p className="text-red-500 text-sm mt-1">{errors.tournament_id}</p>}
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="description">Team Description</Label>
            <Textarea
              id="description"
              value={form.description}
              onChange={(e) => onInputChange('description', e.target.value)}
              placeholder="Describe your team..."
              rows={3}
            />
          </div>

          <div>
            <Label htmlFor="status">Team Status *</Label>
            <Select value={form.status} onValueChange={(value) => onInputChange('status', value)}>
              <SelectTrigger className={errors.status ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="active">Active</SelectItem>
                <SelectItem value="inactive">Inactive</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="suspended">Suspended</SelectItem>
              </SelectContent>
            </Select>
            {errors.status && <p className="text-red-500 text-sm mt-1">{errors.status}</p>}
          </div>

          <div className="md:col-span-2">
            <Label className="text-sm font-medium mb-2 block">Team Logo</Label>
            <div className="space-y-3">
              {form.logo ? (
                <div className="relative inline-block">
                  <img
                    src={form.logo}
                    alt="Team logo preview"
                    className="w-24 h-24 object-cover rounded-lg border"
                  />
                  <Button
                    variant="destructive"
                    size="sm"
                    className="absolute -top-2 -right-2 rounded-full w-6 h-6 p-0"
                    onClick={() => onInputChange('logo', '')}
                  >
                    <X className="h-3 w-3" />
                  </Button>
                </div>
              ) : (
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <Image className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-500 mb-3">No team logo selected</p>
                  <MediaManager
                    trigger={
                      <Button variant="outline" size="sm">
                        <Upload className="h-4 w-4 mr-2" />
                        Select Team Logo
                      </Button>
                    }
                    onSelect={(file) => onInputChange('logo', Array.isArray(file) ? file[0]?.url : file?.url)}
                    acceptedTypes={['image/*']}
                    multiple={false}
                    collection="team-logos"
                  />
                </div>
              )}
            </div>
          </div>
        </div>
      </Card>
    </div>
  );
};

export default BasicTeamInfoStep;