import React from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { router } from '@inertiajs/react';

interface PlayersTableProps {
  players: any;
  tournaments: any[];
  users: any[];
  selectedPlayers: number[];
  onPlayerSelect: (playerId: number, checked: boolean) => void;
  onSelectAll: (checked: boolean) => void;
  onEdit: (player: any) => void;
  onDelete: (player: any) => void;
  onPageChange: (page: number) => void;
}

const PlayersTable: React.FC<PlayersTableProps> = ({
  players,
  tournaments,
  users,
  selectedPlayers,
  onPlayerSelect,
  onSelectAll,
  onEdit,
  onDelete,
  onPageChange,
}) => {
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'eliminated': return 'bg-red-100 text-red-800';
      case 'withdrawn': return 'bg-yellow-100 text-yellow-800';
      case 'suspended': return 'bg-orange-100 text-orange-800';
      case 'banned': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationColor = (status: string) => {
    switch (status) {
      case 'verified': return 'bg-green-100 text-green-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'rejected': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (players.data.length === 0) {
    return (
      <Card className="p-6">
        <div className="text-center text-gray-500 dark:text-gray-400">
          <div className="mb-4">
            <svg className="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z" />
            </svg>
          </div>
          <h3 className="text-lg font-medium">No players found</h3>
          <p className="mt-1">Try adjusting your search or filter criteria.</p>
        </div>
      </Card>
    );
  }

  return (
    <Card className="overflow-hidden">
      {/* Fixed Header */}
      <div className="bg-muted/50 border-b sticky top-0 z-10">
        <div className="overflow-x-auto">
          <table className="w-full min-w-[800px]">
            <thead>
              <tr className="border-b">
                <th className="text-left p-4 w-12">
                  <input
                    type="checkbox"
                    checked={selectedPlayers.length === players.data.length}
                    onChange={(e) => onSelectAll(e.target.checked)}
                    className="rounded border-gray-300"
                  />
                </th>
                <th className="text-left p-4 font-medium">Player</th>
                <th className="text-left p-4 font-medium">Type & Level</th>
                <th className="text-left p-4 font-medium">Rating</th>
                <th className="text-left p-4 font-medium">Status</th>
                <th className="text-left p-4 font-medium">Verification</th>
                <th className="text-left p-4 font-medium">Tournament</th>
                <th className="text-left p-4 font-medium">User</th>
                <th className="text-left p-4 font-medium w-32">Actions</th>
              </tr>
            </thead>
          </table>
        </div>
      </div>

      {/* Scrollable Body */}
      <div className="overflow-x-auto max-h-[600px] overflow-y-auto">
        <table className="w-full min-w-[800px]">
          <tbody>
            {players.data.map((player: any) => (
              <tr key={player.id} className="border-b hover:bg-muted/25 transition-colors">
                <td className="p-4 w-12">
                  <input
                    type="checkbox"
                    checked={selectedPlayers.includes(player.id)}
                    onChange={(e) => onPlayerSelect(player.id, e.target.checked)}
                    className="rounded border-gray-300"
                  />
                </td>
                <td className="p-4">
                  <div className="flex items-center gap-3">
                    <div className="w-10 h-10 rounded-full overflow-hidden border-2 border-primary/20">
                      {player.avatar ? (
                        <img 
                          src={player.avatar} 
                          alt={player.display_name}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <div className="w-full h-full bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white text-sm font-bold">
                          {player.display_name.charAt(0).toUpperCase()}
                        </div>
                      )}
                    </div>
                    <div>
                      <div className="font-medium">{player.user.name}</div>
                      {player.country && (
                        <div className="text-sm text-muted-foreground">{player.country}</div>
                      )}
                    </div>
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <Badge variant="outline" className="text-xs">
                      {player.player_type?.replace('_', ' ').toUpperCase()}
                    </Badge>
                    <div className="text-sm text-muted-foreground capitalize">
                      {player.experience_level}
                    </div>
                  </div>
                </td>
                <td className="p-4">
                  <div className="space-y-1">
                    <div className="font-medium">{player.global_rating || 'N/A'}</div>
                    <div className="text-sm text-muted-foreground">
                      Win Rate: {player.tournament_win_rate || 0}%
                    </div>
                  </div>
                </td>
                <td className="p-4">
                  <Badge className={`text-xs ${getStatusColor(player.status)}`}>
                    {player.status}
                  </Badge>
                </td>
                <td className="p-4">
                  <Badge className={`text-xs ${getVerificationColor(player.verification_status)}`}>
                    {player.verification_status}
                  </Badge>
                </td>
                <td className="p-4">
                  <div className="text-sm">
                    {tournaments.find(t => t.id === player.tournament_id)?.name || 'Unknown'}
                  </div>
                </td>
                <td className="p-4">
                  <div className="text-sm">
                    <div className="font-medium">
                      {users.find(u => u.id === player.user_id)?.name || 'Unknown'}
                    </div>
                    <div className="text-muted-foreground">
                      {users.find(u => u.id === player.user_id)?.email || ''}
                    </div>
                  </div>
                </td>
                <td className="p-4 w-32">
                  <div className="flex gap-1">
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => router.get(route('admin.players.show', { player: player.id }))}
                      className="text-xs px-2 py-1"
                    >
                      View
                    </Button>
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => onEdit(player)}
                      className="text-xs px-2 py-1"
                    >
                      Edit
                    </Button>
                    <Button 
                      size="sm" 
                      variant="destructive"
                      onClick={() => onDelete(player)}
                      className="text-xs px-2 py-1"
                    >
                      Del
                    </Button>
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {/* Fixed Footer */}
      <div className="bg-muted/50 border-t p-4 sticky bottom-0 z-10">
        <div className="flex justify-between items-center">
          <div className="text-sm text-muted-foreground">
            Showing {players.from} to {players.to} of {players.total} results
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              size="sm"
              disabled={players.current_page === 1}
              onClick={() => onPageChange(players.current_page - 1)}
            >
              Previous
            </Button>
            <div className="flex items-center gap-1">
              {Array.from({ length: Math.min(5, players.last_page) }, (_, i) => {
                const page = players.current_page - 2 + i;
                if (page < 1 || page > players.last_page) return null;
                return (
                  <Button
                    key={page}
                    variant={page === players.current_page ? "default" : "outline"}
                    size="sm"
                    onClick={() => onPageChange(page)}
                    className="w-8 h-8 p-0"
                  >
                    {page}
                  </Button>
                );
              })}
            </div>
            <Button
              variant="outline"
              size="sm"
              disabled={players.current_page === players.last_page}
              onClick={() => onPageChange(players.current_page + 1)}
            >
              Next
            </Button>
          </div>
        </div>
      </div>
    </Card>
  );
};

export default PlayersTable; 