import React from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Check, ChevronLeft, ChevronRight } from 'lucide-react';

interface Step {
  id: string;
  title: string;
  description: string;
  icon?: React.ReactNode;
}

interface PlayerFormStepsProps {
  steps: Step[];
  currentStep: number;
  onStepChange: (step: number) => void;
  onNext: () => void;
  onPrevious: () => void;
  onSubmit: () => void;
  isSubmitting?: boolean;
  canProceed?: boolean;
  isLastStep?: boolean;
  children?: React.ReactNode;
}

const PlayerFormSteps: React.FC<PlayerFormStepsProps> = ({
  steps,
  currentStep,
  onStepChange,
  onNext,
  onPrevious,
  onSubmit,
  isSubmitting = false,
  canProceed = true,
  isLastStep = false,
  children,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
        <div className="flex items-center justify-between mb-6">
          <h2 className="text-xl font-semibold">Player Information</h2>
          <Badge variant="secondary">Step {currentStep + 1} of {steps.length}</Badge>
        </div>

        <div className="flex items-center justify-between mb-8">
          {steps.map((step, index) => (
            <div key={step.id} className="flex items-center">
              <div className="flex items-center">
                <button
                  onClick={() => onStepChange(index)}
                  className={`flex items-center justify-center w-10 h-10 rounded-full border-2 transition-all ${
                    index <= currentStep
                      ? 'bg-primary text-primary-foreground border-primary'
                      : 'bg-muted text-muted-foreground border-muted'
                  } ${index < currentStep ? 'cursor-pointer hover:bg-primary/90' : ''}`}
                  disabled={index > currentStep}
                >
                  {index < currentStep ? (
                    <Check className="w-5 h-5" />
                  ) : (
                    <span className="text-sm font-medium">{index + 1}</span>
                  )}
                </button>
                <div className="ml-3">
                  <p className="text-sm font-medium">{step.title}</p>
                  <p className="text-xs text-muted-foreground">{step.description}</p>
                </div>
              </div>
              {index < steps.length - 1 && (
                <div
                  className={`w-16 h-0.5 mx-4 ${
                    index < currentStep ? 'bg-primary' : 'bg-muted'
                  }`}
                />
              )}
            </div>
          ))}
        </div>

        
        
        {/* Form Content */}
        <div className="mb-1">
          {children}
        </div>

        <div className="flex justify-between pt-6 border-t">
          <Button
            variant="outline"
            onClick={onPrevious}
            disabled={currentStep === 0}
            className="flex items-center gap-2"
          >
            <ChevronLeft className="w-4 h-4" />
            Previous
          </Button>

          <div className="flex gap-2">
            {!isLastStep ? (
              <Button
                onClick={onNext}
                disabled={!canProceed || isSubmitting}
                className="flex items-center gap-2"
              >
                Next
                <ChevronRight className="w-4 h-4" />
              </Button>
            ) : (
              <Button
                onClick={onSubmit}
                disabled={!canProceed || isSubmitting}
                className="flex items-center gap-2"
              >
                {isSubmitting ? 'Saving...' : 'Save Player'}
              </Button>
            )}
          </div>
        </div>
      </Card>
    </div>
  );
};

export default PlayerFormSteps; 