import React, { useState, useRef, useEffect } from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { User, UserPlus, Search, Mail, Phone, X, Check } from 'lucide-react';
import { Switch } from '@/components/ui/switch';

interface UserSelectionStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  users: any[];
}

const UserSelectionStep: React.FC<UserSelectionStepProps> = ({
  form,
  errors,
  onInputChange,
  users,
}) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [showDropdown, setShowDropdown] = useState(false);
  const [createUserForm, setCreateUserForm] = useState({
    name: '',
    email: '',
    password: '',
    password_confirmation: '',
    must_change_password: false,
    payment_status: 'pending',
    user_type: 'player',
  });
  const [createUserErrors, setCreateUserErrors] = useState<any>({});
  const [isCreatingUser, setIsCreatingUser] = useState(false);
  const [activeTab, setActiveTab] = useState('existing');
  const dropdownRef = useRef<HTMLDivElement>(null);

  // Filter users based on search term
  const filteredUsers = users.filter(user => 
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setShowDropdown(false);
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const handleCreateUser = async () => {
    // Basic validation
    const newErrors: any = {};
    if (!createUserForm.name) newErrors.name = 'Name is required';
    if (!createUserForm.email) newErrors.email = 'Email is required';
    if (!createUserForm.password) newErrors.password = 'Password is required';
    if (createUserForm.password !== createUserForm.password_confirmation) {
      newErrors.password_confirmation = 'Passwords do not match';
    }

    if (Object.keys(newErrors).length > 0) {
      setCreateUserErrors(newErrors);
      return;
    }

    setIsCreatingUser(true);
    try {
      const response = await fetch(route('admin.users.store'), {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify(createUserForm),
      });

      const data = await response.json();

      if (response.ok && data.success) {
        // Add the new user to the list and select it
        const newUser = data.user;
        users.push(newUser);
        onInputChange('user_id', String(newUser.id));
        setActiveTab('existing');
        setSearchTerm('');
        setShowDropdown(false);
        setCreateUserForm({ 
          name: '', 
          email: '', 
          password: '', 
          password_confirmation: '', 
          must_change_password: false, 
          payment_status: 'pending', 
          user_type: 'player' 
        });
        setCreateUserErrors({});
      } else {
        setCreateUserErrors(data.errors || { general: data.message || 'Failed to create user' });
      }
    } catch (error) {
      setCreateUserErrors({ general: 'Failed to create user. Please try again.' });
    } finally {
      setIsCreatingUser(false);
    }
  };

  const selectedUser = users.find(user => user.id === parseInt(form.user_id));

  return (
    <div className="space-y-6">
      <Card className="p-6">
        <div className="mb-6">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <User className="w-5 h-5" />
            User Selection
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            Select an existing user or create a new one for this player
          </p>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="existing" className="flex items-center gap-2">
              <Search className="w-4 h-4" />
              Select Existing User
            </TabsTrigger>
            <TabsTrigger value="create" className="flex items-center gap-2">
              <UserPlus className="w-4 h-4" />
              Create New User
            </TabsTrigger>
          </TabsList>

          <TabsContent value="existing" className="space-y-4">
            {/* Search and Select User */}
            <div ref={dropdownRef}>
              <Label htmlFor="user_search">Search and Select User *</Label>
              <div className="relative">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4 z-10" />
                  <Input
                    id="user_search"
                    placeholder="Search by name or email..."
                    value={searchTerm}
                    onChange={(e) => {
                      setSearchTerm(e.target.value);
                      setShowDropdown(true);
                    }}
                    onFocus={() => setShowDropdown(true)}
                    className={`pl-10 ${errors.user_id ? 'border-red-500' : ''}`}
                  />
                  {searchTerm && (
                    <button
                      type="button"
                      onClick={() => {
                        setSearchTerm('');
                        setShowDropdown(false);
                      }}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  )}
                </div>

                {/* Dropdown Results */}
                {showDropdown && (
                  <div className="absolute z-50 w-full mt-1 bg-white border rounded-md shadow-lg max-h-60 overflow-y-auto">
                    {filteredUsers.length > 0 ? (
                      <div className="py-1">
                        {filteredUsers.map(user => (
                          <button
                            key={user.id}
                            type="button"
                            onClick={() => {
                              onInputChange('user_id', String(user.id));
                              setSearchTerm('');
                              setShowDropdown(false);
                            }}
                            className={`w-full px-4 py-3 text-left hover:bg-gray-50 transition-colors ${
                              form.user_id === String(user.id) ? 'bg-green-50' : ''
                            }`}
                          >
                            <div className="flex items-center gap-3">
                              <div className="w-8 h-8 bg-primary/10 rounded-full flex items-center justify-center flex-shrink-0">
                                <User className="w-4 h-4 text-primary" />
                              </div>
                              <div className="flex-1 min-w-0">
                                <div className="font-medium text-gray-900 truncate">{user.name}</div>
                                <div className="text-xs text-gray-500 truncate flex items-center gap-1">
                                  <Mail className="w-3 h-3" />
                                  {user.email}
                                </div>
                              </div>
                              {form.user_id === String(user.id) && (
                                <Check className="w-4 h-4 text-green-600 flex-shrink-0" />
                              )}
                            </div>
                          </button>
                        ))}
                      </div>
                    ) : (
                      <div className="p-6 text-center">
                        <User className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                        <p className="text-gray-500 mb-3">
                          {searchTerm ? `No users found matching "${searchTerm}"` : 'Start typing to search users'}
                        </p>
                        {searchTerm && (
                          <Button 
                            variant="outline"
                            size="sm"
                            onClick={() => {
                              setActiveTab('create');
                              setShowDropdown(false);
                              setCreateUserForm(prev => ({ 
                                ...prev, 
                                name: searchTerm.includes('@') ? '' : searchTerm, 
                                email: searchTerm.includes('@') ? searchTerm : '' 
                              }));
                            }}
                          >
                            <UserPlus className="w-4 h-4 mr-2" />
                            Create New User
                          </Button>
                        )}
                      </div>
                    )}
                  </div>
                )}
              </div>
              {errors.user_id && <p className="text-red-500 text-sm mt-1">{errors.user_id}</p>}
            </div>

            {/* Selected User Preview */}
            {selectedUser && (
              <Card className="p-4 bg-green-50 border-green-200">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                    <User className="w-5 h-5 text-green-600" />
                  </div>
                  <div className="flex-1">
                    <h4 className="font-medium text-green-800">{selectedUser.name}</h4>
                    <div className="flex items-center gap-4 text-sm text-green-600">
                      <div className="flex items-center gap-1">
                        <Mail className="w-3 h-3" />
                        {selectedUser.email}
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    <Badge variant="secondary" className="bg-green-100 text-green-800">
                      <Check className="w-3 h-3 mr-1" />
                      Selected
                    </Badge>
                    <button
                      type="button"
                      onClick={() => onInputChange('user_id', '')}
                      className="text-green-600 hover:text-green-800 p-1 hover:bg-green-100 rounded"
                      title="Clear selection"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="create" className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="new_user_name">Full Name *</Label>
                <Input
                  id="new_user_name"
                  value={createUserForm.name}
                  onChange={(e) => setCreateUserForm(prev => ({ ...prev, name: e.target.value }))}
                  placeholder="Enter full name"
                  className={createUserErrors.name ? 'border-red-500' : ''}
                />
                {createUserErrors.name && <p className="text-red-500 text-sm mt-1">{createUserErrors.name}</p>}
              </div>

              <div>
                <Label htmlFor="must_change_password">Must Change Password</Label>
                <div className="flex items-center gap-2">
                  <Switch
                    checked={createUserForm.must_change_password}
                    onCheckedChange={(checked) => setCreateUserForm(prev => ({ ...prev, must_change_password: checked }))}
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="payment_status">Payment Status *</Label>
                <Select
                  value={createUserForm.payment_status}
                  onValueChange={(value) => setCreateUserForm(prev => ({ ...prev, payment_status: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select payment status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="pending">Pending</SelectItem>
                    <SelectItem value="partial">Partial</SelectItem>
                    <SelectItem value="completed">Completed</SelectItem>
                    <SelectItem value="not_required">Not Required</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="new_user_email">Email Address *</Label>
                <Input
                  id="new_user_email"
                  type="email"
                  value={createUserForm.email}
                  onChange={(e) => setCreateUserForm(prev => ({ ...prev, email: e.target.value }))}
                  placeholder="user@example.com"
                  className={createUserErrors.email ? 'border-red-500' : ''}
                />
                {createUserErrors.email && <p className="text-red-500 text-sm mt-1">{createUserErrors.email}</p>}
              </div>

              <div>
                <Label htmlFor="new_user_password">Password *</Label>
                <Input
                  id="new_user_password"
                  type="password"
                  value={createUserForm.password}
                  onChange={(e) => setCreateUserForm(prev => ({ ...prev, password: e.target.value }))}
                  placeholder="Enter password"
                  className={createUserErrors.password ? 'border-red-500' : ''}
                />
                {createUserErrors.password && <p className="text-red-500 text-sm mt-1">{createUserErrors.password}</p>}
              </div>

              <div>
                <Label htmlFor="new_user_password_confirmation">Confirm Password *</Label>
                <Input
                  id="new_user_password_confirmation"
                  type="password"
                  value={createUserForm.password_confirmation}
                  onChange={(e) => setCreateUserForm(prev => ({ ...prev, password_confirmation: e.target.value }))}
                  placeholder="Confirm password"
                  className={createUserErrors.password_confirmation ? 'border-red-500' : ''}
                />
                {createUserErrors.password_confirmation && <p className="text-red-500 text-sm mt-1">{createUserErrors.password_confirmation}</p>}
              </div>
            </div>

            {createUserErrors.general && (
              <div className="p-3 bg-red-50 border border-red-200 rounded-md">
                <p className="text-red-600 text-sm">{createUserErrors.general}</p>
              </div>
            )}

            <div className="flex justify-end">
              <Button 
                type="button" 
                onClick={handleCreateUser}
                disabled={isCreatingUser}
                className="min-w-[140px]"
              >
                <UserPlus className="w-4 h-4 mr-2" />
                {isCreatingUser ? 'Creating...' : 'Create User'}
              </Button>
            </div>
          </TabsContent>
        </Tabs>
      </Card>
    </div>
  );
};

export default UserSelectionStep;