import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Trophy } from 'lucide-react';

interface GamingProfileStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
  tournaments: any[];
}

const GamingProfileStep: React.FC<GamingProfileStepProps> = ({
  form,
  errors,
  onInputChange,
  tournaments,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
        <div className="mb-6">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Trophy className="w-5 h-5" />
            Gaming Profile
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            Tournament and gaming-related information
          </p>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="tournament_id">Tournament *</Label>
            <Select value={form.tournament_id} onValueChange={(value) => onInputChange('tournament_id', value)}>
              <SelectTrigger className={errors.tournament_id ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Tournament" />
              </SelectTrigger>
              <SelectContent>
                {tournaments.map(tournament => (
                  <SelectItem key={tournament.id} value={String(tournament.id)}>
                    {tournament.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {errors.tournament_id && <p className="text-red-500 text-sm mt-1">{errors.tournament_id}</p>}
          </div>

          <div>
            <Label htmlFor="player_type">Player Type *</Label>
            <Select value={form.player_type} onValueChange={(value) => onInputChange('player_type', value)}>
              <SelectTrigger className={errors.player_type ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Player Type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="amateur">Amateur</SelectItem>
                <SelectItem value="semi_pro">Semi Pro</SelectItem>
                <SelectItem value="professional">Professional</SelectItem>
                <SelectItem value="legend">Legend</SelectItem>
              </SelectContent>
            </Select>
            {errors.player_type && <p className="text-red-500 text-sm mt-1">{errors.player_type}</p>}
          </div>

          <div>
            <Label htmlFor="experience_level">Experience Level *</Label>
            <Select value={form.experience_level} onValueChange={(value) => onInputChange('experience_level', value)}>
              <SelectTrigger className={errors.experience_level ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Experience Level" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="beginner">Beginner</SelectItem>
                <SelectItem value="intermediate">Intermediate</SelectItem>
                <SelectItem value="advanced">Advanced</SelectItem>
                <SelectItem value="expert">Expert</SelectItem>
              </SelectContent>
            </Select>
            {errors.experience_level && <p className="text-red-500 text-sm mt-1">{errors.experience_level}</p>}
          </div>

          <div>
            <Label htmlFor="primary_game">Primary Game</Label>
            <Input
              id="primary_game"
              value={form.primary_game}
              onChange={(e) => onInputChange('primary_game', e.target.value)}
              placeholder="e.g., League of Legends"
            />
          </div>

          <div>
            <Label htmlFor="global_rating">Global Rating</Label>
            <Input
              id="global_rating"
              type="number"
              value={form.global_rating}
              onChange={(e) => onInputChange('global_rating', e.target.value)}
              placeholder="800-2500"
              min="800"
              max="2500"
            />
          </div>

          <div>
            <Label htmlFor="tournament_rating">Tournament Rating</Label>
            <Input
              id="tournament_rating"
              type="number"
              value={form.tournament_rating}
              onChange={(e) => onInputChange('tournament_rating', e.target.value)}
              placeholder="800-2500"
              min="800"
              max="2500"
            />
          </div>

          <div>
            <Label htmlFor="jersey_number">Jersey Number</Label>
            <Input
              id="jersey_number"
              type="number"
              value={form.jersey_number}
              onChange={(e) => onInputChange('jersey_number', e.target.value)}
              placeholder="1, 2, 3..."
              min="1"
            />
          </div>

          <div>
            <Label htmlFor="football_position">
              Football Position <span className="text-red-500">*</span>
            </Label>
            <Select value={form.football_position} onValueChange={(value) => onInputChange('football_position', value)}>
              <SelectTrigger className={!form.football_position || errors.football_position ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Football Position" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="Goalkeeper">Goalkeeper</SelectItem>
                <SelectItem value="Defender">Defender</SelectItem>
                <SelectItem value="Midfielder">Midfielder</SelectItem>
                <SelectItem value="Forward">Forward</SelectItem>
              </SelectContent>
            </Select>
            {errors.football_position && <p className="text-red-500 text-sm mt-1">{errors.football_position}</p>}
            {!form.football_position && !errors.football_position && (
              <p className="text-red-500 text-sm mt-1">Required for approval</p>
            )}
          </div>

          <div>
            <Label htmlFor="status">Status *</Label>
            <Select value={form.status} onValueChange={(value) => onInputChange('status', value)}>
              <SelectTrigger className={errors.status ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="registered">Registered</SelectItem>
                <SelectItem value="active">Active</SelectItem>
                <SelectItem value="eliminated">Eliminated</SelectItem>
                <SelectItem value="withdrawn">Withdrawn</SelectItem>
                <SelectItem value="suspended">Suspended</SelectItem>
                <SelectItem value="banned">Banned</SelectItem>
              </SelectContent>
            </Select>
            {errors.status && <p className="text-red-500 text-sm mt-1">{errors.status}</p>}
          </div>

          <div>
            <Label htmlFor="verification_status">Verification Status *</Label>
            <Select value={form.verification_status} onValueChange={(value) => onInputChange('verification_status', value)}>
              <SelectTrigger className={errors.verification_status ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select Verification Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="unverified">Unverified</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="verified">Verified</SelectItem>
                <SelectItem value="rejected">Rejected</SelectItem>
              </SelectContent>
            </Select>
            {errors.verification_status && <p className="text-red-500 text-sm mt-1">{errors.verification_status}</p>}
          </div>
        </div>
      </Card>
    </div>
  );
};

export default GamingProfileStep; 