import React from 'react';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';
import { Users, Upload, X, Image } from 'lucide-react';
import MediaManager from '@/components/MediaManager';

interface BasicInfoStepProps {
  form: any;
  errors: any;
  onInputChange: (field: string, value: any) => void;
}

const BasicInfoStep: React.FC<BasicInfoStepProps> = ({
  form,
  errors,
  onInputChange,
}) => {
  return (
    <div className="space-y-6">
      <Card className="p-6">
      <div className="mb-6">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Users className="w-5 h-5" />
            Basic Information
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            Personal details and user association
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="display_name">Display Name *</Label>
            <Input
              id="display_name"
              value={form.display_name}
              onChange={(e) => onInputChange('display_name', e.target.value)}
              className={errors.display_name ? 'border-red-500' : ''}
              placeholder="Enter display name"
            />
            {errors.display_name && <p className="text-red-500 text-sm mt-1">{errors.display_name}</p>}
          </div>



          <div className="md:col-span-2">
            <Label htmlFor="bio">Bio</Label>
            <Textarea
              id="bio"
              value={form.bio}
              onChange={(e) => onInputChange('bio', e.target.value)}
              placeholder="Tell us about yourself..."
              rows={3}
            />
          </div>

          <div>
            <Label htmlFor="country">Country</Label>
            <Input
              id="country"
              value={form.country}
              onChange={(e) => onInputChange('country', e.target.value)}
              placeholder="e.g., Kenya"
            />
          </div>

          <div>
            <Label htmlFor="city">City</Label>
            <Input
              id="city"
              value={form.city}
              onChange={(e) => onInputChange('city', e.target.value)}
              placeholder="e.g., Kakamega"
            />
          </div>

          <div>
            <Label htmlFor="date_of_birth">
              Date of Birth <span className="text-red-500">*</span>
            </Label>
            <Input
              id="date_of_birth"
              type="date"
              value={form.date_of_birth}
              onChange={(e) => onInputChange('date_of_birth', e.target.value)}
              className={!form.date_of_birth ? 'border-red-500' : ''}
              required
            />
            {!form.date_of_birth && (
              <p className="text-red-500 text-sm mt-1">Required for approval</p>
            )}
          </div>

          <div>
            <Label htmlFor="gender">
              Gender <span className="text-red-500">*</span>
            </Label>
            <Select value={form.gender} onValueChange={(value) => onInputChange('gender', value)}>
              <SelectTrigger className={!form.gender ? 'border-red-500' : ''}>
                <SelectValue placeholder="Select gender" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="male">Male</SelectItem>
                <SelectItem value="female">Female</SelectItem>
                <SelectItem value="other">Other</SelectItem>
                <SelectItem value="prefer_not_to_say">Prefer not to say</SelectItem>
              </SelectContent>
            </Select>
            {!form.gender && (
              <p className="text-red-500 text-sm mt-1">Required for approval</p>
            )}
          </div>

          <div className="md:col-span-2">
            <Label className="text-sm font-medium mb-2 block">Profile Image</Label>
            <div className="space-y-3">
              {form.profile_image ? (
                <div className="relative inline-block">
                  <img
                    src={form.profile_image}
                    alt="Profile preview"
                    className="w-24 h-24 object-cover rounded-full border"
                  />
                  <Button
                    variant="destructive"
                    size="sm"
                    className="absolute -top-2 -right-2 rounded-full w-6 h-6 p-0"
                    onClick={() => onInputChange('profile_image', '')}
                  >
                    <X className="h-3 w-3" />
                  </Button>
                </div>
              ) : (
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <Image className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-500 mb-3">No profile image selected</p>
                  <MediaManager
                    trigger={
                      <Button variant="outline" size="sm">
                        <Upload className="h-4 w-4 mr-2" />
                        Select Profile Image
                      </Button>
                    }
                    onSelect={(file) => onInputChange('profile_image', Array.isArray(file) ? file[0]?.url : file?.url)}
                    acceptedTypes={['image/*']}
                    multiple={false}
                    collection="player-profiles"
                  />
                </div>
              )}
            </div>
          </div>
        </div>
      </Card>
    </div>
  );
};

export default BasicInfoStep; 