import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Trophy, Users, Gamepad, User, Shield, Award, Target } from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';

interface WelcomeFlowProps {
  onComplete: () => void;
}

export function WelcomeFlow({ onComplete }: WelcomeFlowProps) {
  const { isAdmin, isOrganizer, isReferee, isPlayer } = usePermissions();
  const [currentStep, setCurrentStep] = useState(0);

  // Determine primary role
  const getPrimaryRole = () => {
    if (isAdmin()) return 'admin';
    if (isOrganizer()) return 'organizer';
    if (isReferee()) return 'referee';
    if (isPlayer()) return 'player';
    return 'user';
  };

  const primaryRole = getPrimaryRole();

  const roleConfig = {
    admin: {
      title: 'System Administrator',
      description: 'You have full access to manage the tournament system',
      icon: Shield,
      color: 'text-green-600',
      bgColor: 'bg-green-50',
      features: [
        'Manage all tournaments and users',
        'Configure system settings',
        'View analytics and reports',
        'Assign roles and permissions'
      ]
    },
    organizer: {
      title: 'Tournament Organizer',
      description: 'You can create and manage tournaments',
      icon: Award,
      color: 'text-blue-600',
      bgColor: 'bg-blue-50',
      features: [
        'Create and manage tournaments',
        'Add players and teams',
        'Schedule matches',
        'View tournament analytics'
      ]
    },
    referee: {
      title: 'Match Referee',
      description: 'You can officiate matches and manage scores',
      icon: Target,
      color: 'text-orange-600',
      bgColor: 'bg-orange-50',
      features: [
        'View assigned matches',
        'Update match scores',
        'Manage match schedules',
        'Submit match reports'
      ]
    },
    player: {
      title: 'Tournament Player',
      description: 'You can participate in tournaments',
      icon: User,
      color: 'text-purple-600',
      bgColor: 'bg-purple-50',
      features: [
        'View your tournaments',
        'Check match schedules',
        'Track your performance',
        'Update your profile'
      ]
    },
    user: {
      title: 'Tournament User',
      description: 'You can view tournament information',
      icon: Trophy,
      color: 'text-gray-600',
      bgColor: 'bg-gray-50',
      features: [
        'View active tournaments',
        'Browse match results',
        'Access public information'
      ]
    }
  };

  const config = roleConfig[primaryRole as keyof typeof roleConfig];

  const steps = [
    {
      title: 'Welcome to The Milimus Tournament',
      description: 'Let\'s get you started with your role-specific features',
      content: (
        <div className="text-center space-y-4">
          <div className={`inline-flex p-4 rounded-full ${config.bgColor}`}>
            <config.icon className={`w-12 h-12 ${config.color}`} />
          </div>
          <h2 className="text-2xl font-bold">{config.title}</h2>
          <p className="text-muted-foreground">{config.description}</p>
        </div>
      )
    },
    {
      title: 'Your Features',
      description: 'Here\'s what you can do with your role:',
      content: (
        <div className="space-y-4">
          <ul className="space-y-2">
            {config.features.map((feature, index) => (
              <li key={index} className="flex items-center gap-2">
                <div className={`w-2 h-2 rounded-full ${config.color.replace('text-', 'bg-')}`} />
                <span>{feature}</span>
              </li>
            ))}
          </ul>
        </div>
      )
    },
    {
      title: 'Quick Start',
      description: 'Ready to begin? Here are some quick actions:',
      content: (
        <div className="space-y-4">
          <div className="grid gap-3">
            {primaryRole === 'admin' && (
              <>
                <Button variant="outline" className="justify-start">
                  <Shield className="w-4 h-4 mr-2" />
                  Manage Users
                </Button>
                <Button variant="outline" className="justify-start">
                  <Trophy className="w-4 h-4 mr-2" />
                  View All Tournaments
                </Button>
              </>
            )}
            {primaryRole === 'organizer' && (
              <>
                <Button variant="outline" className="justify-start">
                  <Trophy className="w-4 h-4 mr-2" />
                  Create Tournament
                </Button>
                <Button variant="outline" className="justify-start">
                  <Users className="w-4 h-4 mr-2" />
                  Add Players
                </Button>
              </>
            )}
            {primaryRole === 'referee' && (
              <>
                <Button variant="outline" className="justify-start">
                  <Gamepad className="w-4 h-4 mr-2" />
                  View Matches
                </Button>
                <Button variant="outline" className="justify-start">
                  <Target className="w-4 h-4 mr-2" />
                  Match Schedule
                </Button>
              </>
            )}
            {primaryRole === 'player' && (
              <>
                <Button variant="outline" className="justify-start">
                  <Trophy className="w-4 h-4 mr-2" />
                  My Tournaments
                </Button>
                <Button variant="outline" className="justify-start">
                  <Gamepad className="w-4 h-4 mr-2" />
                  My Matches
                </Button>
              </>
            )}
            <Button variant="outline" className="justify-start">
              <User className="w-4 h-4 mr-2" />
              View Dashboard
            </Button>
          </div>
        </div>
      )
    }
  ];

  const handleNext = () => {
    if (currentStep < steps.length - 1) {
      setCurrentStep(currentStep + 1);
    } else {
      onComplete();
    }
  };

  const handleSkip = () => {
    onComplete();
  };

  return (
    <div className="fixed inset-0 bg-background/80 backdrop-blur-sm z-50 flex items-center justify-center p-4">
      <Card className="w-full max-w-md">
        <CardHeader className="text-center">
          <CardTitle>{steps[currentStep].title}</CardTitle>
          <CardDescription>{steps[currentStep].description}</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {steps[currentStep].content}
          
          <div className="flex justify-between items-center">
            <div className="flex gap-1">
              {steps.map((_, index) => (
                <div
                  key={index}
                  className={`w-2 h-2 rounded-full ${
                    index === currentStep ? 'bg-primary' : 'bg-muted'
                  }`}
                />
              ))}
            </div>
            
            <div className="flex gap-2">
              <Button variant="outline" onClick={handleSkip}>
                Skip
              </Button>
              <Button onClick={handleNext}>
                {currentStep === steps.length - 1 ? 'Get Started' : 'Next'}
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 