import { 
    SidebarGroup, 
    SidebarGroupLabel, 
    SidebarMenu, 
    SidebarMenuButton, 
    SidebarMenuItem,
    SidebarMenuSub,
    SidebarMenuSubButton,
    SidebarMenuSubItem
} from '@/components/ui/sidebar';
import { type NavItem } from '@/types';
import { Link, usePage } from '@inertiajs/react';
import { ChevronRight } from 'lucide-react';
import { useState } from 'react';

export function NavMain({ items = [] }: { items: NavItem[] }) {
    const page = usePage();
    const [expandedItems, setExpandedItems] = useState<string[]>([]);
    
    const isActiveRoute = (href: string) => {
        return page.url.startsWith(href);
    };

    const toggleExpanded = (itemTitle: string) => {
        setExpandedItems(prev => 
            prev.includes(itemTitle) 
                ? prev.filter(title => title !== itemTitle)
                : [...prev, itemTitle]
        );
    };
    
    const renderNavItem = (item: NavItem) => {
        const isActive = isActiveRoute(item.href);
        const hasChildren = item.children && item.children.length > 0;
        const isExpanded = expandedItems.includes(item.title);
        
        if (hasChildren) {
            return (
                <SidebarMenuItem key={item.title}>
                    <SidebarMenuSub>
                        <SidebarMenuSubButton 
                            onClick={() => toggleExpanded(item.title)}
                            className="cursor-pointer"
                        >
                            {item.icon && <item.icon />}
                            <span>{item.title}</span>
                            <ChevronRight 
                                className={`transition-transform ${
                                    isExpanded ? 'rotate-90' : ''
                                }`} 
                            />
                        </SidebarMenuSubButton>
                        {isExpanded && item.children!.map((child) => (
                            <SidebarMenuSubItem key={child.title}>
                                <SidebarMenuSubButton asChild isActive={isActiveRoute(child.href)}>
                                    <Link href={child.href} prefetch>
                                        {child.icon && <child.icon />}
                                        <span>{child.title}</span>
                                    </Link>
                                </SidebarMenuSubButton>
                            </SidebarMenuSubItem>
                        ))}
                    </SidebarMenuSub>
                </SidebarMenuItem>
            );
        }
        
        return (
            <SidebarMenuItem key={item.title}>
                <SidebarMenuButton 
                    asChild 
                    isActive={isActive} 
                    tooltip={{ children: item.title }}
                >
                    <Link href={item.href} prefetch>
                        {item.icon && <item.icon />}
                        <span>{item.title}</span>
                    </Link>
                </SidebarMenuButton>
            </SidebarMenuItem>
        );
    };
    
    return (
        <SidebarGroup className="px-2 py-0">
            <SidebarGroupLabel>Platform</SidebarGroupLabel>
            <SidebarMenu>
                {items.length === 0 ? (
                    <SidebarMenuItem>
                        <SidebarMenuButton disabled>
                            <span>No navigation items</span>
                        </SidebarMenuButton>
                    </SidebarMenuItem>
                ) : (
                    items.map(renderNavItem)
                )}
            </SidebarMenu>
        </SidebarGroup>
    );
}
