import React from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { 
  Table, 
  TableBody, 
  TableCell, 
  TableHead, 
  TableHeader, 
  TableRow 
} from '@/components/ui/table';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger 
} from '@/components/ui/dropdown-menu';
import { 
  MoreHorizontal, 
  Eye, 
  Edit, 
  Trash2, 
  Play, 
  Pause, 
  CheckCircle,
  Clock,
  Users,
  Trophy
} from 'lucide-react';
import { router } from '@inertiajs/react';

interface Match {
  id: number;
  name: string;
  match_code: string;
  match_type: string;
  format: string;
  best_of_games: number;
  tournament_id: number;
  bracket_id: number;
  round: number;
  match_number: number;
  player1_id: number;
  player2_id: number;
  team1_id: number;
  team2_id: number;
  player1_score: number;
  player2_score: number;
  status: string;
  progress_status: string;
  scheduled_time: string;
  started_at: string;
  completed_at: string;
  winner_id: number;
  is_streamed: boolean;
  allow_spectators: boolean;
  tournament?: {
    id: number;
    name: string;
  };
  bracket?: {
    id: number;
    name: string;
  };
  player1?: {
    id: number;
    display_name: string;
  };
  player2?: {
    id: number;
    display_name: string;
  };
  team1?: {
    id: number;
    name: string;
  };
  team2?: {
    id: number;
    name: string;
  };
  winner?: {
    id: number;
    display_name?: string;
    name?: string;
  };
}

interface MatchesTableProps {
  matches: {
    data: Match[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search: string;
    status: string;
    progress_status: string;
    match_type: string;
    format: string;
    tournament_id: string;
    bracket_id: string;
    sort_by: string;
    sort_order: string;
  };
}

const MatchesTable: React.FC<MatchesTableProps> = ({ matches, filters }) => {
  const handleFilterChange = (key: string, value: string) => {
    router.get(route('admin.matches.index'), {
      ...filters,
      [key]: value,
    }, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    handleFilterChange('sort_by', column);
    handleFilterChange('sort_order', newOrder);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getProgressStatusColor = (status: string) => {
    switch (status) {
      case 'not_started': return 'bg-gray-100 text-gray-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'paused': return 'bg-orange-100 text-orange-800';
      case 'completed': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getMatchTypeIcon = (type: string) => {
    switch (type) {
      case 'individual': return <Users className="w-4 h-4" />;
      case 'team': return <Trophy className="w-4 h-4" />;
      case 'mixed': return <Users className="w-4 h-4" />;
      default: return <Users className="w-4 h-4" />;
    }
  };

  const getParticipantName = (match: Match, participantType: 'player1' | 'player2' | 'team1' | 'team2') => {
    if (participantType.startsWith('player')) {
      const player = match[participantType as keyof Match] as any;
      return player?.display_name || 'Unknown Player';
    } else {
      const team = match[participantType as keyof Match] as any;
      return team?.name || 'Unknown Team';
    }
  };

  const handleDelete = (matchId: number) => {
    if (confirm('Are you sure you want to delete this match?')) {
      router.delete(route('admin.matches.destroy', matchId));
    }
  };

  const handleBulkDelete = (matchIds: number[]) => {
    if (confirm(`Are you sure you want to delete ${matchIds.length} matches?`)) {
      router.post(route('admin.matches.bulkDelete'), { ids: matchIds });
    }
  };

  return (
    <Card className="overflow-hidden">
      {/* Filters */}
      <div className="p-4 border-b bg-muted/50">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Input
            placeholder="Search matches..."
            value={filters.search}
            onChange={(e) => handleFilterChange('search', e.target.value)}
          />
          
          <Select value={filters.status} onValueChange={(value) => handleFilterChange('status', value)}>
            <SelectTrigger>
              <SelectValue placeholder="Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="scheduled">Scheduled</SelectItem>
              <SelectItem value="in_progress">In Progress</SelectItem>
              <SelectItem value="completed">Completed</SelectItem>
              <SelectItem value="cancelled">Cancelled</SelectItem>
            </SelectContent>
          </Select>

          <Select value={filters.match_type} onValueChange={(value) => handleFilterChange('match_type', value)}>
            <SelectTrigger>
              <SelectValue placeholder="Match Type" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Types</SelectItem>
              <SelectItem value="individual">Individual</SelectItem>
              <SelectItem value="team">Team</SelectItem>
              <SelectItem value="mixed">Mixed</SelectItem>
            </SelectContent>
          </Select>

          <Select value={filters.format} onValueChange={(value) => handleFilterChange('format', value)}>
            <SelectTrigger>
              <SelectValue placeholder="Format" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Formats</SelectItem>
              <SelectItem value="best_of_1">Best of 1</SelectItem>
              <SelectItem value="best_of_3">Best of 3</SelectItem>
              <SelectItem value="best_of_5">Best of 5</SelectItem>
              <SelectItem value="best_of_7">Best of 7</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Table */}
      <div className="max-h-[600px] overflow-auto">
        <Table>
          <TableHeader className="sticky top-0 bg-background z-10">
            <TableRow>
              <TableHead className="w-12">
                <input type="checkbox" className="rounded" />
              </TableHead>
              <TableHead 
                className="cursor-pointer"
                onClick={() => handleSort('name')}
              >
                Match
                {filters.sort_by === 'name' && (
                  <span className="ml-1">{filters.sort_order === 'asc' ? '↑' : '↓'}</span>
                )}
              </TableHead>
              <TableHead>Type</TableHead>
              <TableHead>Participants</TableHead>
              <TableHead>Score</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Progress</TableHead>
              <TableHead>Scheduled</TableHead>
              <TableHead className="w-12">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {matches.data.map((match) => (
              <TableRow key={match.id}>
                <TableCell>
                  <input type="checkbox" className="rounded" />
                </TableCell>
                <TableCell>
                  <div>
                    <div className="font-medium">{match.name}</div>
                    <div className="text-sm text-muted-foreground">
                      {match.tournament?.name} - Round {match.round}
                    </div>
                  </div>
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    {getMatchTypeIcon(match.match_type)}
                    <Badge variant="outline" className="text-xs">
                      {match.match_type}
                    </Badge>
                  </div>
                </TableCell>
                <TableCell>
                  <div className="text-sm">
                    <div>{getParticipantName(match, 'player1')}</div>
                    <div className="text-muted-foreground">vs</div>
                    <div>{getParticipantName(match, 'player2')}</div>
                  </div>
                </TableCell>
                <TableCell>
                  <div className="text-sm font-medium">
                    {match.player1_score} - {match.player2_score}
                  </div>
                </TableCell>
                <TableCell>
                  <Badge className={getStatusColor(match.status)}>
                    {match.status.replace('_', ' ')}
                  </Badge>
                </TableCell>
                <TableCell>
                  <Badge className={getProgressStatusColor(match.progress_status)}>
                    {match.progress_status.replace('_', ' ')}
                  </Badge>
                </TableCell>
                <TableCell>
                  <div className="text-sm">
                    {match.scheduled_time ? new Date(match.scheduled_time).toLocaleDateString() : 'TBD'}
                  </div>
                </TableCell>
                <TableCell>
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" size="sm">
                        <MoreHorizontal className="w-4 h-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                      <DropdownMenuItem onClick={() => router.get(route('admin.matches.show', match.id))}>
                        <Eye className="w-4 h-4 mr-2" />
                        View
                      </DropdownMenuItem>
                      <DropdownMenuItem onClick={() => router.get(route('admin.matches.edit', match.id))}>
                        <Edit className="w-4 h-4 mr-2" />
                        Edit
                      </DropdownMenuItem>
                      <DropdownMenuItem 
                        onClick={() => handleDelete(match.id)}
                        className="text-red-600"
                      >
                        <Trash2 className="w-4 h-4 mr-2" />
                        Delete
                      </DropdownMenuItem>
                    </DropdownMenuContent>
                  </DropdownMenu>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </div>

      {/* Pagination */}
      <div className="p-4 border-t bg-muted/50">
        <div className="flex items-center justify-between">
          <div className="text-sm text-muted-foreground">
            Showing {((matches.current_page - 1) * matches.per_page) + 1} to{' '}
            {Math.min(matches.current_page * matches.per_page, matches.total)} of{' '}
            {matches.total} results
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              size="sm"
              disabled={matches.current_page === 1}
              onClick={() => handleFilterChange('page', String(matches.current_page - 1))}
            >
              Previous
            </Button>
            <span className="text-sm">
              Page {matches.current_page} of {matches.last_page}
            </span>
            <Button
              variant="outline"
              size="sm"
              disabled={matches.current_page === matches.last_page}
              onClick={() => handleFilterChange('page', String(matches.current_page + 1))}
            >
              Next
            </Button>
          </div>
        </div>
      </div>
    </Card>
  );
};

export default MatchesTable; 