import React from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { CheckCircle, Circle, ArrowLeft, ArrowRight } from 'lucide-react';

interface MatchFormStepsProps {
  steps: Array<{
    id: string;
    title: string;
    description: string;
    icon: React.ReactNode;
  }>;
  currentStep: number;
  onStepChange: (step: number) => void;
  onNext: () => void;
  onPrevious: () => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  canProceed: boolean;
  isLastStep: boolean;
  children?: React.ReactNode;
}

const MatchFormSteps: React.FC<MatchFormStepsProps> = ({
  steps,
  currentStep,
  onStepChange,
  onNext,
  onPrevious,
  onSubmit,
  isSubmitting,
  canProceed,
  isLastStep,
  children,
}) => {
  const progress = ((currentStep + 1) / steps.length) * 100;

  return (
    <div className="space-y-6">
      <Card className="p-6">
        {/* Step Indicators */}
        <div className="mb-6">
          <div className="flex items-center justify-between mb-4">
            {steps.map((step, index) => (
              <div key={step.id} className="flex items-center">
                <button
                  onClick={() => onStepChange(index)}
                  className={`flex items-center justify-center w-8 h-8 rounded-full border-2 transition-colors ${
                    index <= currentStep
                      ? 'border-primary bg-primary text-primary-foreground'
                      : 'border-muted-foreground text-muted-foreground'
                  }`}
                >
                  {index < currentStep ? (
                    <CheckCircle className="w-4 h-4" />
                  ) : (
                    <Circle className="w-4 h-4" />
                  )}
                </button>
                {index < steps.length - 1 && (
                  <div
                    className={`w-16 h-0.5 mx-2 ${
                      index < currentStep ? 'bg-primary' : 'bg-muted'
                    }`}
                  />
                )}
              </div>
            ))}
          </div>
          <div className="w-full bg-muted rounded-full h-2">
            <div
              className="bg-primary h-2 rounded-full transition-all duration-300"
              style={{ width: `${progress}%` }}
            />
          </div>
        </div>

        {/* Step Content */}
        <div className="mb-6">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            {steps[currentStep].icon}
            {steps[currentStep].title}
          </h3>
          <p className="text-sm text-muted-foreground mt-1">
            {steps[currentStep].description}
          </p>
        </div>

        {/* Form Content */}
        <div className="mb-6">
          {children}
        </div>

        {/* Navigation */}
        <div className="flex justify-between">
          <Button
            variant="outline"
            onClick={onPrevious}
            disabled={currentStep === 0}
            className="flex items-center gap-2"
          >
            <ArrowLeft className="w-4 h-4" />
            Previous
          </Button>

          <div className="flex gap-2">
            {steps.map((step, index) => (
              <Button
                key={step.id}
                variant={index === currentStep ? 'default' : 'outline'}
                size="sm"
                onClick={() => onStepChange(index)}
                disabled={index > currentStep}
              >
                {step.title}
              </Button>
            ))}
          </div>

          {isLastStep ? (
            <Button
              onClick={onSubmit}
              disabled={!canProceed || isSubmitting}
              className="flex items-center gap-2"
            >
              {isSubmitting ? 'Creating...' : 'Create Match'}
            </Button>
          ) : (
            <Button
              onClick={onNext}
              disabled={!canProceed}
              className="flex items-center gap-2"
            >
              Next
              <ArrowRight className="w-4 h-4" />
            </Button>
          )}
        </div>
      </Card>
    </div>
  );
};

export default MatchFormSteps; 