import React from 'react';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface TournamentBracketStepProps {
  form: any;
  setForm: (form: any) => void;
  errors: any;
  tournaments: any[];
  brackets: any[];
}

const TournamentBracketStep: React.FC<TournamentBracketStepProps> = ({ 
  form, 
  setForm, 
  errors, 
  tournaments = [], 
  brackets = [] 
}) => {
  // Filter brackets based on selected tournament
  const filteredBrackets = form.tournament_id 
    ? brackets.filter(bracket => bracket.tournament_id === parseInt(form.tournament_id))
    : brackets;

  return (
    <div className="space-y-4">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="tournament_id">Tournament *</Label>
          <Select value={form.tournament_id || ''} onValueChange={(value) => setForm({ ...form, tournament_id: value })}>
            <SelectTrigger>
              <SelectValue placeholder="Select tournament" />
            </SelectTrigger>
                         <SelectContent>
               {tournaments && tournaments.length > 0 ? (
                 tournaments.map((tournament) => (
                   <SelectItem key={tournament.id} value={String(tournament.id)}>
                     {tournament.name}
                   </SelectItem>
                 ))
               ) : (
                 <SelectItem value="0" disabled>No tournaments available</SelectItem>
               )}
             </SelectContent>
          </Select>
          {errors.tournament_id && <div className="text-red-500 text-xs mt-1">{errors.tournament_id}</div>}
        </div>

        <div>
          <Label htmlFor="bracket_id">Bracket</Label>
          <Select 
            value={form.bracket_id || ''} 
            onValueChange={(value) => setForm({ ...form, bracket_id: value })}
            disabled={!form.tournament_id}
          >
            <SelectTrigger>
              <SelectValue placeholder={form.tournament_id ? "Select bracket" : "Select tournament first"} />
            </SelectTrigger>
            <SelectContent>
                <SelectItem value="0">No Bracket</SelectItem>
              {filteredBrackets && filteredBrackets.length > 0 ? (
                filteredBrackets.map((bracket) => (
                  <SelectItem key={bracket.id} value={String(bracket.id)}>
                    {bracket.name} ({bracket.type})
                  </SelectItem>
                ))
              ) : (
                <SelectItem value="0" disabled>No brackets available</SelectItem>
              )}
            </SelectContent>
          </Select>
          {errors.bracket_id && <div className="text-red-500 text-xs mt-1">{errors.bracket_id}</div>}
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="round">Round *</Label>
          <Input
            id="round"
            type="number"
            min="1"
            value={form.round || ''}
            onChange={(e) => setForm({ ...form, round: parseInt(e.target.value) })}
            placeholder="Round number"
          />
          {errors.round && <div className="text-red-500 text-xs mt-1">{errors.round}</div>}
        </div>

        <div>
          <Label htmlFor="match_number">Match Number *</Label>
          <Input
            id="match_number"
            type="number"
            min="1"
            value={form.match_number || ''}
            onChange={(e) => setForm({ ...form, match_number: parseInt(e.target.value) })}
            placeholder="Match number"
          />
          {errors.match_number && <div className="text-red-500 text-xs mt-1">{errors.match_number}</div>}
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="scheduled_time">Scheduled Time *</Label>
          <Input
            id="scheduled_time"
            type="datetime-local"
            value={form.scheduled_time || ''}
            onChange={(e) => setForm({ ...form, scheduled_time: e.target.value })}
          />
          {errors.scheduled_time && <div className="text-red-500 text-xs mt-1">{errors.scheduled_time}</div>}
        </div>

        <div>
          <Label htmlFor="estimated_duration_minutes">Estimated Duration (minutes)</Label>
          <Input
            id="estimated_duration_minutes"
            type="number"
            min="1"
            value={form.estimated_duration_minutes || ''}
            onChange={(e) => setForm({ ...form, estimated_duration_minutes: e.target.value ? parseInt(e.target.value) : null })}
            placeholder="Optional"
          />
          {errors.estimated_duration_minutes && <div className="text-red-500 text-xs mt-1">{errors.estimated_duration_minutes}</div>}
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="referee_id">Referee</Label>
          <Select value={form.referee_id || ''} onValueChange={(value) => setForm({ ...form, referee_id: value })}>
            <SelectTrigger>
              <SelectValue placeholder="Select referee" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="0">No Referee</SelectItem>
              {/* Add referees here when available */}
            </SelectContent>
          </Select>
          {errors.referee_id && <div className="text-red-500 text-xs mt-1">{errors.referee_id}</div>}
        </div>

        <div>
          <Label htmlFor="allow_spectators">Allow Spectators</Label>
          <Select value={form.allow_spectators?.toString() || 'true'} onValueChange={(value) => setForm({ ...form, allow_spectators: value === 'true' })}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="true">Yes</SelectItem>
              <SelectItem value="false">No</SelectItem>
            </SelectContent>
          </Select>
          {errors.allow_spectators && <div className="text-red-500 text-xs mt-1">{errors.allow_spectators}</div>}
        </div>
      </div>
    </div>
  );
};

export default TournamentBracketStep; 