import React from 'react';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';

interface StreamingSettingsStepProps {
  form: any;
  setForm: (form: any) => void;
  errors: any;
}

const StreamingSettingsStep: React.FC<StreamingSettingsStepProps> = ({ 
  form, 
  setForm, 
  errors 
}) => {
  return (
    <div className="space-y-6">
      {/* Streaming Settings */}
      <div className="space-y-4">
        <h4 className="font-medium">Streaming & Broadcasting</h4>
        
        <div className="flex items-center space-x-2">
          <Checkbox
            id="is_streamed"
            checked={form.is_streamed || false}
            onCheckedChange={(checked) => setForm({ ...form, is_streamed: checked })}
          />
          <Label htmlFor="is_streamed">This match will be streamed</Label>
        </div>

        {form.is_streamed && (
          <div className="space-y-4 pl-6">
            <div>
              <Label htmlFor="stream_url">Stream URL</Label>
              <Input
                id="stream_url"
                value={form.stream_url || ''}
                onChange={(e) => setForm({ ...form, stream_url: e.target.value })}
                placeholder="https://twitch.tv/..."
              />
              {errors.stream_url && <div className="text-red-500 text-xs mt-1">{errors.stream_url}</div>}
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="twitch_url">Twitch URL</Label>
                <Input
                  id="twitch_url"
                  value={form.twitch_url || ''}
                  onChange={(e) => setForm({ ...form, twitch_url: e.target.value })}
                  placeholder="https://twitch.tv/..."
                />
                {errors.twitch_url && <div className="text-red-500 text-xs mt-1">{errors.twitch_url}</div>}
              </div>

              <div>
                <Label htmlFor="youtube_url">YouTube URL</Label>
                <Input
                  id="youtube_url"
                  value={form.youtube_url || ''}
                  onChange={(e) => setForm({ ...form, youtube_url: e.target.value })}
                  placeholder="https://youtube.com/..."
                />
                {errors.youtube_url && <div className="text-red-500 text-xs mt-1">{errors.youtube_url}</div>}
              </div>
            </div>

            <div>
              <Label htmlFor="broadcaster_notes">Broadcaster Notes</Label>
              <Textarea
                id="broadcaster_notes"
                value={form.broadcaster_notes || ''}
                onChange={(e) => setForm({ ...form, broadcaster_notes: e.target.value })}
                placeholder="Notes for broadcasters..."
                rows={2}
              />
              {errors.broadcaster_notes && <div className="text-red-500 text-xs mt-1">{errors.broadcaster_notes}</div>}
            </div>
          </div>
        )}
      </div>

      {/* Spectator Settings */}
      <div className="space-y-4">
        <h4 className="font-medium">Spectator Settings</h4>
        
        <div className="flex items-center space-x-2">
          <Checkbox
            id="allow_spectators"
            checked={form.allow_spectators !== false}
            onCheckedChange={(checked) => setForm({ ...form, allow_spectators: checked })}
          />
          <Label htmlFor="allow_spectators">Allow spectators to watch this match</Label>
        </div>
      </div>

      {/* Match Summary */}
      <div className="bg-green-50 dark:bg-green-950 p-4 rounded-lg">
        <h4 className="font-medium text-green-900 dark:text-green-100 mb-2">Match Summary</h4>
        <div className="text-sm text-green-800 dark:text-green-200 space-y-1">
          <p><strong>Name:</strong> {form.name || 'Not set'}</p>
          <p><strong>Type:</strong> {form.match_type || 'Not set'}</p>
          <p><strong>Format:</strong> {form.format || 'Not set'}</p>
          <p><strong>Tournament:</strong> {form.tournament_id || 'Not selected'}</p>
          <p><strong>Scheduled:</strong> {form.scheduled_time ? new Date(form.scheduled_time).toLocaleString() : 'Not set'}</p>
          <p><strong>Streamed:</strong> {form.is_streamed ? 'Yes' : 'No'}</p>
          <p><strong>Spectators:</strong> {form.allow_spectators !== false ? 'Allowed' : 'Not allowed'}</p>
        </div>
      </div>

      {/* Final Notes */}
      <div>
        <Label htmlFor="final_notes">Final Notes</Label>
        <Textarea
          id="final_notes"
          value={form.notes || ''}
          onChange={(e) => setForm({ ...form, notes: e.target.value })}
          placeholder="Any final notes or special instructions..."
          rows={3}
        />
        {errors.notes && <div className="text-red-500 text-xs mt-1">{errors.notes}</div>}
      </div>
    </div>
  );
};

export default StreamingSettingsStep; 