import React from 'react';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface ParticipantsStepProps {
  form: any;
  setForm: (form: any) => void;
  errors: any;
  players: any[];
  teams: any[];
  isEditing?: boolean;
}

const ParticipantsStep: React.FC<ParticipantsStepProps> = ({ 
  form, 
  setForm, 
  errors, 
  players = [], 
  teams = [],
  isEditing = false
}) => {
  // Filter players and teams based on selected tournament
  const filteredPlayers = form.tournament_id 
    ? (players || []).filter(player => player.tournament_id === parseInt(form.tournament_id))
    : (players || []);

  const filteredTeams = form.tournament_id 
    ? (teams || []).filter(team => team.tournament_id === parseInt(form.tournament_id))
    : (teams || []);

  const isIndividualMatch = form.match_type === 'individual';
  const isTeamMatch = form.match_type === 'team';

  return (
    <div className="space-y-6">
      <div className="bg-muted p-4 rounded-lg">
        <h4 className="font-medium mb-2">Match Type: {form.match_type || 'Not selected'}</h4>
        <p className="text-sm text-muted-foreground">
          {isIndividualMatch && 'Select individual players for this match'}
          {isTeamMatch && 'Select teams for this match'}
          {form.match_type === 'mixed' && 'Select players and/or teams for this match'}
        </p>
      </div>

      {isIndividualMatch && (
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="player1_id">Player 1</Label>
              <Select value={form.player1_id || ''} onValueChange={(value) => setForm({ ...form, player1_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Select player 1" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No Player</SelectItem>
                  {filteredPlayers && filteredPlayers.length > 0 ? (
                    filteredPlayers.map((player) => (
                      <SelectItem key={player.id} value={String(player.id)}>
                        {player.display_name}
                      </SelectItem>
                    ))
                  ) : (
                    <SelectItem value="no-players" disabled>No players available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.player1_id && <div className="text-red-500 text-xs mt-1">{errors.player1_id}</div>}
            </div>

            <div>
              <Label htmlFor="player2_id">Player 2</Label>
              <Select value={form.player2_id || ''} onValueChange={(value) => setForm({ ...form, player2_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Select player 2" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No Player</SelectItem>
                  {filteredPlayers && filteredPlayers.length > 0 ? (
                    filteredPlayers
                      .filter(player => player.id !== parseInt(form.player1_id))
                      .map((player) => (
                        <SelectItem key={player.id} value={String(player.id)}>
                          {player.display_name}
                        </SelectItem>
                      ))
                  ) : (
                    <SelectItem value="no-players" disabled>No players available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.player2_id && <div className="text-red-500 text-xs mt-1">{errors.player2_id}</div>}
            </div>
          </div>
        </div>
      )}

      {isTeamMatch && (
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="team1_id">Team 1</Label>
              <Select value={form.team1_id || ''} onValueChange={(value) => setForm({ ...form, team1_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Select team 1" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No Team</SelectItem>
                  {filteredTeams && filteredTeams.length > 0 ? (
                    filteredTeams.map((team) => (
                      <SelectItem key={team.id} value={String(team.id)}>
                        {team.name}
                      </SelectItem>
                    ))
                  ) : (
                    <SelectItem value="no-teams" disabled>No teams available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.team1_id && <div className="text-red-500 text-xs mt-1">{errors.team1_id}</div>}
            </div>

            <div>
              <Label htmlFor="team2_id">Team 2</Label>
              <Select value={form.team2_id || ''} onValueChange={(value) => setForm({ ...form, team2_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Select team 2" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No Team</SelectItem>
                  {filteredTeams && filteredTeams.length > 0 ? (
                    filteredTeams
                      .filter(team => team.id !== parseInt(form.team1_id))
                      .map((team) => (
                        <SelectItem key={team.id} value={String(team.id)}>
                          {team.name}
                        </SelectItem>
                      ))
                  ) : (
                    <SelectItem value="no-teams" disabled>No teams available</SelectItem>
                  )}
                </SelectContent>
              </Select>
              {errors.team2_id && <div className="text-red-500 text-xs mt-1">{errors.team2_id}</div>}
            </div>
          </div>
        </div>
      )}

      {form.match_type === 'mixed' && (
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="player1_id">Player/Team 1</Label>
              <Select value={form.player1_id || form.team1_id || ''} onValueChange={(value) => {
                if (value.startsWith('player_')) {
                  setForm({ ...form, player1_id: value.replace('player_', ''), team1_id: null });
                } else if (value.startsWith('team_')) {
                  setForm({ ...form, team1_id: value.replace('team_', ''), player1_id: null });
                } else {
                  setForm({ ...form, player1_id: null, team1_id: null });
                }
              }}>
                <SelectTrigger>
                  <SelectValue placeholder="Select player or team 1" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No Selection</SelectItem>
                  <optgroup label="Players">
                    {filteredPlayers && filteredPlayers.length > 0 ? (
                      filteredPlayers.map((player) => (
                        <SelectItem key={`player_${player.id}`} value={`player_${player.id}`}>
                          {player.display_name} (Player)
                        </SelectItem>
                      ))
                    ) : (
                      <SelectItem value="no-players" disabled>No players available</SelectItem>
                    )}
                  </optgroup>
                  <optgroup label="Teams">
                    {filteredTeams && filteredTeams.length > 0 ? (
                      filteredTeams.map((team) => (
                        <SelectItem key={`team_${team.id}`} value={`team_${team.id}`}>
                          {team.name} (Team)
                        </SelectItem>
                      ))
                    ) : (
                      <SelectItem value="no-teams" disabled>No teams available</SelectItem>
                    )}
                  </optgroup>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="player2_id">Player/Team 2</Label>
              <Select value={form.player2_id || form.team2_id || ''} onValueChange={(value) => {
                if (value.startsWith('player_')) {
                  setForm({ ...form, player2_id: value.replace('player_', ''), team2_id: null });
                } else if (value.startsWith('team_')) {
                  setForm({ ...form, team2_id: value.replace('team_', ''), player2_id: null });
                } else {
                  setForm({ ...form, player2_id: null, team2_id: null });
                }
              }}>
                <SelectTrigger>
                  <SelectValue placeholder="Select player or team 2" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">No Selection</SelectItem>
                  <optgroup label="Players">
                    {filteredPlayers
                      .filter(player => player.id !== parseInt(form.player1_id))
                      .map((player) => (
                        <SelectItem key={`player_${player.id}`} value={`player_${player.id}`}>
                          {player.display_name} (Player)
                        </SelectItem>
                      ))}
                  </optgroup>
                  <optgroup label="Teams">
                    {filteredTeams
                      .filter(team => team.id !== parseInt(form.team1_id))
                      .map((team) => (
                        <SelectItem key={`team_${team.id}`} value={`team_${team.id}`}>
                          {team.name} (Team)
                        </SelectItem>
                      ))}
                  </optgroup>
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      )}

      {/* Score Section - Only visible in editing mode */}
      {isEditing && (
        <div className="space-y-4">
          <div className="bg-yellow-50 dark:bg-yellow-950 p-4 rounded-lg">
            <h4 className="font-medium text-yellow-900 dark:text-yellow-100 mb-2">Match Scores</h4>
            <p className="text-sm text-yellow-800 dark:text-yellow-200 mb-4">
              Update the current scores for this match
            </p>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="player1_score">
                  {isIndividualMatch ? 'Player 1 Score' : 'Team 1 Score'}
                </Label>
                <Input
                  id="player1_score"
                  type="number"
                  min="0"
                  value={form.player1_score || 0}
                  onChange={(e) => setForm({ ...form, player1_score: parseInt(e.target.value) || 0 })}
                  placeholder="0"
                />
                {errors.player1_score && (
                  <p className="text-sm text-red-600 mt-1">{errors.player1_score}</p>
                )}
              </div>
              
              <div>
                <Label htmlFor="player2_score">
                  {isIndividualMatch ? 'Player 2 Score' : 'Team 2 Score'}
                </Label>
                <Input
                  id="player2_score"
                  type="number"
                  min="0"
                  value={form.player2_score || 0}
                  onChange={(e) => setForm({ ...form, player2_score: parseInt(e.target.value) || 0 })}
                  placeholder="0"
                />
                {errors.player2_score && (
                  <p className="text-sm text-red-600 mt-1">{errors.player2_score}</p>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      <div className="bg-blue-50 dark:bg-blue-950 p-4 rounded-lg">
        <h4 className="font-medium text-blue-900 dark:text-blue-100 mb-2">Match Summary</h4>
        <div className="text-sm text-blue-800 dark:text-blue-200">
          <p><strong>Type:</strong> {form.match_type || 'Not selected'}</p>
          <p><strong>Format:</strong> {form.format || 'Not selected'}</p>
          <p><strong>Best of:</strong> {form.best_of_games || 1} games</p>
          {form.player1_id && <p><strong>Player 1:</strong> {filteredPlayers.find(p => p.id === parseInt(form.player1_id))?.display_name || 'Unknown'}</p>}
          {form.player2_id && <p><strong>Player 2:</strong> {filteredPlayers.find(p => p.id === parseInt(form.player2_id))?.display_name || 'Unknown'}</p>}
          {form.team1_id && <p><strong>Team 1:</strong> {filteredTeams.find(t => t.id === parseInt(form.team1_id))?.name || 'Unknown'}</p>}
          {form.team2_id && <p><strong>Team 2:</strong> {filteredTeams.find(t => t.id === parseInt(form.team2_id))?.name || 'Unknown'}</p>}
          {isEditing && (
            <p><strong>Current Score:</strong> {form.player1_score || 0} - {form.player2_score || 0}</p>
          )}
        </div>
      </div>
    </div>
  );
};

export default ParticipantsStep; 