import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { HelpCircle, Shield, Award, Target, User, Trophy, BookOpen, Search, ChevronDown, ChevronUp } from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';

interface RoleBasedHelpProps {
  onClose: () => void;
}

export function RoleBasedHelp({ onClose }: RoleBasedHelpProps) {
  const { isAdmin, isOrganizer, isReferee, isPlayer } = usePermissions();
  const [expandedSections, setExpandedSections] = useState<string[]>([]);

  const getPrimaryRole = () => {
    if (isAdmin()) return 'admin';
    if (isOrganizer()) return 'organizer';
    if (isReferee()) return 'referee';
    if (isPlayer()) return 'player';
    return 'user';
  };

  const primaryRole = getPrimaryRole();

  const helpContent = {
    admin: {
      title: 'Administrator Help',
      description: 'Complete guide for system administrators',
      icon: Shield,
      color: 'text-green-600',
      bgColor: 'bg-green-50',
      sections: [
        {
          id: 'user-management',
          title: 'User Management',
          content: [
            'View all registered users in the system',
            'Assign roles and permissions to users',
            'Manage user accounts and access levels',
            'Monitor user activity and login history'
          ]
        },
        {
          id: 'system-settings',
          title: 'System Settings',
          content: [
            'Configure global tournament settings',
            'Manage system permissions and roles',
            'View system logs and error reports',
            'Monitor system performance and health'
          ]
        },
        {
          id: 'tournament-management',
          title: 'Tournament Management',
          content: [
            'Create and manage all tournaments',
            'Assign organizers to tournaments',
            'Monitor tournament progress and statistics',
            'Manage tournament settings and configurations'
          ]
        }
      ]
    },
    organizer: {
      title: 'Organizer Help',
      description: 'Guide for tournament organizers',
      icon: Award,
      color: 'text-blue-600',
      bgColor: 'bg-blue-50',
      sections: [
        {
          id: 'tournament-creation',
          title: 'Creating Tournaments',
          content: [
            'Set up new tournaments with detailed information',
            'Configure tournament rules and settings',
            'Set registration deadlines and participant limits',
            'Define tournament brackets and match formats'
          ]
        },
        {
          id: 'player-management',
          title: 'Managing Players',
          content: [
            'Add players to your tournaments',
            'Create and manage teams',
            'Track player participation and performance',
            'Handle player registrations and withdrawals'
          ]
        },
        {
          id: 'match-scheduling',
          title: 'Match Scheduling',
          content: [
            'Schedule matches between teams',
            'Assign referees to matches',
            'Set match times and venues',
            'Track match results and progress'
          ]
        }
      ]
    },
    referee: {
      title: 'Referee Help',
      description: 'Guide for match referees',
      icon: Target,
      color: 'text-orange-600',
      bgColor: 'bg-orange-50',
      sections: [
        {
          id: 'match-officiating',
          title: 'Match Officiating',
          content: [
            'View your assigned matches',
            'Update match scores and results',
            'Record match events and incidents',
            'Submit match reports and documentation'
          ]
        },
        {
          id: 'score-management',
          title: 'Score Management',
          content: [
            'Enter and verify match scores',
            'Handle score disputes and corrections',
            'Update match status and progress',
            'Ensure accurate score recording'
          ]
        },
        {
          id: 'communication',
          title: 'Communication',
          content: [
            'Communicate with tournament organizers',
            'Report issues or incidents during matches',
            'Coordinate with other referees',
            'Provide feedback on match conditions'
          ]
        }
      ]
    },
    player: {
      title: 'Player Help',
      description: 'Guide for tournament players',
      icon: User,
      color: 'text-purple-600',
      bgColor: 'bg-purple-50',
      sections: [
        {
          id: 'tournament-participation',
          title: 'Tournament Participation',
          content: [
            'View tournaments you\'re participating in',
            'Check match schedules and times',
            'Track your performance and statistics',
            'View tournament brackets and progress'
          ]
        },
        {
          id: 'match-information',
          title: 'Match Information',
          content: [
            'View your upcoming matches',
            'Check match results and scores',
            'See your team\'s performance',
            'Track your personal statistics'
          ]
        },
        {
          id: 'profile-management',
          title: 'Profile Management',
          content: [
            'Update your personal information',
            'View your tournament history',
            'Check your performance statistics',
            'Manage your account settings'
          ]
        }
      ]
    },
    user: {
      title: 'General Help',
      description: 'General user guide',
      icon: HelpCircle,
      color: 'text-gray-600',
      bgColor: 'bg-gray-50',
      sections: [
        {
          id: 'navigation',
          title: 'Navigation',
          content: [
            'Use the sidebar to navigate between sections',
            'Access your dashboard for an overview',
            'View tournaments and match information',
            'Check your profile and settings'
          ]
        },
        {
          id: 'tournaments',
          title: 'Tournaments',
          content: [
            'Browse available tournaments',
            'View tournament details and rules',
            'Check match schedules and results',
            'Access tournament information'
          ]
        },
        {
          id: 'support',
          title: 'Getting Help',
          content: [
            'Contact tournament organizers for specific questions',
            'Check the help documentation for guidance',
            'Reach out to system administrators for technical issues',
            'Use the feedback system to report problems'
          ]
        }
      ]
    }
  };

  const content = helpContent[primaryRole as keyof typeof helpContent];

  const toggleSection = (sectionId: string) => {
    setExpandedSections(prev => 
      prev.includes(sectionId) 
        ? prev.filter(id => id !== sectionId)
        : [...prev, sectionId]
    );
  };

  return (
    <div className="fixed inset-0 bg-background/80 backdrop-blur-sm z-50 flex items-center justify-center p-4">
      <Card className="w-full max-w-4xl max-h-[90vh] overflow-y-auto">
        <CardHeader className="text-center">
          <div className={`inline-flex p-4 rounded-full ${content.bgColor} mb-4`}>
            <content.icon className={`w-12 h-12 ${content.color}`} />
          </div>
          <CardTitle className="text-2xl font-bold">{content.title}</CardTitle>
          <CardDescription className="text-lg">{content.description}</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {content.sections.map((section) => (
            <Card key={section.id} className="border">
              <CardHeader className="cursor-pointer" onClick={() => toggleSection(section.id)}>
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg">{section.title}</CardTitle>
                  {expandedSections.includes(section.id) ? (
                    <ChevronUp className="h-5 w-5" />
                  ) : (
                    <ChevronDown className="h-5 w-5" />
                  )}
                </div>
              </CardHeader>
              {expandedSections.includes(section.id) && (
                <CardContent>
                  <ul className="space-y-2">
                    {section.content.map((item, index) => (
                      <li key={index} className="flex items-start gap-2 text-sm">
                        <div className={`w-1.5 h-1.5 rounded-full mt-2 ${content.color.replace('text-', 'bg-')}`} />
                        <span className="text-muted-foreground">{item}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
              )}
            </Card>
          ))}

          <div className="flex justify-end pt-4">
            <Button onClick={onClose}>
              Close Help
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 