import React, { useState, useRef, useEffect } from 'react';
import { Play, Pause, Volume2, VolumeX, Maximize2, ExternalLink, Facebook, Twitter, Instagram, Youtube, Video } from 'lucide-react';
import { router } from '@inertiajs/react';

interface VideoCTASectionProps {
  featuredVlog?: {
    id: number;
    title: string;
    description?: string;
    platform: string;
    platform_icon: string;
    video_url: string; // e.g., https://www.youtube.com/watch?v=VIDEO_ID or youtu.be/VIDEO_ID
    embed_url?: string; // e.g., https://www.youtube.com/embed/VIDEO_ID or full iframe markup
    thumbnail_url?: string; // YouTube thumbnail URL
    duration: string;
    views: string;
    likes: number;
    category: string;
    tags?: string[];
  };
}

const VideoCTASection: React.FC<VideoCTASectionProps> = ({ featuredVlog }) => {
  // Default to paused to avoid autoplay policies showing a black screen
  const [isPlaying, setIsPlaying] = useState(false);
  const [isMuted, setIsMuted] = useState(true);
  const [showControls, setShowControls] = useState(true);
  const videoRef = useRef<HTMLIFrameElement>(null);
  const controlsTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  useEffect(() => {
    if (showControls) {
      controlsTimeoutRef.current = setTimeout(() => {
        setShowControls(false);
      }, 3000);
    }

    return () => {
      if (controlsTimeoutRef.current) {
        clearTimeout(controlsTimeoutRef.current);
      }
    };
  }, [showControls]);

  const handleMouseMove = () => {
    setShowControls(true);
  };

  const handleMouseLeave = () => {
    setShowControls(false);
  };

  const togglePlayPause = () => {
    setIsPlaying((prev) => !prev);
  };

  const toggleMute = () => {
    setIsMuted((prev) => !prev);
  };

  const handleFullscreen = () => {
    if (videoRef.current) {
      if (videoRef.current.requestFullscreen) {
        videoRef.current.requestFullscreen();
      }
    }
  };

  const openInNewTab = () => {
    if (featuredVlog?.video_url) {
      window.open(featuredVlog.video_url, '_blank');
    }
  };

  // Helpers to ensure a valid YouTube embed URL
  const extractYouTubeId = (url: string): string | null => {
    try {
      const cleaned = url.trim().replace(/^@/, '');
      const u = new URL(cleaned);

      // youtu.be short links: https://youtu.be/VIDEO_ID?si=...
      if (u.hostname.includes('youtu.be')) {
        const pathId = u.pathname.split('/').filter(Boolean)[0];
        if (pathId) return pathId.substring(0, 11);
      }

      // youtube.com links
      if (u.hostname.includes('youtube.com')) {
        // Standard watch URL: https://www.youtube.com/watch?v=VIDEO_ID
        if (u.pathname.startsWith('/watch')) {
          const v = u.searchParams.get('v');
          if (v) return v.substring(0, 11);
        }
        // Embed or shorts: /embed/VIDEO_ID or /shorts/VIDEO_ID
        const parts = u.pathname.split('/').filter(Boolean);
        const idx = parts.findIndex(p => p === 'embed' || p === 'shorts' || p === 'live');
        if (idx >= 0 && parts[idx + 1]) return parts[idx + 1].substring(0, 11);
      }
    } catch {}

    // Fallback regex checks
    const ytShort = url.match(/youtu\.be\/([\w-]{11})/);
    if (ytShort?.[1]) return ytShort[1];
    const ytWatch = url.match(/[?&]v=([\w-]{11})/);
    if (ytWatch?.[1]) return ytWatch[1];
    const ytEmbed = url.match(/youtube\.com\/(?:embed|shorts)\/([\w-]{11})/);
    if (ytEmbed?.[1]) return ytEmbed[1];

    return null;
  };

  const getEmbedUrl = (): string | null => {
    if (!featuredVlog) return null;

    // If backend provided embed_url, prefer it. If it's an iframe markup, extract the src.
    if (featuredVlog.embed_url) {
      if (featuredVlog.embed_url.includes('<iframe')) {
        const srcMatch = featuredVlog.embed_url.match(/src=["']([^"']+)["']/i);
        if (srcMatch?.[1]) return srcMatch[1];
      } else {
        return featuredVlog.embed_url;
      }
    }

    // Fallback: derive from video_url
    const id = extractYouTubeId(featuredVlog.video_url);
    if (id) return `https://www.youtube.com/embed/${id}`;

    // Last resort: return original URL (may not embed correctly if not an embed URL)
    return featuredVlog.video_url;
  };

  if (!featuredVlog) {
    return null;
  }

  const idForThumb = extractYouTubeId(featuredVlog.video_url);
  const thumbnail = featuredVlog.thumbnail_url || (idForThumb ? `https://img.youtube.com/vi/${idForThumb}/hqdefault.jpg` : undefined);

  const baseEmbedUrl = getEmbedUrl();
  const embedSrc = baseEmbedUrl
    ? `${baseEmbedUrl}${baseEmbedUrl.includes('?') ? '&' : '?'}autoplay=${isPlaying ? 1 : 0}&mute=${isMuted ? 1 : 0}&controls=1&modestbranding=1&rel=0&playsinline=1`
    : '';

  return (
    <section className="py-16 bg-gradient-to-br from-gray-50 to-white">
      <div className="container mx-auto px-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
          {/* Video Section */}
          <div className="relative group">
            <div className="relative rounded-2xl overflow-hidden shadow-2xl">
              {/* Video Container */}
              <div className="relative aspect-video">
                {/* Video iframe */}
                <iframe
                  ref={videoRef}
                  key={`video-${isMuted ? 'muted' : 'unmuted'}-${isPlaying ? 'playing' : 'paused'}`}
                  src={embedSrc}
                  title={featuredVlog.title}
                  className="w-full h-full"
                  allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
                  allowFullScreen
                  frameBorder="0"
                  loading="lazy"
                />
                
                {/* Thumbnail overlay - show when not playing */}
                {thumbnail && !isPlaying && (
                  <div className="absolute inset-0 bg-cover bg-center" style={{ backgroundImage: `url(${thumbnail})` }}>
                    <div className="absolute inset-0 bg-opacity-30 flex items-center justify-center">
                      <button
                        onClick={togglePlayPause}
                        className="bg-white bg-opacity-90 hover:bg-opacity-100 rounded-full p-4 transition-all duration-200"
                      >
                        <Play className="w-8 h-8 text-gray-800 ml-1" />
                      </button>
                    </div>
                  </div>
                )}
                
                {/* Custom Controls Overlay */}
                <div 
                  className={`absolute inset-0 bg-opacity-0 transition-all duration-300 ${
                    showControls ? 'bg-opacity-20' : 'bg-opacity-0'
                  }`}
                  onMouseMove={handleMouseMove}
                  onMouseLeave={handleMouseLeave}
                >
                  {/* Center Play/Pause Button */}
                  <button
                    onClick={togglePlayPause}
                    className={`absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 
                      bg-white bg-opacity-90 hover:bg-opacity-100 rounded-full p-4 transition-all duration-200
                      ${isPlaying ? 'opacity-0 group-hover:opacity-100' : 'opacity-100'}`}
                  >
                    {isPlaying ? (
                      <Pause className="w-8 h-8 text-gray-800" />
                    ) : (
                      <Play className="w-8 h-8 text-gray-800 ml-1" />
                    )}
                  </button>

                  {/* Bottom Controls */}
                  <div className={`absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black to-transparent p-4 transition-all duration-300 ${
                    showControls ? 'opacity-100' : 'opacity-0'
                  }`}>
                    <div className="flex items-center justify-between text-white">
                      <div className="flex items-center space-x-4">
                        <button
                          onClick={togglePlayPause}
                          className="hover:bg-white hover:bg-opacity-20 rounded-full p-2 transition-colors"
                        >
                          {isPlaying ? (
                            <Pause className="w-5 h-5" />
                          ) : (
                            <Play className="w-5 h-5" />
                          )}
                        </button>
                        
                        <button
                          onClick={toggleMute}
                          className="hover:bg-white hover:bg-opacity-20 rounded-full p-2 transition-colors"
                        >
                          {isMuted ? (
                            <VolumeX className="w-5 h-5" />
                          ) : (
                            <Volume2 className="w-5 h-5" />
                          )}
                        </button>
                      </div>
                      
                      <div className="flex items-center space-x-2">
                        <button
                          onClick={openInNewTab}
                          className="hover:bg-white hover:bg-opacity-20 rounded-full p-2 transition-colors"
                          title="Open in new tab"
                        >
                          <ExternalLink className="w-5 h-5" />
                        </button>
                        
                        <button
                          onClick={handleFullscreen}
                          className="hover:bg-white hover:bg-opacity-20 rounded-full p-2 transition-colors"
                          title="Fullscreen"
                        >
                          <Maximize2 className="w-5 h-5" />
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Video Info */}
            <div className="mt-4">
              <div className="flex items-center space-x-2 mb-2">
                <span className="text-sm font-medium text-gray-600">{featuredVlog.category}</span>
                <span className="text-gray-400">•</span>
                <span className="text-sm text-gray-500">{featuredVlog.duration}</span>
                <span className="text-gray-400">•</span>
                <span className="text-sm text-gray-500">{featuredVlog.views} views</span>
              </div>
              
              {featuredVlog.tags && featuredVlog.tags.length > 0 && (
                <div className="flex flex-wrap gap-2 mb-3">
                  {featuredVlog.tags.slice(0, 3).map((tag, index) => (
                    <span
                      key={index}
                      className="px-2 py-1 bg-gray-100 text-gray-700 text-xs rounded-full"
                    >
                      #{tag}
                    </span>
                  ))}
                </div>
              )}
            </div>
          </div>

          {/* About Us Section */}
          <div className="space-y-6">
            <div className="space-y-4">
              <h2 className="text-3xl lg:text-4xl font-bold text-gray-900">
                Experience the <span className="text-green-600">Milimu Cup</span> Magic
              </h2>
              
              <p className="text-lg text-gray-600 leading-relaxed">
                Discover the passion, excitement, and community spirit that makes Milimu Cup the premier 
                football tournament in Kenya. Watch our latest featured content and get a glimpse of the 
                incredible moments that define our tournament.
              </p>
              
              <p className="text-gray-600 leading-relaxed">
                From thrilling matches to behind-the-scenes moments, our vlog captures the essence of 
                what makes Milimu Cup special. Join thousands of players and fans who have experienced 
                the magic of competitive football at its finest.
              </p>
            </div>

            <div className="pt-6 space-y-4">
              <button 
                onClick={() => {router.visit('/vlog') }}
                className="inline-flex items-center px-6 py-3 bg-green-600 text-white font-semibold rounded-lg hover:bg-green-700 transition-colors duration-200 shadow-lg hover:shadow-xl"
              >
                <Play className="w-5 h-5 mr-2" />
                Watch More Videos
              </button>
              
              {/* Social Media Buttons */}
              <div className="flex items-center space-x-3">
                <span className="text-sm font-medium text-gray-600">Follow us:</span>
                <div className="flex space-x-2">
                  <button
                    onClick={() => window.open('https://www.facebook.com/share/16vbkxF1vY/', '_blank')}
                    className="inline-flex items-center justify-center w-10 h-10 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors duration-200 shadow-md hover:shadow-lg"
                    title="Follow us on Facebook"
                  >
                    <Facebook className="w-5 h-5" />
                  </button>
                  
                  {/* <button
                    onClick={() => window.open('https://twitter.com/milimucup', '_blank')}
                    className="inline-flex items-center justify-center w-10 h-10 bg-sky-500 text-white rounded-lg hover:bg-sky-600 transition-colors duration-200 shadow-md hover:shadow-lg"
                    title="Follow us on Twitter"
                  >
                    <Twitter className="w-5 h-5" />
                  </button> */}
                  
                  <button
                    onClick={() => window.open('https://instagram.com/milimucuphttps://www.instagram.com/the_milimus_tournament', '_blank')}
                    className="inline-flex items-center justify-center w-10 h-10 bg-gradient-to-r from-purple-500 to-pink-500 text-white rounded-lg hover:from-purple-600 hover:to-pink-600 transition-all duration-200 shadow-md hover:shadow-lg"
                    title="Follow us on Instagram"
                  >
                    <Instagram className="w-5 h-5" />
                  </button>
                  
                  <button
                    onClick={() => window.open('https://www.youtube.com/@TheMilimusGerryTournamentcup', '_blank')}
                    className="inline-flex items-center justify-center w-10 h-10 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors duration-200 shadow-md hover:shadow-lg"
                    title="Subscribe to our YouTube channel"
                  >
                    <Youtube className="w-5 h-5" />
                  </button>


                  <button
                    onClick={() => window.open('https://www.tiktok.com/@the.milimus.tourn?_t=ZM-8yIfsMBR37l&_r=1', '_blank')}
                    className="inline-flex items-center justify-center w-10 h-10 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors duration-200 shadow-md hover:shadow-lg"
                    title="Subscribe to our YouTube channel"
                  >
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    viewBox="0 0 40 40"
                    fill="none"
                    className="w-5 h-5"
                  >
                    <g>
                      <path
                        d="M28.5 10.5c-1.1 0-2-.9-2-2V5.5h-3.5v18.5c0 2.8-2.2 5-5 5s-5-2.2-5-5 2.2-5 5-5c.3 0 .7 0 1 .1v-3.6c-.3 0-.7-.1-1-.1-4.7 0-8.5 3.8-8.5 8.5s3.8 8.5 8.5 8.5 8.5-3.8 8.5-8.5V16.5c1.5 1.1 3.3 1.8 5.3 1.8v-3.5c-1.7 0-3.3-.6-4.5-1.8z"
                        fill="#25F4EE"
                      />
                      <path
                        d="M29.5 13.5v3.5c2 0 3.8-.7 5.3-1.8v-3.6c-1.5 1.1-3.3 1.9-5.3 1.9z"
                        fill="#FE2C55"
                      />
                      <path
                        d="M26.5 5.5v3c0 1.1.9 2 2 2 .3 0 .7 0 1-.1v-4.9h-3z"
                        fill="#FE2C55"
                      />
                      <path
                        d="M26.5 5.5v18.5c0 2.8-2.2 5-5 5s-5-2.2-5-5c0-2.8 2.2-5 5-5 .3 0 .7 0 1 .1v-3.6c-.3 0-.7-.1-1-.1-4.7 0-8.5 3.8-8.5 8.5s3.8 8.5 8.5 8.5 8.5-3.8 8.5-8.5V16.5c1.5 1.1 3.3 1.8 5.3 1.8v-3.5c-1.7 0-3.3-.6-4.5-1.8z"
                        fill="#000"
                        fillOpacity=".8"
                      />
                    </g>
                  </svg>
                  </button>


                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default VideoCTASection; 