import { Quote, Star, Trophy, Users, Award } from "lucide-react";
import { Card, CardContent } from "@/components/ui/card";

interface TestimonialsSectionProps {
  testimonials: any[];
}

const TestimonialsSection = ({ testimonials }: TestimonialsSectionProps) => {
  // Handle empty testimonials
  const safeTestimonials = testimonials || [];

  const stats = [
    {
      icon: Trophy,
      value: "98%",
      label: "Satisfaction Rate",
      description: "Based on post-tournament surveys"
    },
    {
      icon: Users,
      value: "500+",
      label: "Teams Participated",
      description: "Across all editions"
    },
    {
      icon: Star,
      value: "4.9/5",
      label: "Average Rating",
      description: "From players and coaches"
    },
    {
      icon: Award,
      value: "100%",
      label: "Return Rate",
      description: "Teams want to come back"
    }
  ];

  return (
    <section className="py-20 bg-gradient-to-br from-primary/5 to-secondary/5">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section Header */}
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold text-foreground mb-4">
            What They <span className="text-primary">Say</span>
          </h2>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
            Hear from players, coaches, and partners about their experience with The Milimus Tournament
          </p>
        </div>

        {/* Testimonials Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-16">
          {safeTestimonials.map((testimonial, index) => (
            <Card key={index} className="sports-card border-none bg-white/80 backdrop-blur-sm hover:shadow-xl transition-all duration-300">
              <CardContent className="p-6">
                <div className="flex justify-start items-center mb-4">
                  <div className="text-3xl mr-4">
                      <img src={testimonial.image} alt={testimonial.name} className="w-24 h-24 rounded-full object-cover border-2 border-primary" />
                  </div>
                  <div>
                    <h4 className="font-semibold text-foreground">{testimonial.name}</h4>
                    <p className="text-sm text-muted-foreground">{testimonial.role}</p>
                    <p className="text-xs text-primary font-medium">{testimonial.team}</p>
                  </div>
                </div>
                
                <div className="flex items-center mb-3">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <Star key={i} className="h-4 w-4 text-yellow-400 fill-current" />
                  ))}
                </div>

                <blockquote className="text-muted-foreground text-sm mb-4 italic">
                  "{testimonial.quote}"
                </blockquote>

                <div className="flex items-center justify-between">
                  <div className="text-xs text-primary font-medium">{testimonial.achievement}</div>
                  <Quote className="h-4 w-4 text-primary/30" />
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Statistics */}
        <div className="bg-white/80 backdrop-blur-sm rounded-2xl p-8 shadow-xl">
          <div className="text-center mb-12">
            <h3 className="text-3xl font-bold text-foreground mb-4">
              Tournament <span className="text-primary">Statistics</span>
            </h3>
            <p className="text-lg text-muted-foreground">
              Numbers that speak for themselves
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="text-center">
                <div className="mx-auto mb-4 p-3 rounded-full bg-primary/10 w-fit">
                  <stat.icon className="h-8 w-8 text-primary" />
                </div>
                <div className="text-3xl font-bold text-foreground mb-2">{stat.value}</div>
                <h4 className="text-lg font-semibold text-foreground mb-2">{stat.label}</h4>
                <p className="text-sm text-muted-foreground">{stat.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>
    </section>
  );
};

export default TestimonialsSection; 